﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <curl/curl.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/repair.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/repair/repair_ShutdownButton.h>
#include <nn/repair/repair_Sdcard.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_FormatString.h>
#include <nn/nim/srv/nim_DeviceContext.h>
#include <nn/ssl.h>
#include <nn/spl/spl_Api.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/nim/nim_Result.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/repair/repair_NetworkSettingsParser.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>

#include <functional>

#include "RepairAccountTransfer.h"
#include "RepairAccountTransfer_LabelButton.h"

namespace {

static const char* ToolName = "NX Account Transfer Tool";
const int ToolMajorVersion = 5;
const int ToolMinorVersion = 0;

const int MaxRetryCount = 5;
static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;
static glv::Label s_SerialNumberLabel;
static glv::Label s_TransfererSerial;
static glv::Label s_ToolInformationLabel;
static nn::repair::StreamTextView* s_pStreamText;
LabelButton* s_pLabelButton;
nn::os::ThreadType s_WorkerThread;
nn::os::ThreadType s_SdCardDetectThread;
std::unique_ptr<nn::fs::IEventNotifier> s_Notifier;

nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
nn::nifm::NetworkProfileData g_NetworkSettings;

bool g_AirplaneModeEnabled = false;

const int StackSize = 50 * 4096;
NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t s_Stack[StackSize];

const int OperationStackSize = 50 * 4096;
NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t s_OperationStack[OperationStackSize];

static void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

static void Deallocate(void* address, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(address);
}

static void* Reallocate(void* address, size_t size, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(address, size);
}

static void SetupPeripherals() NN_NOEXCEPT
{
    nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
    glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
}

void WaitTimerForDebug(int waitSeconds) NN_NOEXCEPT
{
    const int BufferSize = 200;
    char buffer[BufferSize];
    nn::util::SNPrintf(buffer, BufferSize, "Automatically retry the operation after %d seconds.\n", waitSeconds);

    s_pStreamText->AppendValue(std::string(buffer));
    int timer = waitSeconds;
    while (timer >= 0)
    {
        --timer;
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
}

void WaitNetworkConnectionAvailable(nn::nifm::TemporaryNetworkProfile* temporaryNetworkProfile,
    nn::nifm::NetworkConnection* networkConnection) NN_NOEXCEPT
{
    nn::nifm::RequestHandle requestHandle = networkConnection->GetRequestHandle();
    nn::util::Uuid profileId = temporaryNetworkProfile->GetId();
    nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);

    while (NN_STATIC_CONDITION(1))
    {
        s_pStreamText->AppendValue("Connecting...\n");
        networkConnection->SubmitRequestAndWait();

        if (networkConnection->IsAvailable())
        {
            s_pStreamText->AppendValue("Network is available.\n");
            break;
        }
        else
        {
            s_pStreamText->AppendValue("Network is not available. retry.\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
        }
    }
}

bool DeviceMigration(nn::nifm::NetworkConnection* networkConnection) NN_NOEXCEPT
{
    for(int i = 0; i < MaxRetryCount; i++)
    {
        s_pStreamText->AppendValue("Progress: Try to migration\n");

        //ネットワーク接続確立
        networkConnection->SubmitRequestAndWait();
        if (!networkConnection->IsAvailable())
        {
            s_pStreamText->AppendValue("Network is not available.\n");
        }
        else
        {
            nn::Result result = RequestRepairDeviceMigration();
            if (result.IsSuccess())
            {
                return true;
            }
            else
            {
                const int BufferSize = 200;
                char buffer[BufferSize];
                nn::util::SNPrintf(buffer, BufferSize, "migration - Failed. Error code = 0x%08x.\n", result.GetInnerValueForDebug());
                s_pStreamText->AppendValue(std::string(buffer));
            }
        }

        WaitTimerForDebug(5);
    }

    return false;
}

bool DeviceAccountTransfer(nn::nifm::NetworkConnection* networkConnection) NN_NOEXCEPT
{
    for(int i = 0; i < MaxRetryCount; i++)
    {
        s_pStreamText->AppendValue("Progress: Try to AccountTransfer\n");

        //ネットワーク接続確立
        networkConnection->SubmitRequestAndWait();
        if (!networkConnection->IsAvailable())
        {
            s_pStreamText->AppendValue("Network is not available.\n");
        }
        else
        {
            nn::Result result = RequestTransferDeviceAccount();

            if (result.IsSuccess())
            {
                //successima
                s_pStreamText->AppendValue("AccountTransfer - Success.\n");
                return true;
            }
            else if (nn::nim::ResultEciUnderMaintenance::Includes(result))
            {
                //server-maintenance
                s_pStreamText->AppendValue("The server is under maintenance.\n");
                return false;
            }
            else if (nn::nim::ResultEciIasTransferRequestNotFoundError::Includes(result))
            {
                //UMT の操作が行われていない
                s_pStreamText->AppendValue("AccountTransfer - Failed. Account transfer request not found.\n");
                return false;
            }
            else
            {
                const int BufferSize = 200;
                char buffer[BufferSize];
                nn::util::SNPrintf(buffer, BufferSize, "AccountTransfer - Failed. Error code = 0x%08x.\n", result.GetInnerValueForDebug());

                //failed
                s_pStreamText->AppendValue(std::string(buffer));
            }
        }

        WaitTimerForDebug(5);
    }

    return false;
}

bool SyncTickets(nn::nifm::NetworkConnection* networkConnection) NN_NOEXCEPT
{
    for (int i = 0; i < MaxRetryCount; i++)
    {
        s_pStreamText->AppendValue("Progress: Try to Sync Tickets\n");

        //ネットワーク接続確立
        networkConnection->SubmitRequestAndWait();
        if (!networkConnection->IsAvailable())
        {
            s_pStreamText->AppendValue("Network is not available.\n");
        }
        else
        {
            nn::Result result = RequestSyncTicket();
            if (result.IsSuccess())
            {
                //success
                s_pStreamText->AppendValue("Progress: Sync Tickets - Success.\n");
                return true;
            }
            else
            {
                const int BufferSize = 200;
                char buffer[BufferSize];
                nn::util::SNPrintf(buffer, BufferSize, "Progress: Sync Tickets - Failed. Error code = 0x%08x. \n", result.GetInnerValueForDebug());

                //failed
                s_pStreamText->AppendValue(std::string(buffer));
            }
        }

        WaitTimerForDebug(5);
    }

    return false;
}

void EnableWifiConnection() NN_NOEXCEPT
{
    g_AirplaneModeEnabled = !nn::nifm::IsWirelessCommunicationEnabled();
    nn::nifm::SetWirelessCommunicationEnabled(true);
}

void RestoreAirplaneMode() NN_NOEXCEPT
{
    if (g_AirplaneModeEnabled) {
        nn::nifm::SetWirelessCommunicationEnabled(false);
    }
}

void OperationFunction() NN_NOEXCEPT
{
    //Wifi 接続を有効化する
    EnableWifiConnection();
    NN_UTIL_SCOPE_EXIT{
        RestoreAirplaneMode();
    };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(g_NetworkSettings);
    nn::nifm::NetworkConnection networkConnection;

    WaitNetworkConnectionAvailable(&temporaryNetworkProfile, &networkConnection);

    if (!DeviceAccountTransfer(&networkConnection))
    {
        //サーバーメンテナンス等で失敗
        s_pStreamText->AppendValue("[[ FAILED ]]\n");
        return;
    }

    if (!DeviceMigration(&networkConnection))
    {
        //サーバーメンテナンス等で失敗
        s_pStreamText->AppendValue("[[ FAILED ]]\n");
        return;
    }

    if (!SyncTickets(&networkConnection))
    {
        //サーバーメンテナンス等で失敗
        s_pStreamText->AppendValue("[[ FAILED ]]\n");
        return;
    }

    s_pStreamText->AppendValue("[[ SUCCESS ]]\n");
}

void StartRepairOperation() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&s_WorkerThread,
                             (nn::os::ThreadFunction)OperationFunction,
                             NULL,
                             s_OperationStack,
                             OperationStackSize,
                             30));

    nn::os::StartThread(&s_WorkerThread);
}

static glv::GLV s_GlvRootView;

nn::Result GetTransfererSerialNumberFromSdCard(nn::settings::factory::SerialNumber *pOutValue) NN_NOEXCEPT
{
    const char* MountName = "SdCard";
    NN_RESULT_DO(nn::fs::MountSdCard(MountName));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName);
    };

    nn::fs::FileHandle handle;
    const std::string directoryPath = std::string(MountName) + ":/" + nn::repair::ArchiveDirectoryPath;
    const std::string path = directoryPath + std::string(nn::repair::SerialNumberFileName);

    NN_RESULT_DO(nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t fileSize = 0;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    int totalSize = static_cast<int>(fileSize);
    std::unique_ptr<char[]> storage(new char[totalSize]);

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, storage.get(), totalSize));

    memcpy(pOutValue, storage.get(), totalSize);

    NN_RESULT_SUCCESS;
}

void DisplayToolInformation(glv::GLV* const pGlvRootView) NN_NOEXCEPT
{
    nn::repair::GetToolInformationLabel(&s_ToolInformationLabel, ToolName, ToolMajorVersion, ToolMinorVersion);

    *pGlvRootView << s_ToolInformationLabel;
}

nn::Result DisplayDeviceId() NN_NOEXCEPT
{
    glv::Label deviceIdLabel;
    nn::repair::GetDeviceIdLabel(&deviceIdLabel);

    s_pStreamText->AppendValue(deviceIdLabel.getValue() + "\n");

    NN_RESULT_SUCCESS;
}

nn::Result DisplayTargetSerialNumber() NN_NOEXCEPT
{
    glv::Label serialNumberLabel;
    nn::repair::GetSerialNumberLabel(&serialNumberLabel);

    s_pStreamText->AppendValue(serialNumberLabel.getValue() + "\n");

    NN_RESULT_SUCCESS;
}

void DisplayTranfererSerialNumber() NN_NOEXCEPT
{
    nn::settings::factory::SerialNumber serialNumber;
    nn::Result result = GetTransfererSerialNumberFromSdCard(&serialNumber);

    if (result.IsFailure())
    {
        std::memset(serialNumber.string, '-', sizeof(serialNumber.string) - 1);
    }

    glv::Label transfererSerial;
    nn::repair::GetTransfererSerialNumberLabel(&transfererSerial, &serialNumber);

    s_pStreamText->AppendValue(transfererSerial.getValue() + "\n");
}

nn::Result ImportNetworkSettingsFromSdCard(bool* pOutValue) NN_NOEXCEPT
{
    *pOutValue = false;

    const std::string MountName = "SdCard";
    NN_RESULT_DO(nn::fs::MountSdCard(MountName.c_str()));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName.c_str());
    };

    nn::fs::FileHandle handle;
    const std::string SerialFileName = (MountName + ":/settings.json");

    NN_RESULT_DO(nn::fs::OpenFile(&handle, SerialFileName.c_str(), nn::fs::OpenMode::OpenMode_Read));

    int64_t fileSize = 0;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    int totalSize = static_cast<int>(fileSize);
    std::unique_ptr<char[]> storage(new char[totalSize]);

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, storage.get(), totalSize));

    auto text = std::string(storage.get(), totalSize);

    if (!nn::repair::ImportNetworkSettings(&g_NetworkSettings, text.c_str()))
    {
        s_pStreamText->AppendValue("Invaild NetworkSettings Format.\n");
        NN_RESULT_SUCCESS;
    }

    *pOutValue = true;

    NN_RESULT_SUCCESS;
}

nn::Result DetectSdCardInsertion() NN_NOEXCEPT
{
    // 移行元のシリアルナンバーを読み取る
    // 刺し忘れに対応するために別スレッドで検出する
    NN_RESULT_DO(nn::os::CreateThread(&s_SdCardDetectThread, [](void *p)
    {
        std::unique_ptr<nn::repair::Sdcard> sdcard(new nn::repair::Sdcard());
        auto result = sdcard->WaitAttach();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        s_pStreamText->AppendValue("---Information--------------------------------------------------------\n");

        DisplayDeviceId();
        DisplayTranfererSerialNumber();
        DisplayTargetSerialNumber();

        s_pStreamText->AppendValue("-----------------------------------------------------------------------\n");
        //進捗
        s_pStreamText->AppendValue("Progress: Waiting for the operation\n");

        bool imported = false;
        ImportNetworkSettingsFromSdCard(&imported);

        s_pLabelButton = new LabelButton("START", [&] { StartRepairOperation(); });
        s_GlvRootView << s_pLabelButton->pos(glv::Place::BC, 0, -30).anchor(glv::Place::BC);


    }, NULL, s_Stack, StackSize, 30));

    nn::os::StartThread(&s_SdCardDetectThread);

    NN_RESULT_SUCCESS;
}

nn::Result Initialize() NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    s_pStreamText = new nn::repair::StreamTextView(glv::Rect(1024, 440), 28.0f);
    s_pStreamText->pos(120, 164);
    *pGlvRootView << s_pStreamText;

    // シャットダウンボタン(FW 3.0.0 以降で有効)
    glv::Button* shutdownButton = new nn::repair::ShutdownButton();
    shutdownButton->pos(800,640);
    *pGlvRootView << shutdownButton;

    DisplayToolInformation(pGlvRootView);

    //進捗
    s_pStreamText->AppendValue("Progress: Waiting for insert the SdCard\n");

    NN_RESULT_DO(DetectSdCardInsertion());

    NN_RESULT_SUCCESS;
}

void Main( glv::Window& window ) NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    nn::Result result = Initialize();
    if (result.IsFailure())
    {
        const int BufferSize = 200;
        char buffer[BufferSize];
        nn::util::SNPrintf(buffer, BufferSize, "Initialization Failed. Error code = 0x%08x.\n", result.GetInnerValueForDebug());

        s_pStreamText->AppendValue(std::string(buffer));
    }

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set( 0.5 );

    window.setGLV(*pGlvRootView);
    glv::Application::run();
}

} // namespace

extern "C" void nnMain()
{
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    //TODO: Finalize 周り
    NN_ABORT_UNLESS_EQUAL(CURLE_OK, curl_global_init(CURL_GLOBAL_DEFAULT));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    nn::repair::Initialize();
    nn::account::Initialize();

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_ABORT_UNLESS_NOT_NULL(window);

    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    Main(*window);

    nn::socket::Finalize();
    nn::ssl::Finalize();
}
