﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "RepairAbortMigration.h"

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>

namespace {
    static bool                         s_AirplaneModeEnabled = false;
    static bool                         s_IsNetworkSettingsValid = false;
    static nn::nifm::NetworkProfileData s_NetworkSettings;
    static nn::nifm::NetworkConnection* s_networkConnection = nullptr;
    static nn::nifm::TemporaryNetworkProfile *s_temporaryNetworkProfile;
}

void InitializeWifi() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());
}

void EnableWifiConnection() NN_NOEXCEPT
{
    s_AirplaneModeEnabled = !nn::nifm::IsWirelessCommunicationEnabled();
    nn::nifm::SetWirelessCommunicationEnabled(true);
}

void RestoreAirplaneMode() NN_NOEXCEPT
{
    if (s_AirplaneModeEnabled) {
        nn::nifm::SetWirelessCommunicationEnabled(false);
    }
}

bool WaitNetworkConnectionAvailable() NN_NOEXCEPT
{
    if(!s_IsNetworkSettingsValid)
      return false;

    if(s_networkConnection == nullptr)
    {
        s_networkConnection = new nn::nifm::NetworkConnection();
    }

    s_temporaryNetworkProfile = new nn::nifm::TemporaryNetworkProfile(s_NetworkSettings);
    nn::nifm::RequestHandle requestHandle = s_networkConnection -> GetRequestHandle();
    nn::util::Uuid profileId = s_temporaryNetworkProfile -> GetId();
    nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);

    bool is = false;

    while (NN_STATIC_CONDITION(1))
    {
        s_networkConnection -> SubmitRequestAndWait();

        if (s_networkConnection -> IsAvailable())
        {
            NN_LOG("Network is available.\n");
            is = true;
            break;
        }
        else
        {
            NN_LOG("Network is not available. retry.\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

    return is;
}

// 接続の放棄
void ReleaseNetworkConnection() NN_NOEXCEPT
{
    if(s_networkConnection != nullptr)
    {
        delete s_networkConnection;
        s_networkConnection = nullptr;
    }
}


// u sd から network 設定を読み込む
nn::Result ImportNetworkSettingsFromSdCard() NN_NOEXCEPT
{
    nn::fs::FileHandle handle;

    // wait sd attach
    std::unique_ptr<nn::repair::Sdcard> sdcard(new nn::repair::Sdcard());
    NN_RESULT_DO( sdcard->WaitAttach() );

    // mount
    NN_RESULT_DO( nn::fs::MountSdCardForDebug(g_VolumeSd.c_str()) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(g_VolumeSd.c_str());
    };

    std::string SerialFileName = (g_VolumeSd + ":/settings.json");
    NN_RESULT_DO(nn::fs::OpenFile(&handle, SerialFileName.c_str(), nn::fs::OpenMode::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t fileSize = 0;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    int totalSize = static_cast<int>(fileSize);
    std::unique_ptr<char[]> storage(new char[totalSize]);

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, storage.get(), totalSize));

    auto text = std::string(storage.get(), totalSize);
    NN_ABORT_UNLESS( nn::repair::ImportNetworkSettings(&s_NetworkSettings, text.c_str()) , "Invaild NetworkSettings Format.\n");

    s_IsNetworkSettingsValid = true;

    NN_RESULT_SUCCESS;
}

