﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "RepairAbortMigration.h"
#include <nn/crypto/crypto_Aes128CmacGenerator.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_ResultPrivate.h>

namespace{

    nn::Bit8 aeskey[] = {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c};
    const std::string                           s_MigrationInfoFile = ":/miginfo.bin";
    const std::string                           s_BackupFolder      = ":/NintendoRepairArchive/";
    const std::string                           s_SerialNumberFile  = "SerialNumber.txt";
    const std::string                           s_MigrationStateFile = "migration.bin";
    const uint32_t                              infofile_magic = 'ABOR';

    typedef struct
    {
        uint32_t magic;
        nn::migration::user::MigrationRole   role;
        nn::migration::user::MigrationState  state;

    }MigrationStateFile;

    void PutMac(RepairAbortMigrationInfoFile *storage)
    {
        nn::crypto::Aes128CmacGenerator aesCmac;

        aesCmac.Initialize(aeskey, sizeof(aeskey));
        aesCmac.Update( &storage->magic, sizeof(storage->magic));
        aesCmac.Update( &storage->role,  sizeof(storage->role));
        aesCmac.Update( &storage->state,  sizeof(storage->state));
        aesCmac.Update( &storage->fwVersion,  sizeof(storage->fwVersion));
        aesCmac.Update( &storage->sessionId,  sizeof(storage->sessionId));
        aesCmac.Update( &storage->user,       sizeof(storage->user));

        aesCmac.GetMac(storage->mac, sizeof(storage->mac));
    }

    bool VerifyMac(RepairAbortMigrationInfoFile *storage)
    {
        nn::Bit8 mac[16];
        nn::crypto::Aes128CmacGenerator aesCmac;

        aesCmac.Initialize(aeskey, sizeof(aeskey));
        aesCmac.Update( &storage->magic, sizeof(storage->magic));
        aesCmac.Update( &storage->role,  sizeof(storage->role));
        aesCmac.Update( &storage->state,  sizeof(storage->state));
        aesCmac.Update( &storage->fwVersion,  sizeof(storage->fwVersion));
        aesCmac.Update( &storage->sessionId,  sizeof(storage->sessionId));
        aesCmac.Update( &storage->user,       sizeof(storage->user));

        aesCmac.GetMac(mac, sizeof(mac));

        bool is = (0 == std::memcmp(mac, storage->mac, 16));
        return is;
    }
}

void InitMac() NN_NOEXCEPT
{
    RepairAbortMigrationInfoFile tmp;
    NN_ABORT_UNLESS ( sizeof(tmp.mac) == nn::crypto::CmacGenerator<nn::crypto::AesEncryptor128>::MacSize );
}

// シリアルナンバーの一致チェック
nn::Result VerifySerialNumber(const std::string &filename) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    const size_t BufferSize = 256;
    std::unique_ptr<char[]> buffer(new char[BufferSize]);
    std::unique_ptr<char[]> serialNumber(new char[nn::repair::SerialNumberLength]);
    size_t readSize = 0;

    NN_RESULT_DO(nn::fs::OpenFile(&handle, filename.c_str(), nn::fs::OpenMode::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(&readSize, handle, 0, buffer.get(), BufferSize));
    NN_RESULT_DO(nn::repair::GetSerialNumber(serialNumber.get()));

    if( readSize == strlen(serialNumber.get()) &&
        (std::memcmp( serialNumber.get(),  buffer.get(), readSize) == 0) )
    {
        // シリアルNo が一致
    }
    else
    {
        NN_LOG("[LoadMigrationStateFromSd] Another Device's data \n");
        return nn::repair::ResultNotMyData();
    }

    NN_RESULT_SUCCESS;
}

nn::Result ReadMigrationStateFile(const std::string &filename, nn::migration::user::LastMigrationInfo *info) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    const size_t BufferSize = 256;
    std::unique_ptr<char[]> buffer(new char[BufferSize]);
    size_t readSize = 0;

    NN_RESULT_DO(nn::fs::OpenFile(&handle, filename.c_str(), nn::fs::OpenMode::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(&readSize, handle, 0, buffer.get(), BufferSize));

    MigrationStateFile stateData;
    std::memcpy(&stateData, buffer.get(), sizeof( MigrationStateFile ));

    if( readSize >= sizeof( MigrationStateFile ) &&
        stateData.magic == 'MIGS')
    {
        info->role  = stateData.role;
        info->state = stateData.state;
    }
    else
    {
        NN_LOG("[LoadMigrationStateFromSd]Wrong Format \n");
        return nn::repair::ResultVerifyFailure();
    }

    NN_RESULT_SUCCESS;

}

// micro sd から(自分の)中断情報を読み込む
nn::Result LoadMigrationStateFromSd(nn::migration::user::LastMigrationInfo *info) NN_NOEXCEPT
{
    // wait sd attach

    if( !nn::fs::IsSdCardInserted() )
    {
        NN_LOG("[LoadMigrationStateFromSd]No Device\n");
        return nn::fs::ResultPortSdCardNoDevice();
    }

    // mount
    NN_RESULT_DO( nn::fs::MountSdCardForDebug(g_VolumeSd.c_str()) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(g_VolumeSd.c_str());
    };

    std::string serialFileName = (g_VolumeSd + s_BackupFolder + s_SerialNumberFile);
    NN_RESULT_DO( VerifySerialNumber(serialFileName) );

    std::string migrationFileName = (g_VolumeSd + s_BackupFolder + s_MigrationStateFile);
    NN_RESULT_DO( ReadMigrationStateFile(migrationFileName, info) );

    NN_RESULT_SUCCESS;
}

// micro sd から中断情報を読み込む
nn::Result LoadMigrationInfoFromSd(RepairAbortMigrationInfoFile *storage) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    nn::Result result;

    // wait sd attach
    std::unique_ptr<nn::repair::Sdcard> sdcard(new nn::repair::Sdcard());
    NN_RESULT_DO( sdcard->WaitAttach() );

    // mount
    NN_RESULT_DO( nn::fs::MountSdCardForDebug(g_VolumeSd.c_str()) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(g_VolumeSd.c_str());
    };

    std::string SerialFileName = (g_VolumeSd + s_MigrationInfoFile);
    NN_RESULT_DO(nn::fs::OpenFile(&handle, SerialFileName.c_str(), nn::fs::OpenMode::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, storage, sizeof(RepairAbortMigrationInfoFile)));

    if( !VerifyMac(storage) || (storage->magic != infofile_magic))
    {
        NN_LOG("[RepairAbortMigrationInfoFile] data verify failure\n");
        return nn::repair::ResultVerifyFailure();
    }

    NN_RESULT_SUCCESS;
}

// micro sd に中断情報を保存する
nn::Result SaveMigrationInfoToSd(RepairAbortMigrationInfoFile *storage) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    nn::Result result;

    // wait sd attach
    std::unique_ptr<nn::repair::Sdcard> sdcard(new nn::repair::Sdcard());
    NN_RESULT_DO( sdcard->WaitAttach() );

    // mount
    NN_RESULT_DO( nn::fs::MountSdCardForDebug(g_VolumeSd.c_str()) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(g_VolumeSd.c_str());
    };

    std::string SerialFileName = (g_VolumeSd + s_MigrationInfoFile);
    result = nn::fs::CreateFile(SerialFileName.c_str(), sizeof(RepairAbortMigrationInfoFile));

    if( !result.IsSuccess() && !nn::fs::ResultPathAlreadyExists::Includes(result) )
    {
        NN_LOG("[fs] CreateFile %08x \n", result.GetInnerValueForDebug());
        return result;
    }

    NN_RESULT_DO(nn::fs::OpenFile(&handle, SerialFileName.c_str(), nn::fs::OpenMode::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int write_size = sizeof(RepairAbortMigrationInfoFile);

    storage->magic = infofile_magic;
    PutMac(storage);

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, (void *)storage, write_size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}
