﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <iostream>
#include <iomanip>
#include <sstream>

#include "RepairAbortMigration.h"

#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/repair/repair_ShutdownButton.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_Boot.h>

namespace {

    typedef enum
    {
        DeletionType_Unknown,
        DeletionType_Leave,
        DeletionType_NAOnly,
        DeletionType_All,
    } MigrationDeletionType;

    typedef enum
    {
        Ownership_Unknown,
        Ownership_Yes,
        Ownership_No,
    } MigrationOwnership;

    static MigrationState                       migration_state; // nn::migration::user::MigrationState
    static MigrationRole                        migration_role;  // original or target
    static bool                                 is_migration_user_exist;
    static RepairAbortMigrationInfoFile         s_migInfo;
    const uint64_t                              migration_volume_content_info_systemSaveDataId = 0x8000000000000131ull;


    const char* ToolName                        = "Abort User Transfer";
    const char* message_restart_nornally[]      = {
        "[Note] \n",
        "The user transfer is almost comleted.\n",
        "It is safer to restart SWITCH normally.\n",
        "Please Shutdown if you would like to do so. \n" };
    const char* message_restart_nornally_button = "No! Proceed anyway!";
    const char* message_proceed                 = "Do you want to proceed abortion? \n";
    const char* message_proceed_button          = "Yes, please!";
    const char* message_done                    = "==== Abort Done !! ====\n";
    const char* message_error                   = "==== Abort Error !! ====\n";
    const char* message_nodest[]                = {
        "[Caution!] \n",
        "No user transfer info from target unit. \n",
        "Make sure if you apply this tool to the target unit at first.\n",
        "It might cause unwanted user data lost if you won't. \n",
        "Can you accept it? \n",
    };
    const char* message_nodest_button           = "Sure. Go ahead";
    const char* message_default_user            = "Default user has been generated! \n";


    const std::string String_MigrationRole[]    = { "Original", "Target", "Unknown"};
    const std::string String_MigrationState[]   = { "Not in transfer", "0", "1", "2", "2", "2", "3", "4", "Unknown state" };
    // データが現在どちらに属しているか
    const std::string String_Ownership[]        = { "??", "Yes", "No"};
    const std::string String_NetworkAssociation[] = { "Associated" , "Not-associated"};
    const std::string String_UserExist[]        = { "Exist" , "Not-exist"};

    const std::string String_Action[]           = { "Nothing", "Leave", "Unlink NA Association", "Delete"};
    const std::string String_Warning_Again      = "(Do you really want this?)";

    const int   ToolMajorVersion     = 5;
    const int   ToolMinorVersion     = 0;

    static bool  s_isOperationFail   = false;
    static bool  s_isMountRomFail    = false;
    static nn::repair::StreamTextView*    s_pTextView;
    static const size_t AppletHeapSize = 312 * 1024 * 1024;                 //!< アプリケーション予約ヒープメモリサイズ
    static const size_t AppletAllocatableSize = 312 * 1024 * 1024;          //!< アプリケーション稼働ヒープ上限メモリサイズ
    static const size_t GraphicsSystemReservedMemorySize = 6 * 1024 * 1024; //!< NVNグラフィクス稼働予約メモリ領域サイズ

    nn::os::ThreadType s_WorkerThread;
    const int OperationStackSize = 192 * 4096;
    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t s_OperationStack[OperationStackSize];

    class UserInputButton : public nn::repair::LabelButton
    {
        private:
        glv::Label *m_label;
        nn::os::Event *m_event;

        public:
        UserInputButton(std::function<void()> mouseDownFunction) NN_NOEXCEPT
             : LabelButton(mouseDownFunction)
               , m_label(nullptr)
        {
            m_event = new nn::os::Event(nn::os::EventClearMode_AutoClear);
            this->disable(glv::Property::DrawBack);
        }
        virtual ~UserInputButton() NN_NOEXCEPT NN_OVERRIDE
        {
            delete m_event;
            if(m_label)
            {
                m_label->remove();
                delete m_label;
            }
        }
        void Signal() NN_NOEXCEPT
        {
            m_event->Signal();
        }
        bool TryWait() NN_NOEXCEPT
        {
            return m_event->TryWait();
        }
        void Enable() NN_NOEXCEPT
        {
            Disable();
            m_event->TryWait(); // event のクリア
            this->enable(glv::Property::Visible);
            this->enable(glv::Property::Controllable);

        }
        void Disable() NN_NOEXCEPT
        {
            this->disable(glv::Property::Visible);
            this->disable(glv::Property::Controllable);
            this->disable(glv::Property::Focused);
        }
        void SetLabel(std::string str) NN_NOEXCEPT
        {
            if(m_label)
            {
                m_label->remove();
                delete m_label;
            }
            m_label = new glv::Label(str);
            this->add( FitButton(m_label, 50, 4) );
            Enable();
        }
    };

    static UserInputButton *m_userInputButton[2];

    void UserInput0ButtonDownHandler() NN_NOEXCEPT
    {
        m_userInputButton[0]->Signal();
        NN_LOG("[DEBUG] button 0 down signal\n");
    }
    void UserInput1ButtonDownHandler() NN_NOEXCEPT
    {
        m_userInputButton[1]->Signal();
        NN_LOG("[DEBUG] button 1 down signal\n");
    }

    void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, size);
    }

    void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        free(addr);
    }

    void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }

    //!--------------------------------------------------------------------------------------
    //! @brief GLV用ユーザ定義アロケータ.
    //!--------------------------------------------------------------------------------------
    void* glvMemoryAllocator( const size_t size, const size_t beginAlignment ) NN_NOEXCEPT
    {
        return ::aligned_alloc( beginAlignment, size );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief GLV用ユーザ定義デアロケータ.
    //!--------------------------------------------------------------------------------------
    void glvMemoryDeallocator( void* address ) NN_NOEXCEPT
    {
        ::free( address );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ペリフェラルセットアップ
    //!--------------------------------------------------------------------------------------
    static void SetupPeripherals() NN_NOEXCEPT
    {
        // this memory allocation will be used from the nvn graphics systems at runtime.
        nv::SetGraphicsAllocator( NvAllocate, NvFree, NvReallocate, nullptr );
        nv::InitializeGraphics( ::malloc( GraphicsSystemReservedMemorySize ), GraphicsSystemReservedMemorySize );

        // this memory allocation interface will be used when compiling of shader code at runtime.
        glslcSetAllocator( NvAllocate, NvFree, NvReallocate, nullptr );

    }

    //!--------------------------------------------------------------------------------------
    //! @brief HID設定初期値
    //!--------------------------------------------------------------------------------------
    static const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration( glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary );

} // namespace

//!--------------------------------------------------------------------------------------
//! nninitStartup() is invoked before calling nnMain().
//! 256MB確保
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::SetMemoryHeapSize( AppletHeapSize ) );
    uintptr_t address;
    const size_t MallocMemorySize = AppletAllocatableSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::AllocateMemoryBlock( &address, MallocMemorySize ) );
    nn::init::InitializeAllocator( reinterpret_cast< void* >( address ), MallocMemorySize );

    // GLV内部で利用されるルートメモリアロケータを上書きします.
    // 上書きしない場合は C++11 の std::aligned_alloc 及び std::free が利用されます.
    glv::detail::ApplicationMemoryAllocator::AttachUserAllocator( glvMemoryAllocator, glvMemoryDeallocator );
}

//!--------------------------------------------------------------------------------------
//! @brief ルートサーフェイスコンテキスト
//!--------------------------------------------------------------------------------------
class RootSurfaceContext : public glv::Window, public glv::GLV, public glv::ApplicationLoopCallback
{
private:

public:
    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ.
    //! @details 指定の幅、高さの領域に対して、ページセレクトヘッダバーとページサーフェイスを追加します.
    //!--------------------------------------------------------------------------------------
    RootSurfaceContext( const unsigned width, const unsigned height ) NN_NOEXCEPT
         : glv::Window( width, height, "Main Window" ), glv::GLV()
    {
        // this->setGLV( *this );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief デストラクタ.
    //! @attention
    //! DropDownオブジェクトは内部で ListViewを利用していますが、親による自動解放が正しく動かないバグがある模様のため、
    //! 明示的に親( m_pSurfaceTable )が解放される前にデストラクションしています.
    //! m_pSurfaceTableもついでに.
    //! ※尚、GLVのViewは glv::SmartObjectクラスにより動的確保したオブジェクトは、親から外される際に自動的に delete されます.
    //!   明示的解放の場合には、親子関係の解除も含めるようにしてください.
    //!--------------------------------------------------------------------------------------
    virtual ~RootSurfaceContext() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ランタイムエンジンにアタッチされた際に呼ばれます.
    //! @see glv::ApplicationLoopCallback::OnLoopAttached( ApplicationLoopContext& )
    //!--------------------------------------------------------------------------------------
    virtual void OnLoopAttached( glv::ApplicationLoopContext& context ) NN_NOEXCEPT NN_OVERRIDE
    {
        ApplicationLoopCallback::OnLoopAttached( context );
        NN_LOG( "[GLV]OnLoopAttached\n" );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ランタイムエンジンからデタッチされた際に呼ばれます.
    //! @see glv::ApplicationLoopCallback::OnLoopDetached( ApplicationLoopContext& )
    //!--------------------------------------------------------------------------------------
    virtual void OnLoopDetached( glv::ApplicationLoopContext& context ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_LOG( "[GLV]OnLoopDetached\n" );
        ApplicationLoopCallback::OnLoopDetached( context );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ループ中の呼び出し.@n
    //! @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます.@n
    //! この時点ではまだ glvコンテキストのレンダリングは開始していません.@n
    //! また、このメソッドが呼び出されるフレームは OnLoopAfterSceneRendererと同じです.@n
    //! @return reserved.@n
    //! 現在はRequiredRestoration::RequireRestrationNothing を返却してください.
    //! @see glv::ApplicationLoopCallback::OnLoopBeforeSceneRenderer( ApplicationLoopContext&, const HidEvents& )
    //!--------------------------------------------------------------------------------------
    virtual const glv::RequiredRestoration OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( events );
        NN_UNUSED( context );
        if ( events.GetAvailableBasicPadCount() > 0 )
        {
            auto& bpad = events.GetBasicPad( 0 );
            if ( bpad.IsButtonDown<glv::BasicPadEventType::Button::R>() || bpad.IsButtonRepeat<glv::BasicPadEventType::Button::R>() )
            {
                NN_LOG( "[GLV]OnLoopBeforeSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }
        }
        return glv::RequiredRestoration::RequireRestrationNothing;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ループ中の呼び出し.@n
    //! @details glvシーンレンダラのレンダリングが終わった後に呼び出されます.@n
    //! また、このメソッドが呼び出されるフレームは OnLoopBeforeSceneRendererと同じです.@n
    //! @return reserved.@n
    //! 現在はRequiredRestoration::RequireRestrationNothing を返却してください.
    //! @see glv::ApplicationLoopCallback::OnLoopAfterSceneRenderer( ApplicationLoopContext&, const HidEvents& )
    //!--------------------------------------------------------------------------------------
    virtual const glv::RequiredRestoration OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( events );
        NN_UNUSED( context );
        const glv::DebugPadEventType& dpad = events.GetDebugPad();
        if ( true == dpad.HasAnyEvent() )
        {
            // SELECT + START 同時押し( 後押し許可版 )
            const glv::DebugPadEventType::ButtonSetType exit( nn::hid::DebugPadButton::Start::Mask | nn::hid::DebugPadButton::Select::Mask );
            if ( exit == ( dpad.GetButtons() & exit )  )
            {
                // アプリケーションループ退場要求
                glv::ApplicationFrameworkExit();
            }
            if ( dpad.IsButtonDown<nn::hid::DebugPadButton::L>() )
            {
                NN_LOG( "[GLV]OnLoopAfterSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }
        }
        else if ( events.GetAvailableBasicPadCount() > 0 )
        {
            const glv::BasicPadEventType& bpad = events.GetBasicPad( 0 );
            // SELECT + START 同時押し( 後押し許可版 )
            const glv::BasicPadEventType::ButtonSetType exit( glv::BasicPadEventType::Button::Start::Mask | glv::BasicPadEventType::Button::Select::Mask );
            if ( exit == ( bpad.GetButtons() & exit )  )
            {
                // アプリケーションループ退場要求
                glv::ApplicationFrameworkExit();
            }

            if ( bpad.IsButtonDown<glv::BasicPadEventType::Button::L>() )
            {
                NN_LOG( "[GLV]OnLoopAfterSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }
        }

        return glv::RequiredRestoration::RequireRestrationNothing;
    }

private:
};

void ShowErrorResult(std::string message, nn::Result &result) NN_NOEXCEPT
{
    std::ostringstream oss;
    std::string txt;

    oss.str("");
    oss << message;
    oss << "(0x";
    oss << std::hex << std::setw(8) << std::setfill('0') << result.GetInnerValueForDebug();
    oss << ")";
    oss << std::endl;
    txt = oss.str();

    s_pTextView->AppendValue(txt); // associated or not
    s_isOperationFail = true;
}

void DoWaitSingleButton(std::string str) NN_NOEXCEPT
{
    m_userInputButton[0]->pos(200, 640);
    m_userInputButton[0]->SetLabel(str);
    m_userInputButton[1]->Disable();

    while( !m_userInputButton[0]->TryWait())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    m_userInputButton[0]->Disable();
}

void DoWaitDoubleButton(int *pSelectedNo, std::string str0, std::string str1) NN_NOEXCEPT
{
    m_userInputButton[0]->pos(30, 640);
    m_userInputButton[0]->SetLabel(str0);
    m_userInputButton[1]->SetLabel(str1);

    while( NN_STATIC_CONDITION(true) )
    {
        if( m_userInputButton[0]->TryWait() )
        {
            *pSelectedNo = 0;
            break;
        }
        if( m_userInputButton[1]->TryWait() )
        {
            *pSelectedNo = 1;
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    m_userInputButton[0]->Disable();
    m_userInputButton[1]->Disable();
}

bool IsOceanMigrationFlagEnabled()
{
    nn::settings::system::AppletLaunchFlagSet flags;
    nn::settings::system::GetAppletLaunchFlags(&flags);

    return flags.Test<nn::settings::system::AppletLaunchFlag::Migration>();
}

//
// You get both info
//  - Original or Target
//  - His state
// and checking compatibility of format
//
// true   : migration is running
// false  : migration is not running or invalid state
//

bool GetMigrationState(MigrationRole *prole, MigrationState *pstate) NN_NOEXCEPT
{
    nn::migration::user::LastMigrationInfo info;

    //
    // SD からバックアップで抽出した State がある場合はそちらを取得する
    // 無い場合はシステムセーブデータからmigration APIで取得
    //
    bool is = LoadMigrationStateFromSd(&info).IsSuccess();

    if(!is)
    {
        //
        // state セーブデータの不整合状態により
        // この APIは abort で失敗する場合がある
        //
        is = nn::migration::GetLastUserMigrationStateInDetail(&info);
    }

    if(is)
    {
        *prole = (info.role == nn::migration::user::MigrationRole_Server) ? Role_Origin :
        ((info.role == nn::migration::user::MigrationRole_Client) ? Role_Destination : Role_Unknown );

        switch(info.state)
        {
            case nn::migration::user::MigrationState_Started:            *pstate = State_None;      break;
            case nn::migration::user::MigrationState_InInitialization:   *pstate = State_Initialize;break;
            case nn::migration::user::MigrationState_InTransfer:         *pstate = State_Transfer;  break;
            case nn::migration::user::MigrationState_InFinalization:     *pstate = (*prole == Role_Destination) ? State_BeforeFinalize_Delete : State_BeforeFinalize_PreDefined;  break;
            case nn::migration::user::MigrationState_Finalized:          *pstate = State_Finalize;  break;
            default:
            *pstate = State_Unknown;  break;
        }
    }
    else if( IsOceanMigrationFlagEnabled() )
    {
        // Ocean フラグだけ立ってるケース
        *pstate = State_OceanFinalize;
        is = true;
    }

    return is;
}

bool GetMigrationContent(MigrationRole role, MigrationState state, nn::account::Uid *puid) NN_NOEXCEPT
{
    // abort 禁止
    nn::fs::ScopedAutoAbortDisabler scopedAbortDisabler;
    nn::Result result;

    if( role == Role_Unknown )
    {
        // format が不確定の為読めない
        NN_LOG("[GetMigrationContent] no role and skip.\n");
        *puid = nn::account::InvalidUid;
        return true;
    }

    bool isGood = false;

    *puid = nn::account::InvalidUid;
    result = nn::fs::MountSystemSaveData( g_VolumeSavedata.c_str() ,
                                          nn::fs::SaveDataSpaceId::System ,
                                          migration_volume_content_info_systemSaveDataId,
                                          nn::fs::InvalidUserId);
    if( nn::fs::ResultTargetNotFound::Includes(result) )
    {
        NN_LOG("[GetMigrationContent] the volume is not found.\n");
        *puid = nn::account::InvalidUid;
        isGood = true;
        return isGood;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(g_VolumeSavedata.c_str());
    };

    NN_LOG("\n MountSystemSaveData %08x\n", result.GetInnerValueForDebug());

    // migrationInfo.bin は
    // role によって 構造体が異なる

    nn::fs::FileHandle handle;
    const std::string targetFilePath = g_VolumeSavedata + ":/"  + "migrationInfo.bin";

    result = nn::fs::OpenFile(&handle, targetFilePath.c_str(), nn::fs::OpenMode_Read);
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        NN_LOG("[GetMigrationContent] no file.\n");
        *puid = nn::account::InvalidUid;
        isGood = true;
    }
    else
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_LOG("\n OpenFile %08x\n", result.GetInnerValueForDebug());

        size_t  read_size = 0;
        const size_t BufferSize = 1024;
        std::unique_ptr<char []> buffer(new char[BufferSize]);
        struct nn::migration::user::ServerInfo originInfo;
        struct nn::migration::user::ClientInfo destInfo;

        NN_RESULT_DO(nn::fs::ReadFile(&read_size, handle, 0, buffer.get(), BufferSize));

        // format check
        // fwVersion may have to be checked in the future.(currently ignored)
        if( role == Role_Origin )
        {
            NN_ABORT_UNLESS( read_size <= sizeof(originInfo) , "Invalid file size");
            std::memcpy(&originInfo, buffer.get(), read_size);
            *puid = originInfo.user; // 使うのはこのデータだけ

            s_migInfo.role      = role;
            s_migInfo.state     = state;
            s_migInfo.fwVersion = originInfo.fwVersion;
            s_migInfo.sessionId = originInfo.sessionId;
            s_migInfo.user      = originInfo.user;
        }
        else if( role == Role_Destination )
        {
            NN_ABORT_UNLESS( read_size <= sizeof(destInfo) , "Invalid file size");
            std::memcpy(&destInfo, buffer.get(), read_size);
            *puid = destInfo.serverInfo.user; // 使うのはこのデータだけ

            s_migInfo.role      = role;
            s_migInfo.state     = state;
            s_migInfo.fwVersion = destInfo.serverInfo.fwVersion;
            s_migInfo.sessionId = destInfo.serverInfo.sessionId;
            s_migInfo.user      = destInfo.serverInfo.user;
        }

        if( read_size == BufferSize )
        {
            NN_LOG("[GetMigrationContent] migrationInfo.bin is over %d byte\n", read_size);
        }

        isGood = true;

    }

    return isGood;
}

void SetOceanMigrationFlag(bool onBit ) NN_NOEXCEPT
{
    nn::settings::system::AppletLaunchFlagSet flags;
    nn::settings::system::GetAppletLaunchFlags(&flags);

    flags.Set<nn::settings::system::AppletLaunchFlag::Migration>(onBit);

    nn::settings::system::SetAppletLaunchFlags(flags);
}

MigrationDeletionType GetDeletionType(MigrationRole role , MigrationState state) NN_NOEXCEPT
{
    MigrationDeletionType deltype = DeletionType_Unknown;
    if(role == Role_Origin)
    {
        switch(state)
        {
            case State_Initialize:
            case State_Transfer:
            case State_BeforeFinalize_Leave:
            case State_OceanFinalize: // Role を取得できないので実際にはこのケース文は使われない
            deltype = DeletionType_Leave;
            break;

            case State_BeforeFinalize_Delete:
            case State_Finalize:
            deltype = DeletionType_All;
            break;

            default:
            break;
        }
    }
    else if(role == Role_Destination)
    {
        switch(state)
        {
            case State_Initialize:
            case State_Transfer:
            case State_BeforeFinalize_Delete:
            deltype = DeletionType_All;
            break;

            case State_Finalize:
            deltype = DeletionType_NAOnly;
            break;

            case State_OceanFinalize: // Role を取得できないので実際にはこのケース文は使われない
            deltype = DeletionType_Leave;
            break;

            default:
            break;
        }
    }

    return deltype;
}

MigrationOwnership GetOwnership(MigrationRole role , MigrationState state) NN_NOEXCEPT
{
    MigrationOwnership ownership = Ownership_Unknown;
    if(role == Role_Origin)
    {
        switch(state)
        {
            case State_Initialize:
            case State_Transfer:
            case State_BeforeFinalize_Leave:
            ownership = Ownership_Yes;
            break;

            case State_BeforeFinalize_Delete:
            case State_Finalize:
            case State_OceanFinalize: // Role を取得できないので実際にはこのケース文は使われない
            ownership = Ownership_No;
            break;

            default:
            break;
        }
    }
    else if(role == Role_Destination)
    {
        switch(state)
        {
            case State_Initialize:
            case State_Transfer:
            case State_BeforeFinalize_Delete:
            ownership = Ownership_No;
            break;

            case State_Finalize:
            case State_OceanFinalize: // Role を取得できないので実際にはこのケース文は使われない
            ownership = Ownership_Yes;
            break;

            default:
            break;
        }
    }

    return ownership;
}

void DeleteUserDataIf(nn::account::Uid user, bool isUAExist) NN_NOEXCEPT
{
    nn::Result result;

    MigrationDeletionType deltype = GetDeletionType(migration_role, migration_state);
    if( deltype == DeletionType_All || deltype == DeletionType_NAOnly )
    {
        // NA 連携を解除する (要インターネット接続)

        bool isLinked = false;
        if( isUAExist )
        {
            nn::account::NetworkServiceAccountAdministrator admin;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
            NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&isLinked));
        }

        // NA 連携あり
        if( isLinked )
        {
            // checking sd card insertion and read
            s_pTextView->AppendValue("reading uSD Card... \n" );
            NN_ABORT_UNLESS_RESULT_SUCCESS( ImportNetworkSettingsFromSdCard() );

            //Wifi 設定を有効にする
            EnableWifiConnection();
            NN_UTIL_SCOPE_EXIT
            {
                //処理が終わったら機内モードを戻す
                RestoreAirplaneMode();
            };

            s_pTextView->AppendValue("Connecting to internet...");
            bool isConnected = WaitNetworkConnectionAvailable();
            NN_UTIL_SCOPE_EXIT
            {
                ReleaseNetworkConnection();
            };

            if( isConnected )
            {
                s_pTextView->AppendValue("  done \n");


                s_pTextView->AppendValue("Unregister network account... " );

                // NA 連携の解除
                bool isDeviceLinkLeft = false;
                result = UnregisterNetworkAccountAssociation(user, (deltype == DeletionType_All), &isDeviceLinkLeft);

                if(result.IsSuccess())
                {
                    s_pTextView->AppendValue("  done.\n" );
                }
                else
                {
                    ShowErrorResult("FAIL.", result);
                }

                if(isDeviceLinkLeft)
                {
                    s_pTextView->AppendValue("Fail to release Device-Link. \n" );
                    s_pTextView->AppendValue("Please try unlink it by using UMT. \n" );
                }
            }
            else
            {
                ShowErrorResult("FAIL.", result);
                NN_LOG("Can't connect to internet.");
            }

        }
        else if(migration_role == Role_Destination)
        {
            // NA 連携が無くても
            // 移行によって存在する可能性
            result = DeleteNetworkAccountRelatedInformation(user);

            // SUCCESS の必要なし
            NN_LOG("DeleteNetworkAccountRelatedInformation(%08x)", result.GetInnerValueForDebug() );
        }
    }

    if( deltype == DeletionType_All)
    {
        result = DeleteUserAccountWithData(user);
        if(!result.IsSuccess())
        {
            // ユーザアカウントの削除に失敗した
            NN_LOG("DeleteUserWithData(%08x)", result.GetInnerValueForDebug() );
            ShowErrorResult("FAIL to delete user data.", result);
        }
    }
}

void CreateDefaultUserIf() NN_NOEXCEPT
{
    int user_count;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::account::GetUserCount(&user_count));

    if( user_count > 0 )
    {
        return;
    }

    NN_LOG("== no user ==\n");

    // ダミーアカウントの作成
    GenerateDefaultUser();

    s_pTextView->AppendValue(message_default_user);
}

void ShowMigrationInfo(nn::account::Uid user, bool isWarning) NN_NOEXCEPT
{
    // role 不明の場合 移行対象 user を取得できない
    if( migration_role == Role_Unknown )
    {
        s_pTextView->AppendValue("========================================================= \n");
        s_pTextView->AppendValue("State   \t\t: " + String_MigrationState[migration_state] + "\n");
        s_pTextView->AppendValue("========================================================= \n");
        return;
    }

    MigrationOwnership ownership = GetOwnership(migration_role, migration_state);

    s_pTextView->AppendValue("========================================================= \n");
    s_pTextView->AppendValue("Role    \t\t: " + String_MigrationRole[migration_role]   + "\n"); // original or target
    s_pTextView->AppendValue("State   \t\t: " + String_MigrationState[migration_state] + "\n");
    s_pTextView->AppendValue("Ownership \t: " + String_Ownership[ownership]            + "\n"); // "Yes or No"

    std::string txt = is_migration_user_exist ? String_UserExist[0] : String_UserExist[1];
    s_pTextView->AppendValue("Target User \t: " + txt + "\n"); // Exist or Not Exist

    if( is_migration_user_exist )
    {
        std::ostringstream oss;
        std::string txt;

        // NA 連携情報
        nn::account::NetworkServiceAccountAdministrator admin;
        bool isNsa = false;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&isNsa));
        // admin.IsLinkedWithNintendoAccount(&isLinked); は不要

        txt = isNsa ? String_NetworkAssociation[0] : String_NetworkAssociation[1];
        s_pTextView->AppendValue("  NetworkAccount\t: " + txt + "\n"); // associated or not

        // user savedata info
        nn::ns::UserSaveDataStatistics stat;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::ns::CalculateUserSaveDataStatistics(&stat, user) );

        int mbyte = stat.sizeInBytes / (1024 * 1024);
        oss.str("");
        oss << std::setw(4)  << std::setfill(' ') << (int) stat.count << " files";
        oss << "(total " << std::setw(10) << std::setfill(' ') << mbyte << " MB)";
        oss << std::endl;
        txt = oss.str();

        s_pTextView->AppendValue("  SaveData\t: " + txt); // count and size

        MigrationDeletionType deltype = GetDeletionType(migration_role, migration_state);

        if( isWarning )
        {
            s_pTextView->AppendValue("  Action  \t: " + String_Action[deltype] + String_Warning_Again + "\n");
        }
        else
        {
            s_pTextView->AppendValue("  Action  \t: " + String_Action[deltype] + "\n"); // Leave or Delete
        }

    }
    s_pTextView->AppendValue("========================================================= \n");

}

bool DetermineActionFromOpponentInfo()
{
    bool isProperOpponent = false;
    bool isGetOpponentInfo = false;
    if( migration_state == State_BeforeFinalize_PreDefined )
    {
        isGetOpponentInfo = true;
        // SD card から修理用一時情報を読み出す(移行先のデータ)

        s_pTextView->AppendValue("Checking uSD Card ... \n");
        RepairAbortMigrationInfoFile op_info;
        nn::Result result = LoadMigrationInfoFromSd(&op_info);
        if(!result.IsSuccess())
        {
            NN_LOG("Fail reading from sd.(%08x)", result.GetInnerValueForDebug());
        }

        // 正しい移行対向デバイスかチェックする
        else
        {
            char buf0[40],buf1[40];
            op_info.sessionId.ToString(buf0,40);
            op_info.sessionId.ToString(buf1,40);

            NN_LOG("[info]magic(%08x)\n", op_info.magic);
            NN_LOG("[info]fwVersion(%08x,%08x)\n", op_info.fwVersion, s_migInfo.fwVersion);
            NN_LOG("[info]sessionId(%s,%s)\n",buf0 ,buf1);
            NN_LOG("[info]user(%016llx,%016llx)\n", op_info.user._data[0], s_migInfo.user._data[0]);
            NN_LOG("[info]state(%d,%d)\n", op_info.state);

            if(
                s_migInfo.role == Role_Origin &&
                op_info.role == Role_Destination &&
                op_info.fwVersion == s_migInfo.fwVersion &&
                op_info.sessionId == s_migInfo.sessionId &&
                op_info.user == s_migInfo.user
                )
            {
                isProperOpponent = true;
            }
        }

        if(isProperOpponent && (GetOwnership(Role_Destination, op_info.state) == Ownership_No))
        {
            migration_state = State_BeforeFinalize_Leave;
        }
        else
        {
            migration_state = State_BeforeFinalize_Delete;
        }

        if(isProperOpponent)
        {
            NN_LOG("[application]The action has to be determined by proper transfer info of target unit.");
        }
        else
        {
            s_pTextView->Clear();
            const int num = sizeof(message_nodest) / sizeof(message_nodest[0]);
            for(int i=0; i<num; i++ )
            {
                s_pTextView->AppendValue(message_nodest[i]);
            }

            DoWaitSingleButton(message_nodest_button);
        }
    }

    return (isGetOpponentInfo && !isProperOpponent);
}

void OperationFunction() NN_NOEXCEPT
{
    nn::account::Uid migration_user = nn::account::InvalidUid;

    s_isOperationFail   = false;

    if(s_isMountRomFail)
    {
        s_pTextView->AppendValue("[Fatal] No Rom resource data!");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
    }

    s_pTextView->AppendValue("[[ Operation Start. ]]\n");

#ifdef FUNCTIONAL_DEBUG
    nn::settings::system::SetInRepairProcessEnabled(false);
    ShowSystemSaveDataList();
    ShowUserSaveDataList();
    DoWaitSingleButton("Now Debug Mode");
#endif

    migration_state = State_None; // nn::migration::user::MigrationState
    migration_role  = Role_Unknown;  // original or target

    GetMigrationState(&migration_role, &migration_state);

#ifdef FUNCTIONAL_DEBUG

    PickupRole8State(&migration_role, &migration_state);

    // 一人選ぶ
    PickupUserFromAccountList(&migration_user);

    // migration content savedata を作る
    CreateMigrtionSaveDataForDebug(0x8000000000000131ull, 0x010000000000003Aull,
                                   migration_role, migration_user);

#endif

    GetMigrationContent(migration_role, migration_state, &migration_user);

    if (migration_user != nn::account::InvalidUid)
    {
        // UA の存在確認
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserExistence(&is_migration_user_exist, migration_user));

        if (migration_role == Role_Destination)
        {
            // SD card に修理用一時情報を保存する
            s_pTextView->AppendValue("checking uSD Card... \n" );
            nn::Result result = SaveMigrationInfoToSd(&s_migInfo);
            if (!result.IsSuccess())
            {
                ShowErrorResult("Fail writing to uSD Card.", result);
            }
        }
    }

    bool isWarning = DetermineActionFromOpponentInfo();

    // 情報を画面表示
    s_pTextView->Clear();
    ShowMigrationInfo(migration_user, isWarning);

    if( migration_state == State_None || migration_state == State_Unknown )
    {
        NN_LOG("I have nothing to do.");
        return;
    }

    if( migration_state == State_OceanFinalize )
    {
        // Ocean のフラグは立っているが
        // migration プロセス側の情報は無い
        // (Ocean(qlaunch) は 自身のセーブデータに role 情報を持っている模様)

        // role が不明だが
        // 解除をするか確認する
        const int num = sizeof(message_restart_nornally) / sizeof(message_restart_nornally[0]);
        for(int i=0; i<num; i++ )
        {
            s_pTextView->AppendValue(message_restart_nornally[i]);
        }

        DoWaitSingleButton(message_restart_nornally_button);
    }
    else
    {
        s_pTextView->AppendValue(message_proceed);
        DoWaitSingleButton(message_proceed_button);
    }

    // ユーザ情報の削除
    DeleteUserDataIf(migration_user, is_migration_user_exist);

    // ユーザが０人になってしまう場合は
    // ダミーのユーザを作成する
    CreateDefaultUserIf();

    // 以下二つはリストアツールで実施
    // migration state 削除 (できない)
    // migration contents 削除 (できるけどしない)
#ifndef FUNCTIONAL_DEBUG
    // 修理中フラグを立てる（通常起動するとFATAL）
    nn::settings::system::SetInRepairProcessEnabled(true);
#endif

    // Ocean migration flag unset
    SetOceanMigrationFlag(false);

    s_pTextView->AppendValue(s_isOperationFail ? message_error : message_done);

}

//!--------------------------------------------------------------------------------------
//! @brief メイン
//!--------------------------------------------------------------------------------------

void MyMain() NN_NOEXCEPT
{
    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    RootSurfaceContext* context = new RootSurfaceContext( width, height );

    // バージョン表示
    auto toolInformationLabel = new glv::Label;
    nn::repair::GetToolInformationLabel(toolInformationLabel, "", ToolMajorVersion, ToolMinorVersion);
    *context << toolInformationLabel;

    // ログビュー表示
    s_pTextView = new nn::repair::StreamTextView(glv::Rect(1024, 512), 32.f);
    s_pTextView->pos(120, 100);
    *context << s_pTextView;

    // タイトルラベル表示
    glv::Label* titleLabel = new glv::Label(ToolName, false);
    titleLabel->pos(10, 25);
    titleLabel->size(32.f);
    *context << titleLabel;

    // シャットダウンボタン(FW 3.0.0 以降で有効)
    glv::Button* shutdownButton = new nn::repair::ShutdownButton();
    shutdownButton->pos(800,640);
    *context << shutdownButton;

    // Yes No ボタン
    m_userInputButton[0] = new UserInputButton(UserInput0ButtonDownHandler);
    m_userInputButton[1] = new UserInputButton(UserInput1ButtonDownHandler);
    m_userInputButton[0]->pos(200, 640);
    m_userInputButton[1]->pos(600, 640);
    m_userInputButton[0]->Disable();
    m_userInputButton[1]->Disable();

    *context << m_userInputButton[0];
    *context << m_userInputButton[1];

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    context->setGLV(*context);

    // メインループコールバックを登録.
    ApplicationFrameworkRegisterLoopCallback( context );

    // 使用するライブラリの初期化
    nn::ns::Initialize();
    nn::account::InitializeForAdministrator();
    nn::friends::Initialize();
    nn::npns::InitializeForSystem();
    nn::Result result = nn::npns::Suspend(); // resume はしない
    NN_LOG("[npns] suspend (%08x)\n", result.GetInnerValueForDebug());

    s_isMountRomFail = !(MountRom().IsSuccess());
    InitializeWifi();
    InitMac();

    // メイン処理用のスレッド
    // (TBD) OnLoopAttached イベントまで処理開始を待たせる
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&s_WorkerThread,
                             (nn::os::ThreadFunction)OperationFunction,
                             NULL,
                             s_OperationStack,
                             OperationStackSize,
                             30));

    nn::os::StartThread(&s_WorkerThread);


    glv::Application::run();

    if ( nullptr != context )
    {
        delete context;
    }

    nn::ns::Finalize();
    nn::npns::FinalizeForSystem();
    nn::account::Finalize();

}

//!--------------------------------------------------------------------------------------
//! @brief プロセスエントリポイント
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nnMain() NN_NOEXCEPT
{
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(LocalHidConfiguration);

    MyMain();
}
