﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/nn_Log.h>

#include "Hid.h"
#include "Draw.h"

namespace {
    nn::hid::NpadIdType g_NpadIds[] = { nn::hid::NpadId::No1 };
    const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(nn::hid::NpadIdType);
    nn::hid::NpadJoyDualState currentNpadJoyDualState[NpadIdCountMax * 2];
}

void PrintNpadButtonState(const nn::hid::NpadButtonSet& state)
{
    char buttons[37];
    buttons[0] = (state.Test<nn::hid::NpadJoyButton::A>()) ? 'A' : '_';
    buttons[1] = (state.Test<nn::hid::NpadJoyButton::B>()) ? 'B' : '_';
    buttons[2] = (state.Test<nn::hid::NpadJoyButton::X>()) ? 'X' : '_';
    buttons[3] = (state.Test<nn::hid::NpadJoyButton::Y>()) ? 'Y' : '_';
    buttons[4] = (state.Test<nn::hid::NpadJoyButton::StickL>()) ? 'L' : '_';
    buttons[5] = (state.Test<nn::hid::NpadJoyButton::StickL>()) ? 'S' : '_';
    buttons[6] = (state.Test<nn::hid::NpadJoyButton::StickR>()) ? 'R' : '_';
    buttons[7] = (state.Test<nn::hid::NpadJoyButton::StickR>()) ? 'S' : '_';
    buttons[8] = (state.Test<nn::hid::NpadJoyButton::L>()) ? 'L' : '_';
    buttons[9] = (state.Test<nn::hid::NpadJoyButton::R>()) ? 'R' : '_';
    buttons[10] = (state.Test<nn::hid::NpadJoyButton::ZL>()) ? 'Z' : '_';
    buttons[11] = (state.Test<nn::hid::NpadJoyButton::ZL>()) ? 'L' : '_';
    buttons[12] = (state.Test<nn::hid::NpadJoyButton::ZR>()) ? 'Z' : '_';
    buttons[13] = (state.Test<nn::hid::NpadJoyButton::ZR>()) ? 'R' : '_';
    buttons[14] = (state.Test<nn::hid::NpadJoyButton::Plus>()) ? '+' : '_';
    buttons[15] = (state.Test<nn::hid::NpadJoyButton::Minus>()) ? '-' : '_';
    buttons[16] = (state.Test<nn::hid::NpadJoyButton::Left>()) ? '<' : '_';
    buttons[17] = (state.Test<nn::hid::NpadJoyButton::Up>()) ? '^' : '_';
    buttons[18] = (state.Test<nn::hid::NpadJoyButton::Right>()) ? '>' : '_';
    buttons[19] = (state.Test<nn::hid::NpadJoyButton::Down>()) ? 'v' : '_';
    buttons[20] = (state.Test<nn::hid::NpadJoyButton::LeftSL>()) ? 'S' : '_';
    buttons[21] = (state.Test<nn::hid::NpadJoyButton::LeftSL>()) ? 'L' : '_';
    buttons[22] = (state.Test<nn::hid::NpadJoyButton::LeftSR>()) ? 'S' : '_';
    buttons[23] = (state.Test<nn::hid::NpadJoyButton::LeftSR>()) ? 'R' : '_';
    buttons[24] = (state.Test<nn::hid::NpadJoyButton::RightSL>()) ? 'S' : '_';
    buttons[25] = (state.Test<nn::hid::NpadJoyButton::RightSL>()) ? 'L' : '_';
    buttons[26] = (state.Test<nn::hid::NpadJoyButton::RightSR>()) ? 'S' : '_';
    buttons[27] = (state.Test<nn::hid::NpadJoyButton::RightSR>()) ? 'R' : '_';
    buttons[28] = (state.Test<nn::hid::NpadJoyButton::StickLRight>()) ? '>' : '_';
    buttons[29] = (state.Test<nn::hid::NpadJoyButton::StickLUp>()) ? '^' : '_';
    buttons[30] = (state.Test<nn::hid::NpadJoyButton::StickLLeft>()) ? '<' : '_';
    buttons[31] = (state.Test<nn::hid::NpadJoyButton::StickLDown>()) ? 'v' : '_';
    buttons[32] = (state.Test<nn::hid::NpadJoyButton::StickRRight>()) ? '>' : '_';
    buttons[33] = (state.Test<nn::hid::NpadJoyButton::StickRUp>()) ? '^' : '_';
    buttons[34] = (state.Test<nn::hid::NpadJoyButton::StickRLeft>()) ? '<' : '_';
    buttons[35] = (state.Test<nn::hid::NpadJoyButton::StickRDown>()) ? 'v' : '_';
    buttons[36] = '\0';

    NN_LOG("%s", buttons);
}

void PrintNpadStickState(const nn::hid::AnalogStickState& state)
{
    NN_LOG("(%6d, %6d)", state.x,
        state.y);
}


template <typename TState>
void PrintNpadState(const TState& state)
{
    NN_LOG("[%6lld] ", state.samplingNumber);
    PrintNpadButtonState(state.buttons);
    NN_LOG(" L");
    PrintNpadStickState(state.analogStickL);
    NN_LOG(" R");
    PrintNpadStickState(state.analogStickR);
    NN_LOG("\n");
}


void Init()
{
    nn::hid::InitializeNpad();

    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleJoyDual::Mask);

    // 使用する Npad を設定
    nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);
}

void WaitJoyInput(int index)
{
    NN_LOG("Waiting Joy-con input");
    for (;;)
    {
        for (int i = 0; i < NpadIdCountMax; i++)
        {
            //現在有効な操作形態(NpadStyleSet)を取得
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);

            // Joy-Con 操作が有効な場合
            if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
            {
                //最新のNpadのステートを取得
                nn::hid::GetNpadState(&(currentNpadJoyDualState[i]), g_NpadIds[i]);

                if (currentNpadJoyDualState[i].buttons.Test(index))
                {
                    NN_LOG(" OK\n");
                    return;
                }
            }
        }
        nn::os::YieldThread();
    }
}

void WaitJoyRelase()
{
    NN_LOG("Waiting Joy-con button is released");
    for (;;)
    {
        bool released = false;
        for (int i = 0; i < NpadIdCountMax; i++)
        {
            //現在有効な操作形態(NpadStyleSet)を取得
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);

            // Joy-Con 操作が有効な場合
            if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
            {
                //最新のNpadのステートを取得
                nn::hid::GetNpadState(&(currentNpadJoyDualState[i]), g_NpadIds[i]);

                if (currentNpadJoyDualState[i].buttons.IsAllOff())
                {
                    released = true;
                    NN_LOG("id %d is all released\n", i);
                }
            }
        }
        if (released)
        {
            return;
        }
        nn::os::YieldThread();
    }
}

void Update(ButtonSampleResult& result)
{
    for (int i = 0; i < NpadIdCountMax; i++)
    {
        //現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);

        // Joy-Con 操作が有効な場合
        if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
        {
            //最新のNpadのステートを取得
            nn::hid::GetNpadState(&(currentNpadJoyDualState[i]), g_NpadIds[i]);

            result.all++;
            //Aボタンのみが押された
            if (currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::A>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::B>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::X>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Y>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickL>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickR>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::L>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::R>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::ZL>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::ZR>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Plus>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Minus>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Left>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Up>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Right>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Down>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickLLeft>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickLUp>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickLRight>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickLDown>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickRLeft>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickRUp>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickRRight>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickRDown>()
                )
            {
                result.aPushed++;
            }

            //Bボタンのみが押された
            if (!currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::A>() &&
                currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::B>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::X>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Y>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickL>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickR>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::L>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::R>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::ZL>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::ZR>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Plus>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Minus>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Left>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Up>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Right>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::Down>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickLLeft>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickLUp>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickLRight>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickLDown>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickRLeft>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickRUp>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickRRight>() &&
                !currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadJoyButton::StickRDown>()
                )
            {
                result.bPushed++;
            }
        }
    }
}

void SampleJoyInputs(ButtonSampleResult& result)
{
    auto start = std::chrono::system_clock::now();
    for (;;)
    {
        Update(result);
        auto end = std::chrono::system_clock::now();
        if (std::chrono::duration_cast<std::chrono::milliseconds>(end - start).count() >= 3000)
        {
            break;
        }
        nn::os::YieldThread();
    }
}

void InspectJoyCon(glv::GLV& glvRootView)
{
    Init();

    NN_LOG("release All button\n");

    auto releaseAllButton = new glv::Label("Release all button of Joy-Con Right", glv::Label::Spec(glv::Place::TL, ResultColOffset, JoyConTestRowOffset, SmallFontSize));
    glvRootView << releaseAllButton;

    WaitJoyRelase();

    releaseAllButton->remove();

    NN_LOG("press A button\n");

    auto waitAButton = new glv::Label("Hold A button of Joy-Con Right for 3 seconds", glv::Label::Spec(glv::Place::TL, ResultColOffset, JoyConTestRowOffset, SmallFontSize));
    glvRootView << waitAButton;

    // Aボタンの入力を待つ
    WaitJoyInput(0);

    auto sampling = new glv::Label("Testing...", glv::Label::Spec(glv::Place::TL, ResultColOffset, JoyConTestRowOffset + 25, SmallFontSize));
    glvRootView << sampling;

    // Aボタンが3秒のうち90%正しく検出できるかテスト
    ButtonSampleResult result = { 0, 0, 0 };
    SampleJoyInputs(result);
    waitAButton->remove();
    sampling->remove();
    if (result.aPushed >= result.all * 0.9)
    {
        NN_LOG("A button Test is OK\n");
    }
    else
    {
        NN_LOG("A button Test is NG\n");
        auto aButtonNG = new glv::Label("Test of A button failed", glv::Label::Spec(glv::Place::TL, ResultColOffset, JoyConTestRowOffset, FontSize));
        aButtonNG->style(GetStyleRed());
        glvRootView << aButtonNG;
        return;
    }

    NN_LOG("release All button\n");
    glvRootView << releaseAllButton;
    WaitJoyRelase();

    releaseAllButton->remove();

    NN_LOG("press B button\n");

    auto waitBButton = new glv::Label("Hold B button of Joy-Con Right for 3 seconds", glv::Label::Spec(glv::Place::TL, ResultColOffset, JoyConTestRowOffset, SmallFontSize));
    glvRootView << waitBButton;

    result.all = 0;
    result.aPushed = 0;
    result.bPushed = 0;
    // Bボタンの入力を待つ
    WaitJoyInput(1);

    glvRootView << sampling;
    // Bボタンが3秒のうち90%正しく検出できるかテスト
    SampleJoyInputs(result);
    waitBButton->remove();
    sampling->remove();
    if (result.bPushed >= result.all * 0.9)
    {
        NN_LOG("B button Test is OK\n");
    }
    else
    {
        NN_LOG("B button Test is NG\n");
        auto BButtonNG = new glv::Label("Test of B button failed", glv::Label::Spec(glv::Place::TL, ResultColOffset, JoyConTestRowOffset, FontSize));
        BButtonNG->style(GetStyleRed());
        glvRootView << BButtonNG;
        return;
    }

    auto joyConOK = new glv::Label("Joy-Con is OK", glv::Label::Spec(glv::Place::TL, ResultColOffset, JoyConTestRowOffset, FontSize));
    joyConOK->style(GetStyleGreen());
    glvRootView << joyConOK;
}
