﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <functional>
#include <string>
#include <sstream>
#include <vector>
#include <algorithm>

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/fs.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>

#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/util/util_FormatString.h>

#include "../Common/Allocator.h"
#include "../Common/Fan.h"
#include "../Common/Hid.h"
#include "../Common/Draw.h"

#include <chrono>
#include "../Ldn/LdnStation/LdnStation.h"



namespace {
// for thread

    const size_t           ThreadStackSize = 32768;              // スレッド操作スレッドのスタックサイズ
    NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ThreadStackSize];   // 1つ目のスレッドのスタック

    nn::os::ThreadType  g_Thread1;
//


    const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;

    glv::GLV s_GlvRootView;

    void ReturnToDevmenu() NN_NOEXCEPT
    {
        NN_LOG("return button pushed\n");
        glv::ApplicationFrameworkExit();
    }


    class LabelButton : public glv::Button
    {
    public:
        explicit LabelButton(const std::string& text, std::function<void()> func) NN_NOEXCEPT
            : Button(glv::Rect(20), true), m_func(func)
        {
            this->add(FitButton(new glv::Label(text, false), FontSize, 4));
        }

        virtual bool onEvent(glv::Event::t e, glv::GLV& g) NN_NOEXCEPT NN_OVERRIDE
        {
            switch (e)
            {
            case glv::Event::MouseDown:
            {
                m_func();
            }
            break;
            default:
                break;
            }
            return true;
        }

    private:
        std::function<void()> m_func;

        glv::Label* FitButton(glv::Label* pLabel, float fontSize, glv::space_t padding) NN_NOEXCEPT
        {
            pLabel->size(fontSize);
            pLabel->paddingX(padding);
            pLabel->paddingY(padding);
            pLabel->pos(glv::Place::CC, 0, 0).anchor(glv::Place::CC);
            glv::space_t width = (pLabel->width() + 1.f) / 2.f * 2.f;
            glv::space_t height = (pLabel->height() + 1.f) / 2.f * 2.f;
            glv::Button::extent(width, height);
            return pLabel;
        }
    };

    static void SetupPeripherals() NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#if defined( NN_BUILD_CONFIG_SPEC_NX )
        nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
        nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
#endif // defined( NN_BUILD_CONFIG_SPEC_NX )
#if NN_GFX_IS_TARGET_NVN
        glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
#endif // defined( NN_GFX_IS_TARGET_NVN )
#endif // defined( NN_BUILD_CONFIG_OS_HORIZON )
    }

    void Main(glv::Window& window) NN_NOEXCEPT
    {
        glv::GLV* const pGlvRootView = &s_GlvRootView;

        auto exitButton = new LabelButton("Return to Devmenu", [&] { ReturnToDevmenu(); });
        exitButton->pos(glv::Place::BC, 0, 0).anchor(glv::Place::BC);

        auto fanTest = glv::Label("Fan Test", glv::Label::Spec(glv::Place::TL, 0, 0, FontSize));
        auto joyconTest = glv::Label("Joy-Con Test", glv::Label::Spec(glv::Place::TL, 0, JoyConTestRowOffset, FontSize));
        auto wlanTest = glv::Label("WLAN Test", glv::Label::Spec(glv::Place::TL, 0, WLANTestRowOffset, FontSize));

        *pGlvRootView <<  fanTest
            << joyconTest
            << wlanTest
            << exitButton;

        window.setGLV(*pGlvRootView);

        glv::Application::run();
    }
}

void InspectFan()
{
    auto isFanValid = IsFanValid();

    auto fanResult = isFanValid ? new glv::Label("Fan is OK", glv::Label::Spec(glv::Place::TL, ResultColOffset, 0, FontSize)) :
        new glv::Label("Fan is NG", glv::Label::Spec(glv::Place::TL, ResultColOffset, 0, FontSize));
    if (isFanValid)
    {
        fanResult->style(GetStyleGreen());
    }
    else
    {
        fanResult->style(GetStyleRed());
    }

    s_GlvRootView << fanResult;
}


void InspectWlan()
{
    auto testing = new glv::Label("Testing...", glv::Label::Spec(glv::Place::TL, ResultColOffset, WLANTestRowOffset, SmallFontSize));
    s_GlvRootView << testing;

    auto isWlanValid = LdnStationMain();
    testing->remove();

    NN_LOG("wlan result = %d", isWlanValid);
    auto wlanResult = isWlanValid ? new glv::Label("WLAN is OK", glv::Label::Spec(glv::Place::TL, ResultColOffset, WLANTestRowOffset, FontSize)) :
        new glv::Label("WLAN is NG", glv::Label::Spec(glv::Place::TL, ResultColOffset, WLANTestRowOffset, FontSize));
    if (isWlanValid)
    {
        wlanResult->style(GetStyleGreen());
    }
    else
    {
        wlanResult->style(GetStyleRed());
    }

    s_GlvRootView << wlanResult;
}

void ThreadFunction1(void *arg)
{
    NN_UNUSED(arg);

    InspectFan();

    InspectJoyCon(s_GlvRootView);

    InspectWlan();
}


extern "C" void nnMain()
{
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    const int width = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_ABORT_UNLESS_NOT_NULL(window);
    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);
    SetupColor();

    // スレッドを生成する
    nn::Result result = nn::os::CreateThread(&g_Thread1, ThreadFunction1, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Cannot create g_Thread1.");
    // スレッドの実行を開始する
    nn::os::StartThread(&g_Thread1);


    Main(*window);
}
