﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/os/os_Tick.h>
#include "Gpio.h"

namespace ptd { namespace lib {

    Gpio::Gpio()
    {
        nn::gpio::Initialize(); // TBD
    }
    Gpio::~Gpio()
    {
        nn::gpio::Finalize(); // TBD
    }

    nn::gpio::GpioValue Gpio::GetValue(nn::gpio::Direction direction)
    {
        nn::gpio::SetDirection(&m_session, direction);
        return nn::gpio::GetValue(&m_session);
    }

    void Gpio::SetGpioTypes(nn::gpio::Direction direction,
        nn::gpio::GpioValue value,
        nn::gpio::InterruptStatus status,
        nn::gpio::InterruptMode mode) NN_NOEXCEPT
    {
        m_types.direction = direction;
        m_types.value = value;
        m_types.interruptStatus = status;
        m_types.interruptMode = mode;

        nn::gpio::SetDirection(&m_session, m_types.direction);
        nn::gpio::SetValue(&m_session, m_types.value);
        nn::gpio::SetInterruptEnable(&m_session, (m_types.interruptStatus == nn::gpio::InterruptStatus_Inactive ? false : true));
        nn::gpio::SetInterruptMode(&m_session, m_types.interruptMode);

        if (m_types.interruptStatus == nn::gpio::InterruptStatus_Active)
        {
            nn::gpio::ClearInterruptStatus(&m_session); // 一度クリアしておく
            nn::gpio::BindInterrupt(&m_event, &m_session); // 割り込みにシステムイベントを紐付け
        }
    }

    void Gpio::OpenSession(nn::gpio::GpioPadName pad) NN_NOEXCEPT
    {
        nn::gpio::OpenSession(&m_session, pad);
    }

    void Gpio::CloseSession() NN_NOEXCEPT
    {
        nn::gpio::CloseSession(&m_session);
    }

    void Gpio::ResetInterrupt()
    {
        // 割り込みステータスをクリアする
        nn::gpio::ClearInterruptStatus(&m_session);
        nn::os::ClearSystemEvent(&m_event);

        // 割り込み許可状態に戻す
        nn::gpio::SetInterruptEnable(&m_session, true);
    }

    int64_t Gpio::GetSamplingTimeCpuFan(int sampling, int timeoutSec) NN_NOEXCEPT
    {
        OpenSession(nn::gpio::GpioPadName_FanTach);
        SetGpioTypes(
            nn::gpio::Direction::Direction_Input,       // 入力方向
            nn::gpio::GpioValue_Low,                            // 無関係
            nn::gpio::InterruptStatus_Active,           // 割り込みを有効に
            nn::gpio::InterruptMode_RisingEdge);        // 立ち上がりエッジのときに割り込みを入れる設定にする

        const auto startTick = nn::os::GetSystemTick();
        for (int count = 0; count < sampling; count++)
        {
            // 一定時間割り込みが来なければ、回転してないとみなし failure
            if (!nn::os::TimedWaitSystemEvent(&m_event, nn::TimeSpan::FromSeconds(timeoutSec)))
            {
                //NN_ABORT("fan does not rotate.");
                NN_LOG("fan does not rotate.\n");
                CloseSession();
                return -1;
            }

            // 割り込みステータスをクリアして許可状態に戻す
            ResetInterrupt();
        }
        const auto endTick = nn::os::GetSystemTick();
        int64_t span = (endTick - startTick).ToTimeSpan().GetMicroSeconds();

        CloseSession();
        return span;
    }

}}
