﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include "PreinstallAppWriter_Progress.h"

namespace PreinstallAppWriter
{

int ReadOnlyProgressInfo::GetInstalledCount()
{
    return m_InstallProgress.installedCount;
}

int ReadOnlyProgressInfo::GetInstallTotalCount()
{
    return m_InstallProgress.totalCount;
}

int64_t ReadOnlyProgressInfo::GetCurrentInstallSize()
{
    return m_InstallProgress.currentSize;;
}

int64_t ReadOnlyProgressInfo::GetInstalledSize()
{
    return m_InstallProgress.installedSize;
}

int64_t ReadOnlyProgressInfo::GetInstallTotalSize()
{
    return m_InstallProgress.totalSize;
}

int ReadOnlyProgressInfo::GetVerifiedCount()
{
    return m_VerifyProgress.verifiedCount;
}

int ReadOnlyProgressInfo::GetVerifyTotalCount()
{
    return m_VerifyProgress.totalCount;
}

int64_t ReadOnlyProgressInfo::GetCurrentVerifySize()
{
    return m_VerifyProgress.currentSize;
}

int64_t ReadOnlyProgressInfo::GetVerifiedSize()
{
    return m_VerifyProgress.verifiedSize;
}

int64_t ReadOnlyProgressInfo::GetVerifyTotalSize()
{
    return m_VerifyProgress.totalSize;
}

int64_t ReadOnlyProgressInfo::GetCurrentInstallSizeInternal()
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    if(m_pTask)
    {
        return m_pTask->GetProgress().installedSize;
    }
    else
    {
        return 0;
    }
}

int64_t ReadOnlyProgressInfo::GetCurrentVerifySizeInternal()
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    if(m_pVerifyProgress)
    {
        return m_pVerifyProgress->currentSize;
    }
    else
    {
        return 0;
    }
}

nn::ncm::InstallThroughput ReadOnlyProgressInfo::GetInstallThroughput()
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    if(m_pTask)
    {
        return m_pTask->GetThroughput();
    }
    else
    {
        static nn::ncm::SubmissionPackageInstallTask s_DummyTask;
        return s_DummyTask.GetThroughput();
    }
}

ReadOnlyProgressInfo::ReadOnlyProgressInfo() : m_InstallProgress(), m_VerifyProgress(), m_Lock(true)
{
}

void ProgressInfo::SetInstallTask(nn::ncm::SubmissionPackageInstallTask* task)
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    m_pTask = task;
}

void ProgressInfo::SetInstallStatus(nn::ncm::SubmissionPackageInstallTask* task, int installedCount, int64_t installedSize, int64_t currentInstallSize)
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    SetInstallTask(task);
    SetCurrentInstallSize(currentInstallSize);
    SetInstalledCount(installedCount);
    SetInstalledSize(installedSize);
}


void ProgressInfo::SetInstalledCount(int count)
{
    m_InstallProgress.installedCount = count;
}

void ProgressInfo::SetInstallTotalCount(int count)
{
    m_InstallProgress.totalCount = count;
}

void ProgressInfo::SetCurrentInstallSize(int64_t size)
{
    m_InstallProgress.currentSize = size;
}

void ProgressInfo::SetInstalledSize(int64_t size)
{
    m_InstallProgress.installedSize = size;
}

void ProgressInfo::SetInstallTotalSize(int64_t size)
{
    m_InstallProgress.totalSize = size;
}

void ProgressInfo::SetAsyncProgress(nn::ns::AsyncVerifyApplicationProgress* progress)
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    m_pVerifyProgress = progress;
}

void ProgressInfo::SetVerifyStatus(nn::ns::AsyncVerifyApplicationProgress* progress, int verifiedCount, int64_t verifiedSize, int64_t currentVerifySize)
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    SetAsyncProgress(progress);
    SetCurrentVerifySize(currentVerifySize);
    SetVerifiedCount(verifiedCount);
    SetVerifiedSize(verifiedSize);
}

void ProgressInfo::SetVerifiedCount(int count)
{
    m_VerifyProgress.verifiedCount = count;
}

void ProgressInfo::SetVerifyTotalCount(int count)
{
    m_VerifyProgress.totalCount = count;
}

void ProgressInfo::SetCurrentVerifySize(int64_t size)
{
    m_VerifyProgress.currentSize = size;
}

void ProgressInfo::SetVerifiedSize(int64_t size)
{
    m_VerifyProgress.verifiedSize = size;
}

void ProgressInfo::SetVerifyTotalSize(int64_t size)
{
    m_VerifyProgress.totalSize = size;
}

void ProgressInfo::Func(void* self)
{
    static_cast<ProgressInfo*>(self)->UpdateProgress(nullptr);

}

void ProgressInfo::UpdateProgress(void* p)
{
    for(;;)
    {
        SetCurrentInstallSize(GetCurrentInstallSizeInternal());
        SetCurrentVerifySize(GetCurrentVerifySizeInternal());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}

}
