﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "PreinstallAppWriter_AudioOut.h"
#include <nn/util/util_Matrix.h>
#include <nn/fs.h>

namespace PreinstallAppWriter {

namespace {
    /**
    @brief FS用アロケータ
    */
    static int count = 0;
    void* Allocate(size_t size)
    {
        count++;
        NN_LOG("Alloc Count = %d\n", count);
        return std::malloc(size);
    }

    /**
    @brief FS用デアロケータ
    */
    void Deallocate(void* p, size_t size)
    {
        count--;
        NN_LOG("Dealloc Count = %d\n", count);
        NN_UNUSED(size);
        std::free(p);
    }


} // namespace

bool AudioOut::Initialize()
{
    if (m_IsInit)
    {
        return true;
    }

    InitializeFileSystem();

    m_AllocateHeap = std::malloc(BUFFER_SIZE);
    std::memset(m_AllocateHeap, 0x00, BUFFER_SIZE);
    if (m_AllocateHeap == NULL)
    {
        return false;
    }

    m_HeapHandle = nn::lmem::CreateExpHeap(m_AllocateHeap, BUFFER_SIZE, nn::lmem::CreationOption_ThreadSafe);


    m_IsInit = true;
    return true;

}

void AudioOut::StopPlaySound()
{
    if (!m_IsStartSoundThread)
    {
        return;
    }
    m_IsStartSoundThread = false;
    nn::os::WaitThread(&m_SoundThreadType);
    nn::os::DestroyThread(&m_SoundThreadType);
    nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
    nn::audio::CloseAudioOut(&m_AudioOut);

    nn::lmem::FreeToExpHeap(m_HeapHandle, m_ThreadStack);
    m_ThreadStack = nullptr;

    if (m_WavFileBuffer != nullptr)
    {
        std::free(m_WavFileBuffer);
        m_WavFileBuffer = nullptr;
    }
}

void AudioOut::Finalize()
{
    if (!m_IsInit)
    {
        return;
    }

    if (m_IsStartSoundThread)
    {
        StopPlaySound();
    }


    nn::lmem::DestroyExpHeap(m_HeapHandle);
    std::free(m_AllocateHeap);

    m_IsInit = false;

    return;

}

void AudioOut::CopyWaveFileToBuffer(void* buffer, size_t outSize)
{
    int copySize = 0;
    if (m_RemindFileSize < outSize)
    {
        copySize = m_RemindFileSize;
    }
    else
    {
        copySize = outSize;
    }

    std::memcpy(buffer, &m_WavFileBuffer[m_format.dataSize - m_RemindFileSize], copySize);

    m_RemindFileSize -= copySize;
    if (m_IsLoop && (m_RemindFileSize == 0))
    {
        m_RemindFileSize = m_format.dataSize;
    }

}

bool AudioOut::StartPlayWavFile(std::string file, bool loop)
{
    nn::audio::AudioOutParameter parameter = { 48000 };

    nn::audio::InitializeAudioOutParameter(&parameter);

    if (nn::audio::OpenDefaultAudioOut(&m_AudioOut, &m_SystemEvent, parameter).IsFailure())
    {
        return false;
    }

    if (!ReadWavFile(file.c_str()))
    {
        return false;
    }
    m_IsLoop = loop;

    m_RemindFileSize = m_format.dataSize;

    m_ThreadStack = static_cast<uint8_t*>(nn::lmem::AllocateFromExpHeap(m_HeapHandle, THREAD_STACK_SIZE, nn::os::ThreadStackAlignment));

    if (nn::os::CreateThread(&m_SoundThreadType, &StartWavSoundThread, this, m_ThreadStack, THREAD_STACK_SIZE, nn::os::DefaultThreadPriority).IsFailure())
    {
        nn::lmem::FreeToExpHeap(m_HeapHandle, m_ThreadStack);
        m_ThreadStack = nullptr;
        return false;
    }

    m_IsStartSoundThread = false;
    nn::os::StartThread(&m_SoundThreadType);

    while (!m_IsStartSoundThread)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    return m_IsSoundThreadSuccess;

}


void AudioOut::WavSoundThreadCommon()
{
    m_IsSoundThreadSuccess = false;
    // オーディオ出力の各種プロパティを取得します。
    int channelCount = nn::audio::GetAudioOutChannelCount(&m_AudioOut);
    int sampleRate = nn::audio::GetAudioOutSampleRate(&m_AudioOut);
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&m_AudioOut);

    // バッファに関するパラメータを準備します。
    const int frameRate = 20;                             // 20fps
    const int frameSampleCount = sampleRate / frameRate;  // 50msecs (in samples)
    const size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioOutBuffer::SizeGranularity);

    nn::audio::AudioOutBuffer audioOutBuffer[BUFFER_NUM];

    for (int i = 0; i < BUFFER_NUM; ++i)
    {
        m_OutBuffer[i] = nn::lmem::AllocateFromExpHeap(m_HeapHandle, bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        NN_ASSERT(m_OutBuffer[i]);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], m_OutBuffer[i], bufferSize, dataSize);
        nn::audio::AppendAudioOutBuffer(&m_AudioOut, &audioOutBuffer[i]);
    }

    // 再生を開始します。

    if (nn::audio::StartAudioOut(&m_AudioOut).IsFailure())
    {
        for (int i = 0; i < BUFFER_NUM; ++i)
        {
            nn::lmem::FreeToExpHeap(m_HeapHandle, m_OutBuffer[i]);
            m_OutBuffer[i] = nullptr;
        }
        m_IsStartSoundThread = true;
        return;
    }

    m_IsSoundThreadSuccess = true;
    m_IsStartSoundThread = true;

    while (m_IsStartSoundThread)
    {
        m_SystemEvent.Wait();

        // 再生が完了したバッファを取得します。
        nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;
        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&m_AudioOut);
        while (pAudioOutBuffer)
        {

            // 矩形波データを生成し、再度登録します。
            void* pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
            size_t outSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
            //NN_LOG("outSize = %d\n", outSize);
            NN_ASSERT(outSize == frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat));
            if (m_RemindFileSize != 0)
            {
                CopyWaveFileToBuffer(pOutBuffer, outSize);
            }
            else
            {
                std::memset(pOutBuffer, 0x00, outSize);
            }

            nn::audio::AppendAudioOutBuffer(&m_AudioOut, pAudioOutBuffer);

            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&m_AudioOut);

        }
    }

    nn::audio::StopAudioOut(&m_AudioOut);

    for (int i = 0; i < BUFFER_NUM; ++i)
    {
        memset(m_OutBuffer[i], 0, bufferSize);
        nn::lmem::FreeToExpHeap(m_HeapHandle, m_OutBuffer[i]);
        m_OutBuffer[i] = nullptr;
    }
}

void AudioOut::InitializeFileSystem()
{
    if ((nn::fs::AllocateFunction() != nullptr))
    {
        nn::fs::SetAllocator(Allocate, Deallocate);
    }
}

bool AudioOut::ReadWavFile(const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    if (!result.IsSuccess())
    {
        return false;
    }

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    if (!result.IsSuccess())
    {
        return false;
    }

    // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
    const std::size_t WavHeaderDataSize = 1024;

    std::unique_ptr<void, decltype(&std::free)> pChunk(std::malloc(1024), std::free);
    result = nn::fs::ReadFile(handle, 0, pChunk.get(), WavHeaderDataSize);
    if (!result.IsSuccess())
    {
        return false;
    }

    nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(&m_format, pChunk.get(), WavHeaderDataSize);
    if (wavResult != nns::audio::WavResult_Success)
    {
        return false;
    }
    NN_LOG("channel = %d\n", m_format.channelCount);
    NN_LOG("bitperSample = %d\n", m_format.bitsPerSample);
    NN_LOG("SampleRate = %d\n", m_format.sampleRate);

    m_WavFileBuffer = static_cast<uint8_t*>(std::malloc(static_cast<std::size_t>(m_format.dataSize)));

    result = nn::fs::ReadFile(handle, static_cast<std::size_t>(m_format.dataOffset), m_WavFileBuffer, static_cast<std::size_t>(m_format.dataSize));
    if (!result.IsSuccess())
    {
        return false;
    }
    nn::fs::CloseFile(handle);

    return true;
}

}
