﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>

#include "../graphics/NoftWriter_Renderer.h"

namespace noftwriter { namespace ui {

/**
 * @brief   画面に表示するオブジェクトのベースクラスです。
 */
class DisplayObject
{
public:
    DisplayObject() NN_NOEXCEPT :
        m_IsVisible(false),
        m_Opacity(255),
        m_X(0),
        m_Y(0),
        m_pRenderer(nullptr)
    {}

    virtual ~DisplayObject() NN_NOEXCEPT {}

    /**
     * @brief   表示可能かどうかを取得します。
     */
    bool IsVisible() const NN_NOEXCEPT
    {
        return m_IsVisible;
    }

    /**
     * @brief   表示します。
     */
    virtual void Show() NN_NOEXCEPT
    {
        m_IsVisible = true;
    }

    /**
     * @brief   非表示にします。
     */
    virtual void Hide() NN_NOEXCEPT
    {
        m_IsVisible = false;
    }

    /**
     * @brief   不透明度を取得します。
     */
    uint8_t GetOpacity() const NN_NOEXCEPT
    {
        return m_Opacity;
    }

    /**
     * @brief   不透明度を設定します。
     */
    void SetOpacity(int opacity) NN_NOEXCEPT
    {
        m_Opacity = std::min<uint8_t>(std::max(opacity, 0), 255);
    }

    /**
     * @brief   表示位置を指定します。
     */
    void SetPosition(float x, float y) NN_NOEXCEPT
    {
        m_X = x;
        m_Y = y;
    }

    /**
     * @brief   表示位置の X 座標を取得します。
     */
    float GetX() const NN_NOEXCEPT
    {
        return m_X;
    }

    /**
     * @brief   表示位置の X 座標を取得します。
     */
    float GetY() const NN_NOEXCEPT
    {
        return m_Y;
    }

    /**
     * @brief   表示位置を取得します。
     */
    void GetPosition(float* pOutX, float* pOutY) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutX);
        NN_ASSERT_NOT_NULL(pOutY);

        *pOutX = m_X;
        *pOutY = m_Y;
    }

    /**
     * @brief   描画に使用するレンダラーを指定します。
     */
    virtual void SetRenderer(graphics::Renderer* pRenderer) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pRenderer);

        m_pRenderer = pRenderer;
    }

    /**
     * @brief   オブジェクトの状態を更新します。
     *          原則 1 フレームにつき 1 回実行します。
     */
    virtual void Update() NN_NOEXCEPT {}

    /**
     * @brief   オブジェクトを描画します。
     *          後に描画したものほど手前に表示されます。
     */
    virtual void Render() NN_NOEXCEPT = 0;

protected:
    /**
     * @brief   描画に使用するレンダラーを取得します。
     */
    graphics::Renderer* GetRenderer() const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pRenderer);

        return m_pRenderer;
    }

private:
    bool                m_IsVisible;    //!< 表示フラグ
    uint8_t             m_Opacity;      //!< 不透明度
    float               m_X;            //!< X 座標
    float               m_Y;            //!< Y 座標
    graphics::Renderer* m_pRenderer;    //!< レンダラー
};

}}  // noftwriter::ui
