﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_Result.h>
#include <nn/nfp.h>
#include <nn/nfp/nfp_PrivateResult.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ResEndian.h>
#include <nn/util/util_ScopeExit.h>

#include "graphics/NoftWriter_Renderer.h"
#include "npad/NoftWriter_NpadController.h"
#include "NoftWriter_Util.h"

namespace {

const struct
{
    nn::hid::NpadIdType     npadId;
    const char*             name;
} NpadNameList[] =
{
    { nn::hid::NpadId::No1,      "No.1" },
    { nn::hid::NpadId::No2,      "No.2" },
    { nn::hid::NpadId::No3,      "No.3" },
    { nn::hid::NpadId::No4,      "No.4" },
    { nn::hid::NpadId::No5,      "No.5" },
    { nn::hid::NpadId::No6,      "No.6" },
    { nn::hid::NpadId::No7,      "No.7" },
    { nn::hid::NpadId::No8,      "No.8" },
    { nn::hid::NpadId::Handheld, "Handheld" },
};

template <typename T>
void SetValueToTag(void* pDst, T value) NN_NOEXCEPT
{
    T tagByteOrderValue;
    nn::util::StoreBigEndian<T>(&tagByteOrderValue, value);
    std::memcpy(pDst, &tagByteOrderValue, sizeof(T));
}

}  // anonymous

namespace noftwriter {

const float DisplayLineOffset   = 30;
const float DisplayColumnOffset = 60;

const char* GetNpadIdName(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    for (const auto& nameset : NpadNameList)
    {
        if (nameset.npadId == npadId)
        {
            return nameset.name;
        }
    }

    return "???";
}

const char* GetNpadStyleName(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    auto style = nn::hid::GetNpadStyleSet(npadId);

    if (style.Test<nn::hid::NpadStyleFullKey>())
    {
        return "FullKeyController";
    }
    else if (style.Test<nn::hid::NpadStyleJoyDual>())
    {
        return "JoyCon L/R";
    }
    else if (style.Test<nn::hid::NpadStyleJoyLeft>())
    {
        return "JoyCon Left";
    }
    else if (style.Test<nn::hid::NpadStyleJoyRight>())
    {
        return "JoyCon Right";
    }
    else if (style.Test<nn::hid::NpadStyleHandheld>())
    {
        return "Handheld";
    }

    return nullptr;
}

void PrintHeader(graphics::Renderer* pRenderer, const char* title) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pRenderer);
    NN_ASSERT_NOT_NULL(title);

    pRenderer->SetTextScale(2.0f, 2.0f);
    pRenderer->SetTextColor(graphics::Colors::White);
    pRenderer->DrawText(20, 20, "%s", title);
    pRenderer->SetTextScale(1.2f, 1.2f);

    pRenderer->DrawHorizontalLine(0, 80, 1280, graphics::Colors::Silver, 2.0f);
}

bool IsSameTag(const nn::nfp::TagId& tagId1, const nn::nfp::TagId& tagId2) NN_NOEXCEPT
{
    if (tagId1.length != tagId2.length)
    {
        return false;
    }

    return std::memcmp(tagId1.uid, tagId2.uid, tagId1.length) == 0;
};

const char* GetResultText(nn::Result result) NN_NOEXCEPT
{
    // ひとまず Mount 関連のみ
    if (nn::nfp::ResultNfcDeviceNotFound::Includes(result))
    {
        return "ResultNfcDeviceNotFound";
    }
    else if (nn::nfp::ResultNeedRetry::Includes(result))
    {
        return "ResultNeedRetry";
    }
    else if (nn::nfp::ResultNeedRestart::Includes(result))
    {
        return "ResultNeedRestart";
    }
    else if (nn::nfp::ResultNfcDisabled::Includes(result))
    {
        return "ResultNfcDisabled";
    }
    else if (nn::nfp::ResultNotSupported::Includes(result))
    {
        return "ResultNotSupported";
    }
    else
    {
        return "-";
    }
}

}  // noftwriter
