﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/nfp/nfp_DebugTypes.h>

#include "NfpDebugTool_BaseTagInfoWindow.h"
#include "../NfpDebugTool_Util.h"

namespace nfpdebug { namespace ui {

float BaseTagInfoWindow::DrawErrorText(float x, float y, const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    float dy = y;

    renderer.SetTextScale(1.6f, 1.6f);
    renderer.SetTextColor(GetDisplayColor(graphics::Colors::Gray));
    renderer.DrawText(x, dy, "%s", text);
    dy += renderer.GetTextLineHeight();

    renderer.SetTextScale(prevScale.x, prevScale.y);

    return dy - y;
}

float BaseTagInfoWindow::DrawInfoTitle(float x, float y, const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    float dy = y;

    renderer.SetTextScale(1.6f, 1.6f);
    renderer.SetTextColor(GetDisplayColor(graphics::Colors::DodgerBlue));
    renderer.DrawText(x, dy, "%s", text);
    dy += renderer.GetTextLineHeight();

    renderer.SetTextScale(prevScale.x, prevScale.y);

    return dy - y;
}

float BaseTagInfoWindow::DrawBinary(float x, float y, const char* pData, size_t dataBytes) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pData);

    auto& renderer = *GetRenderer();
    float dy = y;

    char outputText[64];
    int textLength = 0;
    for (size_t i = 0; i < dataBytes; i++)
    {
        textLength += nn::util::SNPrintf(
            outputText + textLength,
            sizeof(outputText) - textLength,
            "%02X ",
            static_cast<nn::Bit8>(pData[i]));
    }
    renderer.DrawText(x, dy, "%s", outputText);
    dy += renderer.GetTextLineHeight();

    return dy - y;
}

float BaseTagInfoWindow::DrawBinaryFormatted(float x, float y, const char* pData, size_t dataBytes) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pData);

    auto& renderer = *GetRenderer();
    float dy = y;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));
    renderer.SetFixedMode(true, FixedModeTextWidth);

    // ヘッダ部分
    const int lineBytes = 16;
    int remainBytes = dataBytes;
    int rows = (remainBytes + lineBytes - 1) / lineBytes;
    renderer.DrawText(x, dy, "Addr  00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F");
    renderer.DrawHorizontalLine(
        x - 4,
        dy + renderer.GetTextLineHeight() * 1.5f - 1.0f,
        752,
        GetDisplayColor(graphics::Colors::White),
        3.0f);
    renderer.DrawVerticalLine(
        x + renderer.CalculateTextWidth("Addr "),
        dy,
        (rows + 2) * renderer.GetTextLineHeight(),
        GetDisplayColor(graphics::Colors::White));
    dy += renderer.GetTextLineHeight() * 2;

    // 実データ部分
    for (int i = 0; i < rows; i++)
    {
        char header[16];
        nn::util::SNPrintf(header, sizeof(header), "  %02X  ", i * lineBytes);
        renderer.DrawText(x, dy, "%s", header);

        float offset = renderer.CalculateTextWidth(header);
        auto dataBytes = std::min<size_t>(remainBytes, lineBytes);
        dy += DrawBinary(x + offset, dy, &pData[i * lineBytes], dataBytes);
        remainBytes -= lineBytes;
    }

    renderer.SetFixedMode(false);

    return dy - y;
}

float BaseTagInfoWindow::DrawTagInfo(float x, float y, float offsetX, const nn::nfp::TagInfo& tagInfo) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    float dx = x;
    float dy = y;

    dy += DrawInfoTitle(dx, dy, "TagInfo");
    dx += 8;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    renderer.DrawText(dx, dy, "Protocol");
    renderer.DrawText(dx + offsetX, dy, "%d (%s)",
        tagInfo.protocol,
        GetTagProtocolName(static_cast<nn::nfp::NfcProtocol>(tagInfo.protocol)));
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Type");
    renderer.DrawText(dx + offsetX, dy, "%d (%s)",
        tagInfo.type,
        GetTagTypeName(static_cast<nn::nfp::TagType>(tagInfo.type)));
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "UID");
    dy += DrawBinary(dx + offsetX, dy, tagInfo.tagId.uid, tagInfo.tagId.length);

    return dy - y;
}

float BaseTagInfoWindow::DrawModelInfo(float x, float y, float offsetX, const nn::nfp::ModelInfo& modelInfo) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    float dx = x;
    float dy = y;

    dy += DrawInfoTitle(dx, dy, "Model Info");
    dx += 8;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    renderer.DrawText(dx, dy, "Character ID");
    dy += DrawBinary(dx + offsetX, dy, modelInfo.characterId, nn::nfp::CharacterIdSize);

    renderer.DrawText(dx, dy, "Series ID");
    renderer.DrawText(dx + offsetX, dy, "%d", modelInfo.seriesId);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Numbering ID");
    renderer.DrawText(dx + offsetX, dy, "%d", modelInfo.numberingId);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "NFP Type");
    renderer.DrawText(dx + offsetX, dy, "%d", modelInfo.nfpType);
    dy += renderer.GetTextLineHeight();

    return dy - y;
}

float BaseTagInfoWindow::DrawCommonInfo(float x, float y, float offsetX, const nn::nfp::CommonInfo& commonInfo) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    float dx = x;
    float dy = y;

    dy += DrawInfoTitle(dx, dy, "Common Info");
    dx += 8;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    const auto& date = commonInfo.lastWriteDate;
    renderer.DrawText(dx, dy, "Last write date");
    renderer.DrawText(dx + offsetX, dy, "%04d/%02d/%02d", date.year, date.month, date.day);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Write counter");
    renderer.DrawText(dx + offsetX, dy, "%d", commonInfo.writeCounter);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "NFP Version");
    renderer.DrawText(dx + offsetX, dy, "%d", commonInfo.nfpVersion);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "App area size");
    renderer.DrawText(dx + offsetX, dy, "%u", commonInfo.applicationAreaSize);
    dy += renderer.GetTextLineHeight();

    return dy - y;
}

float BaseTagInfoWindow::DrawSystemInfo(float x, float y, float offsetX, const nn::nfp::SystemInfo& systemInfo) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    float dx = x;
    float dy = y;

    dy += DrawInfoTitle(dx, dy, "System Info");
    dx += 8;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    renderer.DrawText(dx, dy, "Write status");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy, "0x%02X", systemInfo.writeStatus);
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "System write counter");
    renderer.DrawText(dx + offsetX, dy, "%d", systemInfo.systemWriteCounter);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Hardware ID");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy, "0x%08X", systemInfo.hardwareId);
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();

    return dy - y;
}

float BaseTagInfoWindow::DrawAdminInfo(float x, float y, float offsetX, const nn::nfp::AdminInfoDebug& adminInfo) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    float dx = x;
    float dy = y;

    dy += DrawInfoTitle(dx, dy, "Admin Info");
    dx += 8;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

#ifdef NFPDEBUG_SYSTEM
    renderer.DrawText(dx, dy, "Application ID");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy,
        "0x%08X%08X",
        static_cast<uint32_t>((adminInfo.applicationId.value >> 32) & 0xFFFFFFFF),
        static_cast<uint32_t>(adminInfo.applicationId.value & 0xFFFFFFFF));
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Ext application ID");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy, "0x%02X", adminInfo.applicationIdExt);
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();
#endif  // ifdef NFPDEBUG_SYSTEM

    renderer.DrawText(dx, dy, "Access ID");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy, "0x%08X", adminInfo.accessId);
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();

#ifdef NFPDEBUG_SYSTEM
    renderer.DrawText(dx, dy, "Move counter");
    renderer.DrawText(dx + offsetX, dy, "%d", adminInfo.moveCounter);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Register info");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy, "0x%02X", adminInfo.registerInfo);
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Format version");
    renderer.DrawText(dx + offsetX, dy, "%d", adminInfo.formatVersion);
    dy += renderer.GetTextLineHeight();
#endif  // ifdef NFPDEBUG_SYSTEM

    return dy - y;
}

float BaseTagInfoWindow::DrawRegisterInfo(float x, float y) NN_NOEXCEPT
{
    float dy = y;

    dy += DrawInfoTitle(x, dy, "Register Info");
    DrawErrorText(x + 8, dy, "Not registered");
    dy += 32;

    return dy - y;
}

float BaseTagInfoWindow::DrawRegisterInfo(
    float x,
    float y,
    float offsetX,
    const nn::nfp::RegisterInfo& registerInfo) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    float dx = x;
    float dy = y;

    dy += DrawInfoTitle(dx, dy, "Register Info");
    dx += 8;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    const auto& date = registerInfo.registerDate;
    renderer.DrawText(dx, dy, "Register date");
    renderer.DrawText(dx + offsetX, dy, "%04d/%02d/%02d", date.year, date.month, date.day);
    dy += renderer.GetTextLineHeight();

    char name[64] = {};
    nn::util::ConvertStringUtf16NativeToUtf8(
        name,
        sizeof(name),
        reinterpret_cast<const uint16_t*>(registerInfo.nickname));
    name[sizeof(name) - 1] = '\0';
    renderer.DrawText(dx, dy, "Nickname");
    renderer.DrawText(dx + offsetX, dy, "%s", name);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Font region");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy, "0x%02X (%s)",
        registerInfo.fontRegion,
        GetFontRegionName(static_cast<nn::nfp::FontRegion>(registerInfo.fontRegion)));
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();

#ifdef NFPDEBUG_SYSTEM
    renderer.DrawText(dx, dy, "Mii data");
    dy += renderer.GetTextLineHeight();
    dy += DrawBinaryFormatted(
        dx + 8,
        dy,
        reinterpret_cast<const char*>(&registerInfo.miiData),
        sizeof(registerInfo.miiData));
#endif  // ifdef NFPDEBUG_SYSTEM

    return dy - y;
}

float BaseTagInfoWindow::DrawRegisterInfo(
    float x,
    float y,
    float offsetX,
    const nn::nfp::RegisterInfoDebug& registerInfo) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    float dx = x;
    float dy = y;

    dy += DrawInfoTitle(dx, dy, "Register Info");
    dx += 8;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    const auto& date = registerInfo.registerDate;
    renderer.DrawText(dx, dy, "Register date");
    renderer.DrawText(dx + offsetX, dy, "%04d/%02d/%02d", date.year, date.month, date.day);
    dy += renderer.GetTextLineHeight();

    char name[64] = {};
    nn::util::ConvertStringUtf16NativeToUtf8(
        name,
        sizeof(name),
        reinterpret_cast<const uint16_t*>(registerInfo.nickname));
    name[sizeof(name) - 1] = '\0';
    renderer.DrawText(dx, dy, "Nickname");
    renderer.DrawText(dx + offsetX, dy, "%s", name);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Font region");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy, "0x%02X (%s)",
        registerInfo.fontRegion,
        GetFontRegionName(static_cast<nn::nfp::FontRegion>(registerInfo.fontRegion)));
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();

#ifdef NFPDEBUG_SYSTEM
    renderer.DrawText(dx, dy, "Mii data core");
    dy += renderer.GetTextLineHeight();
    dy += DrawBinaryFormatted(
        dx + 8,
        dy,
        reinterpret_cast<const char*>(&registerInfo.miiDataCore),
        sizeof(registerInfo.miiDataCore)) + 8;

    renderer.DrawText(dx, dy, "Mii data extention");
    dy += DrawBinary(
        dx + offsetX,
        dy,
        reinterpret_cast<const char*>(&registerInfo.miiDataExtention),
        sizeof(registerInfo.miiDataExtention));

    renderer.DrawText(dx, dy, "Mii ext version");
    renderer.DrawText(dx + offsetX, dy, "%d", registerInfo.miiExtentionVersion);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Extention CRC");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy, "0x%08X", registerInfo.extentionCrc);
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();
#endif  // ifdef NFPDEBUG_SYSTEM

    return dy - y;
}

float BaseTagInfoWindow::DrawApplicationArea(float x, float y) NN_NOEXCEPT
{
    float dy = y;

    dy += DrawInfoTitle(x, dy, "Application area");
    DrawErrorText(x + 8, dy, "Not created");
    dy += 32;

    return dy - y;
}

float BaseTagInfoWindow::DrawApplicationArea(
    float x,
    float y,
    const char* pApplicationArea,
    size_t applicationAreaSize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pApplicationArea);

    float dy = y;

    dy += DrawInfoTitle(x, dy, "Application area");
    dy += DrawBinaryFormatted(x + 8, dy, pApplicationArea, applicationAreaSize);

    return dy - y;
}

}}  // nfpdebug::ui
