﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os/os_MultipleWaitUtility.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/hid.h>
#include <nn/nfp.h>
#include <nn/nfp/nfp_DebugApi.h>

#include "npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_NfpProcessor.h"

//#define VERBOSE

#define NFPDEBUG_LOG(...)   NN_LOG("[NfpProcessor] " __VA_ARGS__)

#ifdef VERBOSE
#define NFPDEBUG_LOG_VERBOSE(...)   NFPDEBUG_LOG(__VA_ARGS__)
#else
#define NFPDEBUG_LOG_VERBOSE(...)   static_cast<void>(0)
#endif  // ifdef VERBOSE

namespace
{

// 検出可能なデバイス数の上限
const int DetectableHandleCount = nn::nfp::DeviceCountMax;

/**
 * @brief   cabinet の起動パラメータを構築
 */
void SetupStartParam(
    nn::nfp::AmiiboSettingsStartParam* pOutParam,
    const nn::nfp::DeviceHandle& handle,
    bool isNicknameRequired) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutParam);

    pOutParam->deviceHandle = handle;
    pOutParam->optionFlags  = isNicknameRequired
        ? nn::nfp::AmiiboSettingsOptionFlags_RequiredNickname
        : nn::nfp::AmiiboSettingsOptionFlags_Default;
}

}

namespace nfpdebug {

void NfpProcessor::Setup(char* pThreadStack, size_t stackSize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pThreadStack);

    if (m_IsRunning)
    {
        return;
    }

    nn::os::InitializeEvent(&m_ResumeEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_SuspendEvent, false, nn::os::EventClearMode_AutoClear);

    m_pThreadStack = pThreadStack;
    auto result = nn::os::CreateThread(
        &m_Thread,
        ThreadFunction,
        this,
        pThreadStack,
        stackSize,
        nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    m_IsRunning = true;
    nn::os::StartThread(&m_Thread);
}

void NfpProcessor::Shutdown() NN_NOEXCEPT
{
    if (m_pThreadStack == nullptr)
    {
        return;
    }

    m_IsRunning = false;
    nn::os::DestroyThread(&m_Thread);
    nn::os::FinalizeEvent(&m_ResumeEvent);
    nn::os::FinalizeEvent(&m_SuspendEvent);

    m_pThreadStack = nullptr;
}

bool NfpProcessor::Activate(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    Deactivate();

    nn::nfp::DeviceHandle handles[DetectableHandleCount];
    int handleCount;
    auto result = nn::nfp::ListDevices(handles, &handleCount, DetectableHandleCount);
    if (result.IsFailure())
    {
        NFPDEBUG_LOG("%s: Failed to detect device\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }
    else if (handleCount <= 0)
    {
        NFPDEBUG_LOG("%s: No device detected\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    // 渡された NpadId に対応するハンドルを探す
    for (int i = 0; i < handleCount; i++)
    {
        nn::hid::NpadIdType id;
        if (nn::nfp::GetNpadId(&id, handles[i]).IsFailure() || id != npadId)
        {
            continue;
        }

        m_DeviceHandle = handles[i];
        nn::nfp::AttachActivateEvent(&m_DetectEvent, m_DeviceHandle);
        nn::nfp::AttachDeactivateEvent(&m_LostEvent, m_DeviceHandle);
        m_IsActivated = true;
        NFPDEBUG_LOG_VERBOSE("Activated\n");
        return true;
    }

    return false;
}

void NfpProcessor::Deactivate() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!m_IsActivated)
    {
        return;
    }

    //m_NfpInfo.Clear();
    nn::nfp::StopDetection(m_DeviceHandle);

    nn::os::DestroySystemEvent(&m_DetectEvent);
    nn::os::DestroySystemEvent(&m_LostEvent);

    m_IsActivated = false;

    NFPDEBUG_LOG_VERBOSE("Deactivated\n");
}

void NfpProcessor::Suspend() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    nn::os::ClearEvent(&m_ResumeEvent);
    m_NextCommand = Command::Suspend;

    nn::os::WaitEvent(&m_SuspendEvent);
}

void NfpProcessor::Resume() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    nn::os::SignalEvent(&m_ResumeEvent);
}

void NfpProcessor::NotifyLostFocus() NN_NOEXCEPT
{
    if (!m_IsActivated)
    {
        return;
    }

    if (m_CancelCallback != nullptr)
    {
        NFPDEBUG_LOG_VERBOSE("Lost focus\n");
        m_CancelCallback(m_pCancelCallbackArgument);
    }
}

void NfpProcessor::ThreadFunction(void* pArg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pArg);

    reinterpret_cast<NfpProcessor*>(pArg)->ThreadFunctionImpl();
}

void NfpProcessor::ThreadFunctionImpl() NN_NOEXCEPT
{
    while (m_IsRunning)
    {
        CheckEvent();
        if (!HandleCommand())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
        }
    }
}

void NfpProcessor::CheckEvent() NN_NOEXCEPT
{
    auto WaitEvent = [this]() -> int
    {
        if (m_IsActivated)
        {
            return nn::os::TryWaitAny(&m_DetectEvent, &m_LostEvent);
        }

        return -1;
    };

    switch (WaitEvent())
    {
    case 0:  // Detect
        NFPDEBUG_LOG("Tag detected\n");
        nn::os::ClearSystemEvent(&m_DetectEvent);
        ReadAmiibo();
        if (m_DetectCallback != nullptr)
        {
            m_DetectCallback(m_pDetectCallbackArgument);
        }

        break;
    case 1:  // Lost
        NFPDEBUG_LOG("Tag lost\n");
        nn::os::ClearSystemEvent(&m_LostEvent);
        if (m_LostCallback != nullptr)
        {
            m_LostCallback(m_pLostCallbackArgument);
        }
        break;
    default:
        break;
    }
}

bool NfpProcessor::HandleCommand() NN_NOEXCEPT
{
    auto command = m_NextCommand;
    m_NextCommand = Command::None;

    nn::Result result;
    switch (command)
    {
    case Command::None:
        // 何もしない
        return false;
    case Command::Flush:
        {
            NFPDEBUG_LOG_VERBOSE("Command: Flush\n");
            result = nn::nfp::Flush(m_DeviceHandle);
        }
        break;
    case Command::FlushDebug:
        {
            NFPDEBUG_LOG_VERBOSE("Command: FlushDebug\n");
            nn::nfp::SetAll(m_DeviceHandle, m_NfpInfo.allData);

            result = nn::nfp::FlushDebug(m_DeviceHandle);
        }
        break;
    case Command::Format:
        {
            NFPDEBUG_LOG_VERBOSE("Command: Format\n");
            nn::nfp::Unmount(m_DeviceHandle);
            result = nn::nfp::Format(m_DeviceHandle);
        }
        break;
    case Command::Restore:
        {
            NFPDEBUG_LOG_VERBOSE("Command: Restore\n");
            nn::nfp::Unmount(m_DeviceHandle);
            result = nn::nfp::Restore(m_DeviceHandle);
        }
        break;
    case Command::DeleteApplicationArea:
        {
            NFPDEBUG_LOG_VERBOSE("Command: DeleteApplicationArea\n");
            result = nn::nfp::DeleteApplicationArea(m_DeviceHandle);
        }
        break;
    case Command::DeleteRegisterInfo:
        {
            NFPDEBUG_LOG_VERBOSE("Command: DeleteRegisterInfo\n");
            result = nn::nfp::DeleteRegisterInfo(m_DeviceHandle);
        }
        break;
    case Command::BreakActivation:
        {
            NFPDEBUG_LOG_VERBOSE("Command: BreakActivation\n");
            result = nn::nfp::BreakTag(m_DeviceHandle, nn::nfp::BreakType_Activation);
        }
        break;
    case Command::BreakHmac:
        {
            NFPDEBUG_LOG_VERBOSE("Command: BreakHmac\n");
            result = nn::nfp::BreakTag(m_DeviceHandle, nn::nfp::BreakType_Hmac);
        }
        break;
    case Command::Suspend:
        {
            nn::os::SignalEvent(&m_SuspendEvent);
            nn::os::WaitEvent(&m_ResumeEvent);
        }
        return true;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    HandleFlushResult(result);
    return true;
}

void NfpProcessor::ReadAmiibo() NN_NOEXCEPT
{
    m_NfpInfo.Clear();

    if (nn::nfp::GetTagInfo(&m_NfpInfo.tagInfo, m_DeviceHandle).IsFailure())
    {
        return;
    }

    m_NfpInfo.isValidTag = true;

#ifdef VERBOSE
    {
        auto& info = m_NfpInfo.tagInfo;
        NFPDEBUG_LOG_VERBOSE("TagInfo:\n");
        NN_LOG("  Protocol : %d\n", info.protocol);
        NN_LOG("  TagType  : %d\n", info.type);
        NN_LOG("  UID      : ", info.type);
        for (int i = 0; i < info.tagId.length; i++)
        {
            NN_LOG("%02X ", info.tagId.uid[i]);
        }
        NN_LOG("\n");
    }
#endif  // ifdef VERBOSE

    // ROM 領域を読む
    if (nn::nfp::Mount(m_DeviceHandle, nn::nfp::ModelType_Amiibo, nn::nfp::MountTarget_Rom).IsFailure())
    {
        NFPDEBUG_LOG("%s: Not amiibo\n", NN_CURRENT_FUNCTION_NAME);
        return;
    }

    m_NfpInfo.isAmiibo = true;
    nn::nfp::GetModelInfo(&m_NfpInfo.modelInfo, m_DeviceHandle);
    nn::nfp::Unmount(m_DeviceHandle);

    // RAM 領域を読む
    auto result = nn::nfp::Mount(m_DeviceHandle, nn::nfp::ModelType_Amiibo);
    if (result.IsFailure())
    {
        NFPDEBUG_LOG("%s: Cannot mount ram (%s)\n",
            NN_CURRENT_FUNCTION_NAME,
            nfpdebug::GetResultText(result));
        return;
    }

    m_NfpInfo.hasValidRam = true;
    nn::nfp::GetAll(&m_NfpInfo.allData, m_DeviceHandle);

    // 登録済みか判定
    {
        m_NfpInfo.hasRegisterInfo = nn::nfp::GetRegisterInfo(
            &m_NfpInfo.publicRegisterInfo,
            m_DeviceHandle).IsSuccess();
    }

    //nn::nfp::Unmount(m_DeviceHandle);
}

nn::Result NfpProcessor::HandleCommonResult(nn::Result result) NN_NOEXCEPT
{
    auto HandleFailure = [this](NfpResult nfpResult)
    {
        Deactivate();
        if (m_FinishCallback != nullptr)
        {
            m_FinishCallback(nfpResult, m_pFinishCallbackArgument);
        }
    };

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(nn::nfp::ResultNfcDeviceNotFound)
        {
            NFPDEBUG_LOG("Error: ResultNfcDeviceNotFound\n");
            HandleFailure(NfpResult::DeviceLost);
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH(nn::nfp::ResultNfcDisabled)
        {
            NFPDEBUG_LOG("Error: ResultNfcDisabled\n");
            HandleFailure(NfpResult::DeviceLost);
            NN_RESULT_RETHROW;
        }
#if 0  // Result が変わるかもしれないので一時無効化
        NN_RESULT_CATCH(nn::nfp::ResultWaitForeground)
        {
            NFPDEBUG_LOG("Error: ResultWaitForeground\n");
            HandleFailure(NfpResult::NotForeground);
            NN_RESULT_RETHROW;
        }
#endif
        NN_RESULT_CATCH_ALL
        {
            NFPDEBUG_LOG("Error: 0x%08X\n", result.GetInnerValueForDebug());
            HandleFailure(NfpResult::Failed);
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

nn::Result NfpProcessor::HandleFlushResult(nn::Result result) NN_NOEXCEPT
{
    auto HandleFailure = [this](NfpResult nfpResult)
    {
        Deactivate();
        if (m_FinishCallback != nullptr)
        {
            m_FinishCallback(nfpResult, m_pFinishCallbackArgument);
        }
    };

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(nn::nfp::ResultNeedRestart)
        {
            NFPDEBUG_LOG("Error: ResultNeedRestart\n");
            HandleFailure(NfpResult::Failed);
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH(nn::nfp::ResultNotBroken)
        {
            NFPDEBUG_LOG("Error: ResultNotBroken\n");
            HandleFailure(NfpResult::NotBroken);
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            // 他のエラーは共通処理に回す
            NN_RESULT_THROW(HandleCommonResult(result));
        }
    NN_RESULT_END_TRY

    if (m_FinishCallback != nullptr)
    {
        m_FinishCallback(NfpResult::Success, m_pFinishCallbackArgument);
    }

    NN_RESULT_SUCCESS;
}

void NfpProcessor::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
#if 0
    if (!m_IsActivated)
    {
        return;
    }

    nn::hid::NpadIdType npadIdForNfp;
    auto result = nn::nfp::GetNpadId(&npadIdForNfp, m_DeviceHandle);
    if (result.IsFailure() || controller.GetNpadId() != npadIdForNfp)
    {
        return;
    }

    auto state = nn::nfp::GetDeviceState(m_DeviceHandle);

    if (state == nn::nfp::DeviceState_Search &&
        controller.IsTriggered(nn::hid::NpadButton::B::Mask))
    {
        StopDetection();
    }
#endif
}

nn::Result NfpProcessor::StartDetection() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");

    m_NfpInfo.Clear();
    auto result = nn::nfp::StartDetection(m_DeviceHandle);
    if (result.IsSuccess())
    {
        NFPDEBUG_LOG_VERBOSE("Detection started\n");
    }

    return HandleCommonResult(result);
}

nn::Result NfpProcessor::StopDetection() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");

    auto result = nn::nfp::StopDetection(m_DeviceHandle);

    if (m_CancelCallback != nullptr)
    {
        NFPDEBUG_LOG_VERBOSE("Cancelled\n");
        m_CancelCallback(m_pCancelCallbackArgument);
    }

    return HandleCommonResult(result);
}

nn::Result NfpProcessor::Flush() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    m_NextCommand = Command::Flush;
    NN_RESULT_SUCCESS;
}

nn::Result NfpProcessor::FlushDebug() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    {
        auto result = nn::nfp::SetAll(m_DeviceHandle, m_NfpInfo.allData);
        if (result.IsFailure())
        {
            return HandleCommonResult(result);
        }
    }

    m_NextCommand = Command::FlushDebug;
    NN_RESULT_SUCCESS;
}

nn::Result NfpProcessor::Format() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    m_NextCommand = Command::Format;
    NN_RESULT_SUCCESS;
}

nn::Result NfpProcessor::Restore() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    m_NextCommand = Command::Restore;
    NN_RESULT_SUCCESS;
}

nn::Result NfpProcessor::DeleteInfo(DeleteInfoType infoType) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    switch (infoType)
    {
    case DeleteInfoType_ApplicationArea:
        m_NextCommand = Command::DeleteApplicationArea;
        break;
    case DeleteInfoType_RegisterInfo:
        m_NextCommand = Command::DeleteRegisterInfo;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

nn::Result NfpProcessor::BreakTag(nn::nfp::BreakType breakType) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    switch (breakType)
    {
    case nn::nfp::BreakType_Activation:
        m_NextCommand = Command::BreakActivation;
        break;
    case nn::nfp::BreakType_Hmac:
        m_NextCommand = Command::BreakHmac;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

nn::Result NfpProcessor::CallNicknameAndOwner(
    nn::nfp::DeviceHandle* pOutDeviceHandle,
    nn::nfp::TagInfo* pOutTagInfo,
    bool* pOutIsRegistered,
    nn::nfp::RegisterInfo* pOutRegisterInfo,
    bool isNicknameRequired) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutDeviceHandle);
    NN_ASSERT_NOT_NULL(pOutTagInfo);
    NN_ASSERT_NOT_NULL(pOutIsRegistered);
    NN_ASSERT_NOT_NULL(pOutRegisterInfo);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // タグ未検出でも呼べる
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    nn::nfp::AmiiboSettingsStartParam startParam = {};
    SetupStartParam(&startParam, m_DeviceHandle, isNicknameRequired);

    // 本体設定から呼ばれる API (TagInfo, RegisterInfo 指定なし)
    Suspend();
    auto result = nn::nfp::StartNicknameAndOwnerSettings(
        pOutTagInfo,
        pOutDeviceHandle,
        pOutIsRegistered,
        pOutRegisterInfo,
        startParam);
    Resume();

    return result;
}

nn::Result NfpProcessor::CallNicknameAndOwner(
    nn::nfp::DeviceHandle* pOutDeviceHandle,
    bool* pOutIsRegistered,
    nn::nfp::RegisterInfo* pOutRegisterInfo,
    bool isNicknameRequired,
    const nn::nfp::TagInfo& tagInfo) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutDeviceHandle);
    NN_ASSERT_NOT_NULL(pOutIsRegistered);
    NN_ASSERT_NOT_NULL(pOutRegisterInfo);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    nn::nfp::AmiiboSettingsStartParam startParam = {};
    SetupStartParam(&startParam, m_DeviceHandle, isNicknameRequired);

    // 通常のアプリから呼ばれる API (RegisterInfo 指定なし)
    Suspend();
    auto result = nn::nfp::StartNicknameAndOwnerSettings(
        pOutDeviceHandle,
        pOutIsRegistered,
        pOutRegisterInfo,
        startParam,
        tagInfo);
    Resume();

    return result;
}

nn::Result NfpProcessor::CallNicknameAndOwner(
    nn::nfp::DeviceHandle* pOutDeviceHandle,
    bool* pOutIsRegistered,
    nn::nfp::RegisterInfo* pOutRegisterInfo,
    bool isNicknameRequired,
    const nn::nfp::TagInfo& tagInfo,
    const nn::nfp::RegisterInfo& registerInfo) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutDeviceHandle);
    NN_ASSERT_NOT_NULL(pOutIsRegistered);
    NN_ASSERT_NOT_NULL(pOutRegisterInfo);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    nn::nfp::AmiiboSettingsStartParam startParam = {};
    SetupStartParam(&startParam, m_DeviceHandle, isNicknameRequired);

    // 通常のアプリから呼ばれる API
    Suspend();
    auto result = nn::nfp::StartNicknameAndOwnerSettings(
        pOutDeviceHandle,
        pOutIsRegistered,
        pOutRegisterInfo,
        startParam,
        tagInfo,
        registerInfo);
    Resume();

    return result;
}

nn::Result NfpProcessor::CallGameDataEraser(
    nn::nfp::DeviceHandle* pOutDeviceHandle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsActivated, "Not activated");
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    nn::nfp::AmiiboSettingsStartParam startParam = {};
    startParam.deviceHandle = m_DeviceHandle;

    // 通常のアプリから呼ばれる API
    Suspend();
    auto result = nn::nfp::StartGameDataEraser(pOutDeviceHandle, startParam, m_NfpInfo.tagInfo);
    Resume();

    return result;
}

nn::Result NfpProcessor::CallGameDataEraser(
    nn::nfp::DeviceHandle* pOutDeviceHandle,
    nn::nfp::TagInfo* pOutTagInfo) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutDeviceHandle);
    NN_ASSERT_NOT_NULL(pOutTagInfo);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // タグ未検出でも呼べる
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    nn::nfp::AmiiboSettingsStartParam startParam = {};
    startParam.deviceHandle = m_DeviceHandle;

    // 本体設定から呼ばれる API
    Suspend();
    auto result = nn::nfp::StartGameDataEraser(pOutTagInfo, pOutDeviceHandle, startParam);
    Resume();

    return result;
}

nn::Result NfpProcessor::CallRestorer(
    nn::nfp::DeviceHandle* pOutDeviceHandle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // タグ未検出でも呼べる
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    nn::nfp::AmiiboSettingsStartParam startParam = {};
    startParam.deviceHandle = m_DeviceHandle;

    // 通常のアプリから呼ばれる API
    Suspend();
    auto result = nn::nfp::StartRestorer(pOutDeviceHandle, startParam, m_NfpInfo.tagInfo);
    Resume();

    return result;
}

nn::Result NfpProcessor::CallRestorer(
    nn::nfp::DeviceHandle* pOutDeviceHandle,
    nn::nfp::TagInfo* pOutTagInfo) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutDeviceHandle);
    NN_ASSERT_NOT_NULL(pOutTagInfo);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // タグ未検出でも呼べる
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    nn::nfp::AmiiboSettingsStartParam startParam = {};
    startParam.deviceHandle = m_DeviceHandle;

    // 本体設定から呼ばれる API
    Suspend();
    auto result = nn::nfp::StartRestorer(pOutTagInfo, pOutDeviceHandle, startParam);
    Resume();

    return result;
}

nn::Result NfpProcessor::CallFormatter(
    nn::nfp::DeviceHandle* pOutDeviceHandle,
    nn::nfp::TagInfo* pOutTagInfo) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutTagInfo);
    NN_ASSERT_NOT_NULL(pOutDeviceHandle);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // タグ未検出でも呼べる
    NN_ASSERT(m_NextCommand == Command::None, "Another command is requested");

    nn::nfp::AmiiboSettingsStartParam startParam = {};
    startParam.deviceHandle = m_DeviceHandle;

    Suspend();
    auto result = nn::nfp::StartFormatter(pOutTagInfo, pOutDeviceHandle, startParam);
    Resume();

    return result;
}

}  // nfpdebug
