﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
  * @file
  * @brief   ControllerSequenceManager クラスの定義
  */

#pragma once

#include <mutex>

#include <nn/nn_Macro.h>
#include <nn/oe.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_ControllerSupport.h>
#include <nn/hid/hid_ResultControllerSupport.h>

#include "hidfw/hid.h"
#include "hidfw/layout.h"

//#include "VibrationDemo.h"

namespace VibrationDemo
{
    // コントローラシーケンスの起動時引数
    struct ControllerSequenceManagerArg
    {
        void SetDefault()
        {
            isSingleMode = false;
            minPlayerCount = 1;
            maxPlayerCount = 4;
        }
        bool                    isSingleMode;                   //!< 1人プレイの場合 true にします
        int8_t                  minPlayerCount;                 //!< 最小のプレイヤー人数 (マルチプレイ時)
        int8_t                  maxPlayerCount;                 //!< 最大のプレイヤー人数 (マルチプレイ時)
    };

    /**
    * @brief   コントローラの接続状態を監視し、適宜コントローラサポートアプレットの呼び出しを行うクラスです
    *
    * @details
    *  ControllerSequenceManager は、コントローラの接続状態を確認し
    *  アプリの動作に必要なコントローラの接続が消失した際に
    *  コントローラサポートアプレットの呼び出しを行い、常に適切なコントローラの接続を維持します。
    *
    *  @ref 処理の開始前に SetSupportedNpadIds で有効な NpadId を登録します
    *  @ref 処理の開始前に SetArgument で起動パラメータを設定します
    *  @ref StartControllerCheckThread の呼び出しでコントローラの監視が開始されます
    *
    */
    class ControllerSequenceManager
    {
        NN_DISALLOW_COPY(ControllerSequenceManager);
        NN_DISALLOW_MOVE(ControllerSequenceManager);

    public:
        enum CheckControllerMode
        {
            CheckControllerMode_Single,
            CheckControllerMode_Multi,
        };
    public:
        static const size_t         MemorySize = 1 * 1024 * 1024;   //!< この機能が利用するメモリの量
        static const int            SupportedNpadIdCountMax = 9;    //!< サポートする最大のコントローラ数
        static const nn::TimeSpan   DefaultWaitTime;                //!< 標準の待ち時間 (5s)
    public:
        static ControllerSequenceManager& GetInstance() NN_NOEXCEPT
        {
            static ControllerSequenceManager instance;
            return instance;
        }

        ControllerSequenceManager() NN_NOEXCEPT :
        m_IsAwaitConnection(false),
            m_IsAddWaitTime(false),
            m_IsDecidedMasterController(false),
            m_Mode(CheckControllerMode_Multi),
            m_MasterController(nn::hid::NpadId::No1),
            m_MinPlayerCount(1),
            m_MaxPlayerCount(4),
            m_CheckInterval(nn::TimeSpan::FromMilliSeconds(10)),
            m_LostTime(nn::TimeSpan()),
            m_WaitTime(nn::TimeSpan()),
            m_AdditionalTime(nn::TimeSpan::FromSeconds(3)),
            m_SupportedNpadIdCount(0),
            m_ConnectedControllerCount(0),
            m_PrevConnectedControllerCount(0)
        {
            nn::oe::Initialize();
            m_PrevFocusState = nn::oe::GetCurrentFocusState();
            m_CurFocusState = nn::oe::GetCurrentFocusState();
            m_PrevOperationMode = nn::oe::GetOperationMode();
            nn::os::InitializeTimerEvent(&m_CheckEvent, nn::os::EventClearMode_AutoClear);
            nn::Result result = nn::os::CreateThread(&Thread, ControllerCheckThread, NULL, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
            NN_ASSERT(result.IsSuccess(), "Cannot create thread.");
        }

        ~ControllerSequenceManager() NN_NOEXCEPT
        {
            m_MemoryPool.Finalize();
        }

        static void ControllerCheckThread(void *arg) NN_NOEXCEPT;

        void Initialize() NN_NOEXCEPT;

        // コンサポの呼び出しが発生しウェイトが必要となった場合 true になります
        bool TryWait() NN_NOEXCEPT;

        void UpdateConnectedControllers() NN_NOEXCEPT;

        void UpdateFocusState() NN_NOEXCEPT;

        void StartControllerCheckThread(void) NN_NOEXCEPT;

        void StartControllerCheckThread(const ControllerSequenceManagerArg& arg) NN_NOEXCEPT;

        void StopControllerCheckThread() NN_NOEXCEPT;

        bool SwitchMasterController() NN_NOEXCEPT;

        void ChangeSingleMode() NN_NOEXCEPT;

        void ChangeSingleMode(nn::hid::NpadIdType master) NN_NOEXCEPT;

        void ChangeMultiMode(void) NN_NOEXCEPT;

        bool CheckNecessaryCallApplet() NN_NOEXCEPT;

        nn::Result CallControllerSupportApplet() NN_NOEXCEPT;

        // Getter / Setter

        nn::hid::NpadIdType GetMasterControllerId() NN_NOEXCEPT { return m_MasterController; }

        CheckControllerMode GetMode() const NN_NOEXCEPT { return m_Mode; }

        nn::TimeSpan GetEventInterval() const NN_NOEXCEPT { return m_CheckInterval; }

        nn::os::TimerEventType GetTimerEvent() const NN_NOEXCEPT { return m_CheckEvent; }

        void CalcCurrentTime() NN_NOEXCEPT
        {
            m_CurrentTime = nn::os::GetSystemTick().ToTimeSpan();
        }

        void SetArgument(const ControllerSequenceManagerArg& arg) NN_NOEXCEPT;

        void SetSupportedNpadIds(nn::hid::NpadIdType* ids, int count);

        void SetSupportedNpadIds(const nn::hid::NpadIdType* ids, const int count) NN_NOEXCEPT
        {
            NN_ASSERT_LESS_EQUAL(count, 9);
            for (int i = 0; i < count; ++i)
            {
                m_NpadIds[i] = ids[i];
            }
            m_SupportedNpadIdCount = count;
        }

        int CalcConnectedControllerCount() NN_NOEXCEPT;

        bool IsInputController(nn::hid::NpadIdType id) NN_NOEXCEPT;

        bool ShowContinueDialog() NN_NOEXCEPT;
    private:
        static const size_t                                         ThreadStackSize = 1024 * 64;
    private:
        // コントローラの接続本数が足りている場合は true
        bool IsShortControllerCount() NN_NOEXCEPT;

        // マスターでないコントローラに入力がある場合は true
        bool IsInputByOtherController() NN_NOEXCEPT;

        // コントローラ数が過剰に接続された場合
        // マルチプレイ時は常に false が返ります
        bool IsConnectedControllerCountExcessive() NN_NOEXCEPT;

        // マスターコントローラが消失した場合は true
        bool IsLostMasterController() NN_NOEXCEPT;

        // BGフォーカスから戻ってきた直後は true
        bool IsResumeFocus() NN_NOEXCEPT;

        bool IsSwitchOperationMode() NN_NOEXCEPT
        {
            // Handheld -> Console のみを検知します
            // Console -> Handheld は検知対象外とします
            return (m_PrevOperationMode == nn::oe::OperationMode_Handheld) && (nn::oe::GetOperationMode() == nn::oe::OperationMode_Console);
        }
    private:
        bool                        m_IsAwaitConnection;            //!< マスターコントローラの接続待ち (シングルモード用)
        bool                        m_IsAddWaitTime;                //!< 待ち時間を延ばします
        bool                        m_IsDecidedMasterController;    //!< マスターコントローラが確定しているか

        CheckControllerMode         m_Mode;                         //!< 確認モード
        nn::hid::NpadIdType         m_MasterController;             //!< マスターコントローラ
        int8_t                      m_MinPlayerCount;               //!< 最少人数
        int8_t                      m_MaxPlayerCount;               //!< 最大人数

        nn::TimeSpan                m_CurrentTime;                  //!< 現在のシステムチック
        nn::TimeSpan                m_CheckInterval;                //!< 確認間隔
        nn::os::TimerEventType      m_CheckEvent;
        nn::TimeSpan                m_LostTime;                     //!< マスターコントローラを失った時間 (シングルモード用)
        nn::oe::FocusState          m_PrevFocusState;               //!< 直前のアプリケーションフォーカス状態 (シングルモード用)
        nn::oe::FocusState          m_CurFocusState;                //!< 現在のアプリケーションフォーカス状態 (シングルモード用)
        nn::oe::OperationMode       m_PrevOperationMode;            //!< 直前の動作モード (シングルモード用)

        nn::TimeSpan                m_WaitTime;                     //!< マスターコントローラが切断されてからの許容時間 (シングルモード用)
        nn::TimeSpan                m_AdditionalTime;               //!< 追加時間

        nn::mem::StandardAllocator  m_MemoryPool;                   //!< コントローラサポート機能が利用するメモリ領域

                                                                    //!< サポートしているコントローラ
        nn::hid::NpadIdType         m_NpadIds[SupportedNpadIdCountMax];
        int                         m_SupportedNpadIdCount;

        //!< 接続中のコントローラ
        nn::hid::NpadIdType         m_ConnectedNpadIds[SupportedNpadIdCountMax];
        int                         m_ConnectedControllerCount;
        int                         m_PrevConnectedControllerCount; //!< 直前に接続されていたコントローラの本数

        static NN_OS_ALIGNAS_THREAD_STACK char                      ThreadStack[ThreadStackSize];
        static nn::os::ThreadType                                   Thread;
    };
}

#define gControllerSequenceManager (VibrationDemo::ControllerSequenceManager::GetInstance())
