﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Base.h"
#include "File.h"

namespace VibrationCollection
{
    nn::Result FileManager::Mount() NN_NOEXCEPT
    {
        nn::Result result;
        TT_LOG("# -----------------------------------------------\n");
        TT_LOG("# Mount (MountType: %s)\n", m_MountType == MountType_Sd ? "SD" : "HOST PC");
        switch (m_MountType)
        {
        case VibrationCollection::MountType_Sd:
        {
            TT_LOG("# (SD) MountSdCardForDebug (MountName: SD)\n");
            TT_LOG("# (SD) %s\n", m_RootPath.c_str());
            result = nn::fs::MountSdCardForDebug("SD");

            auto pos = m_RootPath.find_first_of(":");
            if (pos != std::string::npos)
            {
                // パスに「:」が含まれている場合、失敗とします
                TT_LOG("# %s\n", m_RootPath.c_str());
                TT_LOG("# %*s\n", pos + 1, "^");
                TT_LOG("# (SD) Mount name contains illegal characters. (near ^)\n", m_RootPath.c_str());
                UnMount();
                return nn::fs::ResultUnexpected();
            }
            m_MountPath = m_RootPath;
            break;
        }
        case VibrationCollection::MountType_Host:
        {
            TT_LOG("# (HOST PC) MountHost (MountName: PC) (RootPath: %s)\n", m_RootPath.c_str());

            // ドライブレターを検索します
            std::string divChar = ":/";
            auto drivePos = m_RootPath.find(divChar, 0);
            auto isFindDriveLetter = (drivePos != std::string::npos) && (m_RootPath.find(divChar, drivePos + 2) == std::string::npos);

            if (!isFindDriveLetter)
            {
                // 「:/」が見つからない場合「:\」を検索
                divChar = ":\\";
                drivePos = m_RootPath.find(divChar, 0);
                isFindDriveLetter = (drivePos != std::string::npos) && (m_RootPath.find(divChar, drivePos + 2) == std::string::npos);
            }

            if (!isFindDriveLetter)
            {
                // ドライブレターが見つからない、もしくは区切り文字が複数個発見された場合失敗とします
                return nn::fs::ResultUnexpected();
            }

            // マウント名に不正な文字が含まれていないか検索します
            if (
                (m_RootPath.substr(0, drivePos).find("@", 0) == 0) ||                       // マウント名が@から始まる
                (m_RootPath.substr(0, drivePos).find(":", 0) != std::string::npos) ||       // マウント名に:が含まれる
                (m_RootPath.substr(0, drivePos).find("/", 0) != std::string::npos)          // マウント名に/が含まれる
                )
            {
                // マウント名として不正な文字が含まれていた場合は失敗とする
                return nn::fs::ResultUnexpected();
            }

            // 区切り文字を含めたドライブ名をコピーします
            std::string driveName = m_RootPath.substr(0, drivePos + 2);
            TT_LOG("# (HOST PC) DriveLetter : %s\n", driveName.c_str());
            result = nn::fs::MountHost("PC", driveName.c_str());
            m_TempRootPath = m_RootPath;
            m_MountPath = m_RootPath.substr(driveName.size(), m_RootPath.size() - (driveName.size()));
            break;
        }
        default: NN_UNEXPECTED_DEFAULT;
        }
        if (result.IsSuccess())
        {
            TT_LOG("# Success\n");
        }
        else
        {
            TT_LOG("# Failure\n");
        }
        TT_LOG("# -----------------------------------------------\n");
        return result;
    }

    void FileManager::UnMount() NN_NOEXCEPT
    {
        TT_LOG("# -----------------------------------------------\n");
        TT_LOG("# UnMount (MountType: %s)\n", m_MountType == MountType_Sd ? "SD" : "HOST PC");
        switch (m_MountType)
        {
        case VibrationCollection::MountType_Sd:
        {
            TT_LOG("# (SD) UnMount\n");
            nn::fs::Unmount("SD");
            break;
        }
        case VibrationCollection::MountType_Host:
        {
            TT_LOG("# (HOST PC) UnMount\n");
            nn::fs::Unmount("PC");
            break;
        }
        default: NN_UNEXPECTED_DEFAULT;
        }
        TT_LOG("# -----------------------------------------------\n");
        m_MountPath.clear();
    }

    bool FileManager::CheckPath(std::string str, PathType type) NN_NOEXCEPT
    {
        bool result = false;
        switch (type)
        {
        case VibrationCollection::PathType_Directory:
            result = (str.find_first_of("<>*?:|", 0) == std::string::npos);
            break;
        case VibrationCollection::PathType_FileName:
            result = (str.find_first_of("<>*?:|/\\\"", 0) == std::string::npos);
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
        return result;
    }

    nn::Result FileManager::ExportFile(const char* fileName, uint8_t* buffer, size_t size) NN_NOEXCEPT
    {
        TT_LOG("# (Bnvib) Export -> %s (%d byte)\n", fileName, size);
        std::string mountName = (m_MountType == MountType_Sd) ? "SD:/" : "PC:/";

        if (Mount().IsFailure())
        {
            ShowAssertDialog("error", "Mount name contains illegal characters.\n\n" + m_RootPath + "\n", AssertDialogType_Error, AssertButtonType_OK);
            return nn::fs::ResultUnexpected();
        }

        nn::Result result;
        TT_LOG("# (Bnvib) ファイルの書き込みを開始します\n");
        nn::fs::DirectoryHandle hDirectory;
        nn::fs::FileHandle hFile;

        auto path = (mountName + m_MountPath);
        auto filePath = (mountName + m_MountPath + fileName);

        if (!CheckPath(m_MountPath, PathType_Directory))
        {
            ShowAssertDialog("error", "Output path contains illegal characters.\n\n" + m_MountPath + "\n", AssertDialogType_Error, AssertButtonType_OK);
            UnMount();
            return nn::fs::ResultUnexpected();
        }

        if (!CheckPath(fileName, PathType_FileName))
        {
            ShowAssertDialog("error", "File name contains illegal characters.\n\n" + filePath + "\n", AssertDialogType_Error, AssertButtonType_OK);
            UnMount();
            return nn::fs::ResultUnexpected();
        }

        if (filePath.length() > 260 || filePath.size() > nn::fs::EntryNameLengthMax)
        {
            ShowAssertDialog("error", "File name too long.\n\n" + filePath + "\n", AssertDialogType_Error, AssertButtonType_OK);
            UnMount();
            return nn::fs::ResultUnexpected();
        }

        if (filePath.length() - path.length()  < 1)
        {
            ShowAssertDialog("error", "File name too short.\n\n" + filePath + "\n", AssertDialogType_Error, AssertButtonType_OK);
            UnMount();
            return nn::fs::ResultUnexpected();
        }

        TT_LOG("# (Bnvib) ディレクトリを開きます [%s]\n", path.c_str());
        result = nn::fs::OpenDirectory(&hDirectory, path.c_str(), nn::fs::OpenDirectoryMode_File);
        if (result.IsFailure())
        {
            TT_LOG("# (Bnvib) オープンに失敗しました。読み込みを中止します\n");
            UnMount();
            ShowAssertDialog("error", "failed to open directory.\n\n" + path + "\n", AssertDialogType_Error, AssertButtonType_OK);
            return result;
        }

        result = nn::fs::OpenFile(&hFile, filePath.c_str(), nn::fs::OpenMode_AllowAppend | nn::fs::OpenMode_Write);

        if (result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result))
        {
            if (!nn::fs::ResultPathNotFound::Includes(result))
            {
                nn::fs::CloseFile(hFile);
                if (ShowAssertDialog("warning", "Are you sure you want to overwrite this file?\n\n" + filePath + "\n", AssertDialogType_Warning, AssertButtonType_YesNo) == nns::hidfw::layout::Dialog::DialogResult_No)
                {
                    TT_LOG("# (Bnvib) ディレクトリを閉じます\n");
                    nn::fs::CloseDirectory(hDirectory);
                    TT_LOG("# (Bnvib) マウントを解除します\n");
                    UnMount();
                    return nn::fs::ResultAlreadyExists();
                }
                TT_LOG("# (Bnvib)既に同一名のファイルが存在するため削除します\n");
                nn::fs::DeleteFile(filePath.c_str());
            }
        }
        else
        {
            nn::fs::CloseFile(hFile);
            TT_LOG("# (Bnvib) ディレクトリを閉じます\n");
            nn::fs::CloseDirectory(hDirectory);
            TT_LOG("# (Bnvib) マウントを解除します\n");
            UnMount();
            return result;
        }

        nn::fs::CreateFile(filePath.c_str(), 0);
        nn::fs::OpenFile(&hFile, filePath.c_str(), nn::fs::OpenMode_AllowAppend | nn::fs::OpenMode_Write);
        nn::fs::WriteFile(hFile, 0, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush));

        nn::fs::CloseFile(hFile);
        TT_LOG("# (Bnvib) ディレクトリを閉じます\n");
        nn::fs::CloseDirectory(hDirectory);
        TT_LOG("# (Bnvib) マウントを解除します\n");
        UnMount();

        return nn::ResultSuccess();

    }

    void FileManager::ChangePath() NN_NOEXCEPT
    {
        m_MountType = m_TempMountType;
        m_RootPath = m_TempRootPath;
    }

    int FileManager::LoadBnvib(bool isLoadMemory) NN_NOEXCEPT
    {
        ChangePath();

        nn::Result result;

        result = Mount();
        if (result.IsFailure())
        {
            TT_LOG("# (Bnvib) マウントに失敗しました\n");
            return 0;
        }
        std::string mountName = (m_MountType == MountType_Sd) ? "SD:/" : "PC:/";

        TT_LOG("# (Bnvib) ファイルの読み込みを開始します (最大 %d)\n", MaxEntryCount);
        int64_t fileCount = 0;
        nn::fs::DirectoryEntry entry[MaxEntryCount];
        nn::fs::DirectoryHandle hDirectory;

        TT_LOG("# (Bnvib) ディレクトリを開きます [%s]\n", (mountName + m_MountPath + "/Bnvib/").c_str());
        result = nn::fs::OpenDirectory(&hDirectory, (mountName + m_MountPath + "/Bnvib/").c_str(), nn::fs::OpenDirectoryMode_File);
        if (result.IsFailure())
        {
            TT_LOG("# (Bnvib) オープンに失敗しました。読み込みを中止します\n");
            UnMount();
            return 0;
        }
        TT_LOG("# (Bnvib) ディレクトリの子エントリを列挙します\n");
        result = nn::fs::ReadDirectory(&fileCount, entry, hDirectory, MaxEntryCount);
        if (result.IsFailure())
        {
            TT_LOG("# (Bnvib) 失敗しました。\n");
            nn::fs::CloseDirectory(hDirectory);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            UnMount();
            return 0;
        }

        for (size_t i = 0; i < m_BnvibFile.size(); ++i)
        {
            if (m_BnvibFile.at(i).FileData != nullptr)
            {
                delete m_BnvibFile.at(i).FileData;
                m_BnvibFile.at(i).FileData = nullptr;
            }
        }
        m_BnvibMemorySize = 0;
        m_BnvibFile.clear();
        m_BnvibFile.resize(0);

        BnvibFileData fileData;
        if (fileCount > 0)
        {
            TT_LOG("# (Bnvib) %d 個のファイルを発見しました。リストへの追加を開始します\n", fileCount - 1);
            for (int i = 0; i < fileCount; ++i)
            {
                if ((i < 3) || (i >= fileCount - 3))
                {
                    TT_LOG("# (Bnvib) (%03d/%03d) : %s\n", i, fileCount - 1, entry[i].name);
                    if ((i == 3) && (fileCount > 6))
                    {
                        TT_LOG("# (Bnvib) ...(中略)...\n");
                    }
                }
                if (static_cast<size_t>(entry[i].fileSize) > BnvibFileData::FileSizeMax)
                {
                    TT_LOG("# (Bnvib) ファイルサイズが大きすぎます -> (%03d/%03d) : %s %ld\n", i, fileCount - 1, entry[i].name, entry[i].fileSize);
                }
                else
                {
                    fileData.FileName = entry[i].name;
                    fileData.FileSize = std::min(entry[i].fileSize, (int64_t)BnvibFileData::FileSizeMax);

                    const auto extPos = fileData.FileName.find_last_of(".");
                    auto ext = fileData.FileName.substr(extPos, fileData.FileName.size() - extPos);
                    std::transform(ext.begin(), ext.end(), ext.begin(), tolower);

                    if (strcmp(ext.c_str(), ".bnvib") == 0)
                    {
                        m_BnvibFile.push_back(fileData);
                        if (isLoadMemory)
                        {
                            if (ReadBnvib(m_BnvibFile.size() - 1).IsSuccess())
                            {
                                m_BnvibMemorySize += m_BnvibFile.back().FileSize;
                                auto& loopSetting = m_BnvibFile.back().Loop;

                                nn::hid::VibrationFileInfo outInfo;
                                nn::hid::VibrationFileParserContext outContext;
                                nn::hid::ParseVibrationFile(&outInfo, &outContext, m_BnvibFile.back().FileData, m_BnvibFile.back().FileSize);

                                loopSetting.isLoop = outInfo.isLoop;
                                loopSetting.loopStartPosition = outInfo.loopStartPosition;
                                loopSetting.loopEndPosition = outInfo.loopEndPosition;
                                loopSetting.loopInterval = outInfo.loopInterval;
                            }
                            else
                            {
                                TT_LOG("# (Bnvib) メモリへの展開に失敗しました (%03d/%03d) : %s\n", i, fileCount - 1, entry[i].name);
                            }
                        }
                    }
                    else
                    {
                        TT_LOG("# (Bnvib) Bnvibファイルではありません。スキップします (%03d/%03d) : %s\n", i, fileCount - 1, entry[i].name);
                    }
                }
            }
        }
        else
        {
            TT_LOG("# (Bnvib) ファイルが見つかりませんでした\n");
        }
        TT_LOG("# (Bnvib) ディレクトリを閉じます\n");
        nn::fs::CloseDirectory(hDirectory);
        TT_LOG("# (Bnvib) マウントを解除します\n");
        UnMount();
        return static_cast<int>(m_BnvibFile.size());
    } // NOLINT(readability/fn_size)

    int FileManager::LoadAudio(bool isLoadMemory) NN_NOEXCEPT
    {
        ChangePath();

        nn::Result result;

        result = Mount();
        if (result.IsFailure())
        {
            TT_LOG("# (Audio) マウントに失敗しました\n");
            return 0;
        }
        std::string mountName = (m_MountType == MountType_Sd) ? "SD:/" : "PC:/";
        int64_t fileCount = 0;
        nn::fs::DirectoryEntry entry[MaxEntryCount];
        nn::fs::DirectoryHandle hDirectory;
        std::string fileName;

        TT_LOG("# (Audio) ファイルのリストアップを開始します (最大 %d)\n", MaxEntryCount);
        TT_LOG("# (Audio) ディレクトリを開きます [%s]\n", (mountName + m_MountPath + "/se/").c_str());
        result = nn::fs::OpenDirectory(&hDirectory, (mountName + m_MountPath + "/se/").c_str(), nn::fs::OpenDirectoryMode_File);
        if (result.IsFailure())
        {
            TT_LOG("# (Audio) オープンに失敗しました。読み込みを中止します\n");
            UnMount();
            return 0;
        }
        TT_LOG("# (Audio) ディレクトリの子エントリを列挙します\n");
        result = nn::fs::ReadDirectory(&fileCount, entry, hDirectory, MaxEntryCount);
        if (result.IsFailure())
        {
            TT_LOG("# (Audio) 失敗しました。\n");
            nn::fs::CloseDirectory(hDirectory);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            UnMount();
            return 0;
        }

        AudioFileData fileData;

        for (size_t i = 0; i < m_AudioFile.size(); ++i)
        {
            if (m_AudioFile.at(i).FileData != nullptr)
            {
                delete m_AudioFile.at(i).FileData;
                m_AudioFile.at(i).FileData = nullptr;
            }
        }
        m_AudioMemorySize = 0;
        m_AudioFile.clear();
        m_AudioFile.resize(0);

        if (fileCount > 0)
        {
            for (int i = 0; i < fileCount; ++i)
            {
                if ((i < 3) || (i >= fileCount - 3))
                {
                    TT_LOG("# (Audio) (%03d/%03d) : %s\n", i, fileCount - 1, entry[i].name);
                    if ((i == 3) && (fileCount > 6))
                    {
                        TT_LOG("# (Audio) ...(中略)...\n");
                    }
                }
                fileData.FileName = entry[i].name;
                fileData.FileSize = entry[i].fileSize;
                {
                    const auto extPos = fileData.FileName.find_last_of(".");
                    auto ext = fileData.FileName.substr(extPos, fileData.FileName.size() - extPos);
                    std::transform(ext.begin(), ext.end(), ext.begin(), tolower);

                    if (strcmp(ext.c_str(), ".wav") == 0 || strcmp(ext.c_str(), ".wave") == 0)
                    {
                        fileData.FileType = AudioFileData::FileType_Wave;
                    }
                    else if (strcmp(ext.c_str(), ".adpcm") == 0)
                    {
                        fileData.FileType = AudioFileData::FileType_Adpcm;
                    }
                    else
                    {
                        fileData.FileType = AudioFileData::FileType_UnKnown;
                    }
                }
                m_AudioFile.push_back(fileData);
                if (isLoadMemory)
                {
                    if (ReadAudio(m_AudioFile.size() - 1).IsSuccess())
                    {
                        m_AudioMemorySize += m_AudioFile.back().FileSize;
                    }
                    else
                    {
                        TT_LOG("# (Audio) メモリへの展開に失敗しました (%03d/%03d) : %s\n", i, fileCount - 1, entry[i].name);
                    }
                }
            }
        }
        else
        {
            TT_LOG("# (Audio) ファイルが見つかりませんでした\n");
        }
        TT_LOG("# (Audio) ディレクトリを閉じます\n");
        nn::fs::CloseDirectory(hDirectory);
        TT_LOG("# (Audio) マウントを解除します\n");
        UnMount();
        return static_cast<int>(m_AudioFile.size());
    } // NOLINT(readability/fn_size)

    void FileManager::UnloadBnvib() NN_NOEXCEPT
    {
        m_BnvibMemorySize = 0;
        m_BnvibFile.clear();
        m_BnvibFile.resize(0);
    }

    void FileManager::UnloadAudio() NN_NOEXCEPT
    {
        m_AudioMemorySize = 0;
        m_AudioFile.clear();
        m_AudioFile.resize(0);
    }

    nn::Result FileManager::ReadBnvib(BnvibHandle handle) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(handle, (int32_t)0, (int32_t)m_BnvibFile.size());
        TT_LOG("# Read Bnvib File\n");
        nn::Result result;
        nn::fs::FileHandle file;
        int64_t fileSize = 0;

        std::string mountName = (m_MountType == MountType_Sd) ? "SD:/" : "PC:/";

        TT_LOG("# %s\n", (mountName + m_MountPath + "/vib/" + m_BnvibFile.at(handle).FileName).c_str());
        result = nn::fs::OpenFile(&file, (mountName + m_MountPath + "/Bnvib/" + m_BnvibFile.at(handle).FileName).c_str(), nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());

        // ファイルサイズを確認する
        result = nn::fs::GetFileSize(&fileSize, file);
        if (
            m_BnvibFile.at(handle).FileSize != static_cast<size_t>(fileSize)
            )
        {
            TT_LOG("# <警告> リストアップ時とファイルサイズが異なります。ファイルサイズが大きすぎる可能性があります\n");
            m_BnvibFile.at(handle).FileSize = static_cast<size_t>(fileSize);
        }
        NN_ASSERT(result.IsSuccess());

        if (m_BnvibFile.at(handle).FileData != nullptr)
        {
            delete[] m_BnvibFile.at(handle).FileData;
            m_BnvibFile.at(handle).FileData = nullptr;
        }
        m_BnvibFile.at(handle).FileData = new uint8_t[m_BnvibFile.at(handle).FileSize];
        // データを読み込む
        result = nn::fs::ReadFile(file, 0, m_BnvibFile.at(handle).FileData, m_BnvibFile.at(handle).FileSize);
        NN_ASSERT(result.IsSuccess());

        if (m_BnvibFile.at(handle).FileSize <= 1)
        {
            nn::fs::CloseFile(file);

            return result;
        }

        size_t dataSizeOffset = 4 + (m_BnvibFile.at(handle).FileData[3] << 24) + (m_BnvibFile.at(handle).FileData[2] << 16) + (m_BnvibFile.at(handle).FileData[1] << 8) + (m_BnvibFile.at(handle).FileData[0]);
        size_t dataSize = (m_BnvibFile.at(handle).FileData[dataSizeOffset + 3] << 24) + (m_BnvibFile.at(handle).FileData[dataSizeOffset + 2] << 16) + (m_BnvibFile.at(handle).FileData[dataSizeOffset + 1] << 8) + (m_BnvibFile.at(handle).FileData[dataSizeOffset]);

        m_BnvibFile.at(handle).Time = nn::TimeSpan::FromMilliSeconds(5 * (dataSize / 4));

        nn::fs::CloseFile(file);

        return result;
    }

    nn::Result FileManager::ReadAudio(AudioHandle handle) NN_NOEXCEPT
    {
        if (handle < 0 || handle >= static_cast<int32_t>(m_AudioFile.size()))
        {
            return nn::Result();
        }
        TT_LOG("# Read Audio File\n");
        nn::Result result;
        nn::fs::FileHandle file;
        int64_t fileSize = 0;

        std::string mountName = (m_MountType == MountType_Sd) ? "SD:/" : "PC:/";

        TT_LOG("# %s\n", (mountName + m_MountPath + "/se/" + m_AudioFile.at(handle).FileName).c_str());
        result = nn::fs::OpenFile(&file, (mountName + m_MountPath + "/se/" + m_AudioFile.at(handle).FileName).c_str(), nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());

        // ファイルサイズを確認する
        result = nn::fs::GetFileSize(&fileSize, file);
        if (
            m_AudioFile.at(handle).FileSize != static_cast<size_t>(fileSize) ||
            static_cast<size_t>(fileSize) >= AudioFileData::FileSizeMax
            )
        {
            TT_LOG("# <警告> リストアップ時とファイルサイズが異なります。ファイルサイズが大きすぎる可能性があります\n");
            m_AudioFile.at(handle).FileSize = static_cast<size_t>(fileSize);
        }
        NN_ASSERT(result.IsSuccess());

        if (m_AudioFile.at(handle).FileData != nullptr)
        {
            delete[] reinterpret_cast<uint8_t*>(m_AudioFile.at(handle).FileData);
            m_AudioFile.at(handle).FileData = nullptr;
        }

        m_AudioFile.at(handle).FileData = new uint8_t[m_AudioFile.at(handle).FileSize];
        // データを読み込む
        result = nn::fs::ReadFile(file, 0, m_AudioFile.at(handle).FileData, m_AudioFile.at(handle).FileSize);
        NN_ASSERT(result.IsSuccess());

        nn::fs::CloseFile(file);

        return result;
    }

    FileManager::FileManager() NN_NOEXCEPT :
        m_MountType(MountType_Sd),
        m_TempMountType(MountType_Sd),
        m_BnvibMemorySize(0),
        m_AudioMemorySize(0)
    {
        m_BnvibFile.clear();
        m_AudioFile.clear();
        m_MountPath = m_RootPath = m_TempRootPath = "";
    }

    void Reload(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(pushButton);
        NN_UNUSED(param);

        gFileManager.LoadBnvib(true);
        gFileManager.LoadAudio(true);
    }

}
