﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <string>
#include <vector>

#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>

#include "../sgx/SimpleGfx_Types.h"

// 方向キーで決定・キャンセル操作が行えるようにする
//#define NNS_HID_ENABLE_CURSOR_DECISION

#define NN_APP_LOG0(log)    NN_LOG("[ControllerFlashWriter] %s", (log))

namespace nns { namespace hid { namespace util {

// リソースデータを扱うファイルシステムのマウント名
const char* const RomMountName = "rom";

// 背景色
const nns::sgx::Color BackgroundColor = { { 0xF0, 0xF0, 0xF0, 0xFF } };

const nn::hid::NpadButtonSet DecideButtonMask = nn::hid::NpadButton::A::Mask | nn::hid::NpadButton::L::Mask;

const nn::hid::NpadButtonSet CancelButtonMask = nn::hid::NpadButton::B::Mask | nn::hid::NpadButton::ZL::Mask;

template <typename T>
int FindIndex(const std::vector<T>& vec, const T& target) NN_NOEXCEPT
{
    auto iter = std::find(vec.begin(), vec.end(), target);
    if (iter != vec.end())
    {
        // 先頭からの距離 = インデックス
        return std::distance(vec.begin(), iter);
    }

    // 見つからなかった
    return -1;
}

template <typename T>
bool Contains(const std::vector<T>& vec, const T& target) NN_NOEXCEPT
{
    return FindIndex(vec, target) >= 0;
}

void ReplaceAll(std::string* pSrouceStr, const char* targetStr, const char* replaceStr) NN_NOEXCEPT;

/**
 * @brief   サポートしているデバイスか判定します。
 */
bool IsSupportedDevice(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT;

/**
 * @brief   UniquePadType の名前を取得します。
 */
const char* GetUniquePadTypeName(nn::hid::system::UniquePadType type) NN_NOEXCEPT;

/**
 * @brief   UniquePadType の名前を取得します。
 */
const char* GetUniquePadTypeName(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT;

/**
 * @brief   コントローラーの LED インジケーター点灯パターンを取得します。
 */
uint8_t GetControllerIndicatorPattern(int controllerNumber) NN_NOEXCEPT;

/**
 * @brief   コントローラーの LED インジケーター点灯パターンを取得します。
 */
uint8_t GetControllerIndicatorPattern(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT;

/**
 * @brief   コントローラーの LED インジケーターを描画します。
 */
void DrawControllerIndicator(float x, float y, uint8_t pattern, float fontSize) NN_NOEXCEPT;

/**
 * @brief   RomFs をマウントします。
 */
void InitializeFileSystem() NN_NOEXCEPT;

/**
 * @brief   RomFs のマウントを解除します。
 */
void FinalizeFileSystem() NN_NOEXCEPT;

/**
 * @brief   コントローラのファームウェアバージョンを読み込むクラスです。
 */
class FirmwareVersionReader final
{
    NN_DISALLOW_COPY(FirmwareVersionReader);
    NN_DISALLOW_MOVE(FirmwareVersionReader);

public:
    FirmwareVersionReader() NN_NOEXCEPT :
        m_Mutex(true),
        m_UniquePadId(),
        m_Version(),
        m_IsValid(false),
        m_HasValidVersion(false)
    {
    }

    // 対象コントローラーの UniquePadId を設定
    void SetUniquePadId(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT;

    // ファームウェアバージョンを無効化
    void Invalidate() NN_NOEXCEPT;

    // 有効化されているか
    bool IsValid() const NN_NOEXCEPT;

    // 有効なバージョンを取得済みか
    bool HasValidVersion() const NN_NOEXCEPT;

    // 読み込んだバージョン情報を取得
    bool Get(nn::hid::system::FirmwareVersion* pOutVersion) const NN_NOEXCEPT;

    // バージョン情報の更新
    void Update() NN_NOEXCEPT;

private:
    void InvalidateVersion() NN_NOEXCEPT;

private:
    mutable nn::os::Mutex m_Mutex;

    nn::hid::system::UniquePadId        m_UniquePadId;
    nn::hid::system::FirmwareVersion    m_Version;
    bool m_IsValid;
    bool m_HasValidVersion;
};
}}}  // nns::hid::util
