﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <memory>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/debug/hid_FirmwareUpdate.h>
#include <nn/result/result_HandlingUtility.h>

#include "util.h"
#include "UniquePadManager.h"

namespace nns { namespace hid { namespace util {

namespace
{

// UniquePad のコントローラー番号と NpadId の対応
const nn::hid::NpadIdType NpadIdList[] =
{
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
    nn::hid::NpadId::No5,
    nn::hid::NpadId::No6,
    nn::hid::NpadId::No7,
    nn::hid::NpadId::No8,
    nn::hid::NpadId::Handheld
};

const int NpadIdListCount = NN_ARRAY_SIZE(NpadIdList);

}  // anonymous

UniquePadManager::UniquePadManager() NN_NOEXCEPT
    : m_Mutex(true)
    , m_IsInitialized(false)
    , m_IsInvalidateVersionRequested(false)
    , m_UniquePadConnectionEvent()
    , m_pUpdateEvent(nullptr)
    , m_PadInfos()
    , m_PadCount(0)
{
}

void UniquePadManager::Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(!m_IsInitialized, "Already initialized");

    nn::hid::system::BindUniquePadConnectionEvent(&m_UniquePadConnectionEvent, nn::os::EventClearMode_ManualClear);
    m_IsInitialized = true;
}

void UniquePadManager::Finalize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    nn::os::DestroySystemEvent(&m_UniquePadConnectionEvent);
    UnbindUpdateEvent();
    m_IsInitialized = false;
}

void UniquePadManager::BindUpdateEvent(nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pEvent);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    m_pUpdateEvent = pEvent;

    // 最初はシグナル状態
    nn::os::SignalLightEvent(m_pUpdateEvent);
}

void UniquePadManager::UnbindUpdateEvent() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    m_pUpdateEvent = nullptr;
}

nn::Result UniquePadManager::GetNpadId(nn::hid::NpadIdType* pOutId, int padIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutId);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    if (padIndex >= GetPadCount())
    {
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    auto padId = m_PadInfos[padIndex].id;
    if (!IsSupportedDevice(padId))
    {
        // 非サポートデバイス
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    int controllerNumber;
    auto result = nn::hid::system::GetUniquePadControllerNumber(
        &controllerNumber,
        padId);
    if (nn::hid::system::ResultUniquePadNotAvailable::Includes(result) ||
        controllerNumber == 0)
    {
        // 割り当て無しは Handheld と見なす
        *pOutId = nn::hid::NpadId::Handheld;
    }
    else if (result.IsFailure())
    {
        NN_RESULT_THROW(result);
    }
    else
    {
        // コントローラー番号に対応する NpadId
        NN_ABORT_UNLESS_LESS(controllerNumber, NpadIdListCount);
        *pOutId = NpadIdList[controllerNumber - 1];
    }

    NN_RESULT_SUCCESS;
}

nn::Result UniquePadManager::GetUniquePadId(nn::hid::system::UniquePadId* pOutId, int padIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutId);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    if (padIndex >= GetPadCount())
    {
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    auto padId = m_PadInfos[padIndex].id;
    if (!IsSupportedDevice(padId))
    {
        // 非サポートデバイス
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    *pOutId = padId;

    NN_RESULT_SUCCESS;
}

bool UniquePadManager::GetFirmwareVersion(nn::hid::system::FirmwareVersion* pOutVersion, int padIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutVersion);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    if (padIndex >= GetPadCount())
    {
        return false;
    }

    return m_PadInfos[padIndex].versionReader.Get(pOutVersion);
}

bool UniquePadManager::IsFirmwareUpdating(nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    for (const auto& pad : m_PadInfos)
    {
        if (pad.id._storage == id._storage &&
            pad.isUpdating)
        {
            return true;
        }
    }

    return false;
}

void UniquePadManager::RequestInvalidateFirmwareVersion() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    m_IsInvalidateVersionRequested = true;
}

void UniquePadManager::UpdateFirmwareUpdatingDevice() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    for (int i = 0; i < m_PadCount; i++)
    {
        auto& pad = m_PadInfos[i];
        bool isUpdating;
        if (nn::hid::debug::IsFirmwareUpdatingDevice(&isUpdating, pad.id).IsSuccess())
        {
            pad.isUpdating = isUpdating;
        }
        else
        {
            pad.isUpdating = false;
        }
    }
}

void UniquePadManager::Refresh() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    nn::hid::system::UniquePadId ids[NN_ARRAY_SIZE(m_PadInfos)];
    m_PadCount = nn::hid::system::ListUniquePads(ids, NN_ARRAY_SIZE(m_PadInfos));
    for (int i = 0; i < NN_ARRAY_SIZE(m_PadInfos); i++)
    {
        auto& pad = m_PadInfos[i];
        pad.Clear();
        if (i < m_PadCount)
        {
            pad.id = ids[i];
        }
    }

    UpdateFirmwareUpdatingDevice();

#if 0
    NN_LOG("[UniquePadManager:%s] PadCount: %d\n", NN_CURRENT_FUNCTION_NAME, m_PadCount);
    for (int i = 0; i < m_PadCount; i++)
    {
        auto& pad = m_PadInfos[i];

        nn::hid::system::UniquePadInterface padInterface;
        nn::hid::system::GetUniquePadInterface(&padInterface, pad.id);
        NN_LOG("  Pad%d: ID=%d, Type=%d, Interface=%d\n",
            i,
            pad.id,
            nn::hid::system::GetUniquePadType(pad.id),
            padInterface);
    }
#endif
}

void UniquePadManager::Update() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    if (nn::os::TryWaitSystemEvent(&m_UniquePadConnectionEvent))
    {
        nn::os::ClearSystemEvent(&m_UniquePadConnectionEvent);
        Refresh();
        if (m_pUpdateEvent != nullptr)
        {
            nn::os::SignalLightEvent(m_pUpdateEvent);
        }
    }

    for (int i = 0; i < GetPadCountMax(); i++)
    {
        auto& pad    = m_PadInfos[i];
        auto& reader = pad.versionReader;
        if (i < m_PadCount && IsSupportedDevice(pad.id))
        {
            if (!reader.IsValid())
            {
                // サポートしているデバイスを有効化
                reader.SetUniquePadId(pad.id);
            }
        }
        else
        {
            // サポートしていないデバイスはバージョンを読まない
            reader.Invalidate();
        }

        reader.Update();
    }
}

void UniquePadManager::InvalidateFirmwareVersionIfNeeded() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT(m_IsInitialized, "Not initialized");

    if (m_IsInvalidateVersionRequested)
    {
        m_IsInvalidateVersionRequested = false;

        for (auto& pad : m_PadInfos)
        {
            pad.versionReader.Invalidate();
        }
    }
}

void UniquePadManager::PadInfo::Clear() NN_NOEXCEPT
{
    id = nn::hid::system::UniquePadId();
    versionReader.Invalidate();
    isUpdating = false;
}

}}}  // nns::hid::util
