﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <memory>

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_ScopeExit.h>

#include "FirmwareImageEnumerator.h"
#include "Json.h"
#include "util.h"

namespace nns { namespace hid { namespace util {

namespace
{

// SD カードのマウント名
const char SdMountName[] = "sd";

// FW イメージを格納するディレクトリ名
const char ImageDirectory[] = "ControllerFirmware";

// リストアップする対象か判定
bool IsEntryAcceptable(const nn::fs::DirectoryEntry& entry) NN_NOEXCEPT
{
    if (entry.directoryEntryType != nn::fs::DirectoryEntryType_File)
    {
        // ファイル以外は拾わないはずだが一応弾く
        return false;
    }

    nn::util::string_view view(entry.name);
    if (!view.ends_with(".json") && !view.ends_with(".meta"))
    {
        // メタデータ以外は無視
        return false;
    }

    return true;
}

bool ParseMetaData(FirmwareImageInfo* pInfo, const char* path) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pInfo);
    NN_ASSERT_NOT_NULL(path);

    JsonDocument json;
    if (json.Open(path) != JsonResult::Success)
    {
        return false;
    }

    const auto& doc = json.GetDocument();
    auto hasArrayKey = [&doc](const char* key)
    {
        return doc.HasMember(key) && doc[key].IsArray();
    };
    auto hasStringKey = [&doc](const char* key)
    {
        return doc.HasMember(key) && doc[key].IsString();
    };

    if (!(hasArrayKey("device") && hasStringKey("chip") && hasStringKey("version")))
    {
        // 必須キーが定義されていない
        return false;
    }

    // Device type
    {
        const auto& devices = doc["device"];
        for (nne::rapidjson::SizeType i = 0; i < devices.Size(); i++)
        {
            const auto& device = devices[i];
            if (!device.IsString())
            {
                continue;
            }

            const auto* name = device.GetString();
            if (nn::util::Strnicmp(name, DeviceTypeNameJoyLeft, sizeof(DeviceTypeNameJoyLeft)) == 0)
            {
                pInfo->device.Set<DeviceType::JoyLeft>();
            }
            else if (nn::util::Strnicmp(name, DeviceTypeNameJoyRight , sizeof(DeviceTypeNameJoyRight)) == 0)
            {
                pInfo->device.Set<DeviceType::JoyRight>();
            }
            else if (nn::util::Strnicmp(name, DeviceTypeNameFullKey , sizeof(DeviceTypeNameFullKey)) == 0 ||
                     nn::util::Strnicmp(name, DeviceTypeNameProCon , sizeof(DeviceTypeNameProCon)) == 0)
            {
                pInfo->device.Set<DeviceType::FullKey>();
            }
        }
    }

    // Chip type
    {
        const auto* chipName = doc["chip"].GetString();
        if (nn::util::Strnicmp(chipName, ChipTypeNameMcu1, sizeof(ChipTypeNameMcu1)) == 0)
        {
            pInfo->chip = ChipType::Mcu1;
        }
        else if (nn::util::Strnicmp(chipName, ChipTypeNameMcu2, sizeof(ChipTypeNameMcu2)) == 0)
        {
            pInfo->chip = ChipType::Mcu2;
        }
        else
        {
            // 対応するチップではない
            return false;
        }
    }

    // Version
    {
        const auto* version = doc["version"].GetString();
        nn::util::Strlcpy(pInfo->version, version, sizeof(pInfo->version));
    }

    // File
    if (hasStringKey("file"))
    {
        const auto* file= doc["file"].GetString();
        nn::util::Strlcpy(pInfo->name, file, sizeof(pInfo->name));
    }
    else
    {
        pInfo->name[0] = '\0';
    }

    return true;
}

}  // anonymous

NN_DISABLE_WARNING_ARRAY_DEFAULT_INITIALIZATION_IN_CONSTRUCTOR;

FirmwareImageEnumerator::FirmwareImageEnumerator() NN_NOEXCEPT
    : m_Mutex(true)
    , m_IsMounted(false)
    , m_IsSdCardDetected(false)
    , m_Images()
    , m_FileCount(0)
{
}

FirmwareImageEnumerator::~FirmwareImageEnumerator() NN_NOEXCEPT
{
    Unmount();
}

FirmwareImageEnumerator* FirmwareImageEnumerator::GetInstance() NN_NOEXCEPT
{
    static FirmwareImageEnumerator s_Instance;
    return &s_Instance;
}

bool FirmwareImageEnumerator::IsBusy() const NN_NOEXCEPT
{
    // XXX: Mutex のロックを取得できなかったら busy と見なす
    bool isLockSuccess = m_Mutex.TryLock();
    NN_UTIL_SCOPE_EXIT
    {
        if (isLockSuccess)
        {
            m_Mutex.Unlock();
        }
    };

    return !isLockSuccess;
}

void FirmwareImageEnumerator::Mount() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    auto result = ::nn::fs::MountSdCardForDebug(SdMountName);
    if (::nn::fs::ResultSdCardAccessFailed::Includes(result))
    {
        // SD カード未挿入
        m_IsMounted        = false;
        m_IsSdCardDetected = false;
        //NN_APP_LOG0("SD card not found\n");
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        m_IsMounted        = true;
        m_IsSdCardDetected = true;
        ListupFiles();
    }
}

void FirmwareImageEnumerator::Unmount() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_IsMounted)
    {
        ::nn::fs::Unmount(SdMountName);
        m_IsMounted        = false;
        m_IsSdCardDetected = false;
        ClearFileList();
    }
}

bool FirmwareImageEnumerator::TryAutoMount() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_IsSdCardDetected == nn::fs::IsSdCardInserted())
    {
        // SD カードの挿入状態が更新されていない
        return false;
    }

    if (nn::fs::IsSdCardInserted())
    {
        Mount();
    }
    else
    {
        Unmount();
    }

    return true;
}

int FirmwareImageEnumerator::GetFileCount() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_FileCount;
}

bool FirmwareImageEnumerator::IsEmpty() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return GetFileCount() <= 0;
}

const FirmwareImage& FirmwareImageEnumerator::GetImage(int index) const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_ASSERT_RANGE(index, 0, m_FileCount);

    return m_Images[index];
}

void FirmwareImageEnumerator::ClearFileList() NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_Images); i++)
    {
        m_Images[i].Clear();
    }

    m_FileCount = 0;
}

void FirmwareImageEnumerator::ListupFiles() NN_NOEXCEPT
{
    NN_ASSERT(m_IsMounted, "Not mounted");

    ClearFileList();

    char directoryPath[nn::fs::EntryNameLengthMax];
    nn::util::SNPrintf(directoryPath, sizeof(directoryPath), "%s:/%s", SdMountName, ImageDirectory);
    FirmwareImage::SetBaseDirectory(directoryPath);

    nn::fs::DirectoryHandle directoryHandle;
    {
        auto result = nn::fs::OpenDirectory(&directoryHandle, directoryPath, nn::fs::OpenDirectoryMode_File);
        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            // ディレクトリがない
            NN_LOG("Directory \"%s\" is not found in the SD card\n", ImageDirectory);
            return;
        }
    }

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(directoryHandle);
    };

    // ディレクトリ内のイメージ一覧を作成
    auto* entries = new nn::fs::DirectoryEntry[FileCountMax];
    NN_ABORT_UNLESS_NOT_NULL(entries);
    NN_UTIL_SCOPE_EXIT
    {
        delete[] entries;
    };
    int64_t entryCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::ReadDirectory(&entryCount, entries, directoryHandle, FileCountMax)
    );

    for (decltype(entryCount) i = 0; i < entryCount; i++)
    {
        const auto& entry = entries[i];
        if (!IsEntryAcceptable(entry))
        {
            continue;
        }

        char metaPath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(metaPath, sizeof(metaPath), "%s/%s", directoryPath, entry.name);

        FirmwareImageInfo info;
        info.Clear();
        if (!ParseMetaData(&info, metaPath))
        {
            // メタデータが読めない場合は無視
            continue;
        }

        {
            if (info.name[0] == '\0')
            {
                // ファイルが指定されていない場合は、メタデータの拡張子を .bin に置き換えたファイルを使う
                std::string name(entry.name);
                util::ReplaceAll(&name, ".json", ".bin");
                nn::util::Strlcpy(info.name, name.c_str(), sizeof(info.name));
            }

            char imagePath[nn::fs::EntryNameLengthMax];
            nn::util::SNPrintf(imagePath, sizeof(imagePath), "%s/%s", directoryPath, info.name);
            nn::fs::FileHandle fileHandle;
            if (nn::fs::OpenFile(&fileHandle, imagePath, nn::fs::OpenMode_Read).IsFailure())
            {
                // イメージデータが読めない
                continue;
            }
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::GetFileSize(&info.size, fileHandle)
            );
        }
        //NN_LOG("Entry: %s\n", entry.name);

#if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::GetFileTimeStampForDebug(&info.timeStamp, metaPath)
        );
#endif  // if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)

        m_Images[m_FileCount].SetInfo(info);

        m_FileCount++;
    }
}

}}}  // nns::hid::util
