﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <vector>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_GuiWindow.h"
#include "../../input/Input.h"

namespace nns { namespace sgx { namespace gui {

bool Window::g_IsFirstInitialized = false;

// 初回使用時に設定するので初期値は入れない
WindowColorScheme Window::g_DefaultActiveColor   = {};
WindowColorScheme Window::g_DefaultInactiveColor = {};

std::vector<Window> Window::g_WindowList;

void Window::SetText(const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);
    NNS_SGX_GUI_SCOPED_LOCK;

    // UTF8 -> UTF32 変換
    {
        int lengthUtf32;
        auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&lengthUtf32, text);
        if (result != nn::util::CharacterEncodingResult_Success ||
            lengthUtf32 >= WindowTextLengthMax)
        {
            // 変換不能
            m_Caption[0] = 0;
            return;
        }
    }

    auto result = nn::util::ConvertStringUtf8ToUtf32(m_Caption, WindowTextLengthMax, text);
    if (result != nn::util::CharacterEncodingResult_Success)
    {
        // 変換失敗
        m_Caption[0] = 0;
        return;
    }
}

void Window::SetText(const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);
    NNS_SGX_GUI_SCOPED_LOCK;

    int length;
    for (length = 0; length < WindowTextLengthMax - 1 && text[length] != 0; length++)
    {
        m_Caption[length] = text[length];
    }

    m_Caption[length] = 0;
}

void Window::Update() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    InitializeIfNeeded();

    if (!IsVisible() || !IsEnabled())
    {
        return;
    }

    UiContainer::Update();
}

void Window::Render() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    InitializeIfNeeded();

    if (!IsVisible())
    {
        return;
    }

    // 背景要素の描画
    RenderBackground();

    // 子要素を描画
    UiContainer::Render();
}

void Window::RenderBackground() NN_NOEXCEPT
{
    const auto& color =
        IsFocused() ? m_ActiveColor :
        m_InactiveColor;
    auto opacity = GetOpacity();

    // 枠と背景
    Rectangle rect;
    GetRenderPosition(&rect.position);
    GetSize(&rect.size);

    Point2D points[] =
    {
        { { rect.x,              rect.y + m_CaptionHeight } },
        { { rect.x + rect.width, rect.y + m_CaptionHeight } },
        { { rect.x + rect.width, rect.y + rect.height } },
        { { rect.x,              rect.y + rect.height } },
    };
    Color colors[] =
    {
        color.contentsBackgroundBegin.BlendAlpha(opacity),
        color.contentsBackgroundBegin.BlendAlpha(opacity),
        color.contentsBackgroundEnd.BlendAlpha(opacity),
        color.contentsBackgroundEnd.BlendAlpha(opacity)
    };
    FillGradientPolygon(NN_ARRAY_SIZE(points), points, colors);
    FillRectangle(rect.x, rect.y, rect.width, m_CaptionHeight, color.captionBackground.BlendAlpha(opacity));
    DrawRectangle(rect, color.border.BlendAlpha(opacity), 1.0f);
    DrawLine(rect.x, rect.y + m_CaptionHeight, rect.x + rect.width, rect.y + m_CaptionHeight, color.border.BlendAlpha(opacity), 1.0f);

    ScopedFontContextSaver saver;

    // フォントサイズ設定
    SetTextSize(m_TextSize);

    // 文字色設定
    SetTextColor(color.captionText.BlendAlpha(opacity));

    // キャプション描画
    Size textSize;
    GetTextDrawSize(&textSize, m_Caption);
    DrawText(rect.x + 6, rect.y + (m_CaptionHeight - textSize.height) / 2, m_Caption);
}

void Window::InitializeDefaultColorScheme() NN_NOEXCEPT
{
    if (g_IsFirstInitialized)
    {
        return;
    }

    g_DefaultActiveColor   = { Colors::Silver(),   Colors::Navy().BlendAlpha(192), Colors::MidNightBlue().BlendAlpha(192), Colors::RoyalBlue(), Colors::White() };
    g_DefaultInactiveColor = { Colors::DarkGray(), Colors::Navy().BlendAlpha(192), Colors::MidNightBlue().BlendAlpha(192), Colors::Gray(),      Colors::Shadow() };
    g_IsFirstInitialized   = true;
}

void Window::SetupColorScheme() NN_NOEXCEPT
{
    InitializeDefaultColorScheme();
    m_ActiveColor   = g_DefaultActiveColor;
    m_InactiveColor = g_DefaultInactiveColor;
}

}}}  // nns::sgx::gui
