﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommon.h"
#include "SimpleGfx_GuiButtonBase.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   ボタンの配色です。
 */
struct ButtonColorScheme
{
    Color border;           //!< 枠線
    Color backgroundBegin;  //!< 背景始点
    Color backgroundEnd;    //!< 背景終点
    Color text;             //!< 文字
};

/**
 * @brief   スキンを使用しない単純なボタンを扱うクラスです。
 */
class SimpleButton : public ButtonBase
{
    NN_DISALLOW_COPY(SimpleButton);
    NN_DISALLOW_MOVE(SimpleButton);

public:
    /**
     * @brief   通常時のデフォルトの配色を設定します。
     */
    static void SetDefaultNormalColor(const ButtonColorScheme& scheme) NN_NOEXCEPT
    {
        g_DefaultNormalColor = scheme;
    }

    /**
     * @brief   無効時のデフォルトの配色を設定します。
     */
    static void SetDefaultDisabledColor(const ButtonColorScheme& scheme) NN_NOEXCEPT
    {
        g_DefaultDisabledColor = scheme;
    }

    /**
     * @brief   押下時のデフォルトの配色を設定します。
     */
    static void SetDefaultPressColor(const ButtonColorScheme& scheme) NN_NOEXCEPT
    {
        g_DefaultPressColor = scheme;
    }

public:
    SimpleButton() NN_NOEXCEPT
        : ButtonBase()
        , m_Text()
        , m_TextSize(24.0f)
        , m_NormalColor()
        , m_DisabledColor()
        , m_PressColor()
    {
    }

    /**
     * @brief   通常時の配色を設定します。
     */
    void SetNormalColor(const ButtonColorScheme& scheme) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_NormalColor = scheme;
    }

    /**
     * @brief   無効時の配色を設定します。
     */
    void SetDisabledColor(const ButtonColorScheme& scheme) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_DisabledColor = scheme;
    }

    /**
     * @brief   押下時の配色を設定します。
     */
    void SetPressColor(const ButtonColorScheme& scheme) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_PressColor = scheme;
    }

    /**
     * @brief   ボタンのテキストを UTF-8 で設定します。
     */
    void SetText(const char* text) NN_NOEXCEPT;

    /**
     * @brief   ボタンのテキストを UTF-32 で設定します。
     */
    void SetText(const uint32_t* text) NN_NOEXCEPT;

    /**
     * @brief   テキストの描画サイズを設定します。
     */
    void SetTextSize(float size) NN_NOEXCEPT
    {
        m_TextSize = size;
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

private:
    /**
     * @brief   デフォルト配色を初期化します。
     */
    static void InitializeDefaultColorScheme() NN_NOEXCEPT;

    /**
     * @brief   必要に応じて初期化処理を行います。
     *
     * @note    静的初期化時に配色が初期化されていない可能性があるため、
     *          最初に更新または描画されるまで初期化を遅延します。
     */
    void InitializeIfNeeded() NN_NOEXCEPT
    {
        if (GetState() == ButtonState_Init)
        {
            SetupColorScheme();
            SetState(ButtonState_Idle);
        }
    }

    /**
     * @brief   配色を初期化します。
     */
    void SetupColorScheme() NN_NOEXCEPT;

private:
    static bool                 g_IsDefaultColorInitialized;
    static ButtonColorScheme    g_DefaultNormalColor;       //!< デフォルトの通常配色
    static ButtonColorScheme    g_DefaultDisabledColor;     //!< デフォルトの無効時配色
    static ButtonColorScheme    g_DefaultPressColor;        //!< デフォルトの押下時配色

private:
    uint32_t            m_Text[ButtonTextLengthMax];

    float               m_TextSize;         //!< 文字の描画サイズ
    ButtonColorScheme   m_NormalColor;      //!< 通常配色
    ButtonColorScheme   m_DisabledColor;    //!< 無効時配色
    ButtonColorScheme   m_PressColor;       //!< 押下時配色
};

}}}  // nns::sgx::gui
