﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_GuiSimpleButton.h"
#include "../../input/NpadManager.h"
#include "../../input/TouchPanel.h"

namespace nns { namespace sgx { namespace gui {

bool SimpleButton::g_IsDefaultColorInitialized = false;

// 初期化時に設定するので初期値は入れない
ButtonColorScheme SimpleButton::g_DefaultNormalColor   = {};
ButtonColorScheme SimpleButton::g_DefaultDisabledColor = {};
ButtonColorScheme SimpleButton::g_DefaultPressColor    = {};

void SimpleButton::SetText(const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);
    NNS_SGX_GUI_SCOPED_LOCK;

    // UTF8 -> UTF32 変換
    {
        int lengthUtf32;
        auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&lengthUtf32, text);
        if (result != nn::util::CharacterEncodingResult_Success ||
            lengthUtf32 >= ButtonTextLengthMax)
        {
            // 変換不能
            m_Text[0] = 0;
            return;
        }
    }

    auto result = nn::util::ConvertStringUtf8ToUtf32(m_Text, ButtonTextLengthMax, text);
    if (result != nn::util::CharacterEncodingResult_Success)
    {
        // 変換失敗
        m_Text[0] = 0;
        return;
    }
}

void SimpleButton::SetText(const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);
    NNS_SGX_GUI_SCOPED_LOCK;

    int length = 0;
    for (; length < ButtonTextLengthMax - 1 && text[length] != 0; length++)
    {
        m_Text[length] = text[length];
    }

    m_Text[length] = 0;
}

void SimpleButton::Update() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    InitializeIfNeeded();

    ButtonBase::Update();
}

void SimpleButton::Render() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    InitializeIfNeeded();

    if (!IsVisible())
    {
        return;
    }

    bool isPressed = IsPressed();
    ButtonColorScheme color;
    switch (GetAppearance())
    {
    case ButtonAppearance::Normal:
    case ButtonAppearance::Focused:
        color = m_NormalColor;
        break;
    case ButtonAppearance::Pressed:
        color = m_PressColor;
        break;
    case ButtonAppearance::Disabled:
        color = m_DisabledColor;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    auto opacity = GetDisplayOpacity();

    // 枠と背景
    Rectangle rect;
    GetRenderPosition(&rect.position);
    GetSize(&rect.size);

    Point2D points[] =
    {
        { { rect.x,              rect.y } },
        { { rect.x + rect.width, rect.y } },
        { { rect.x + rect.width, rect.y + rect.height } },
        { { rect.x,              rect.y + rect.height } },
    };
    Color colors[] =
    {
        color.backgroundBegin.BlendAlpha(opacity),
        color.backgroundBegin.BlendAlpha(opacity),
        color.backgroundEnd.BlendAlpha(opacity),
        color.backgroundEnd.BlendAlpha(opacity)
    };
    FillGradientPolygon(NN_ARRAY_SIZE(points), points, colors);
    DrawRectangle(rect, color.border.BlendAlpha(opacity), 1.0f);

    nns::sgx::ScopedFontContextSaver saver;

    // フォントサイズ設定
    nns::sgx::SetTextSize(m_TextSize);

    // 文字色設定
    nns::sgx::SetTextColor(color.text.BlendAlpha(opacity));

    // テキスト描画
    Size textSize;
    nns::sgx::GetTextDrawSize(&textSize, m_Text);

    float textX = rect.x + (rect.width  - textSize.width)  / 2.0f;
    float textY = rect.y + (rect.height - textSize.height) / 2.0f + (isPressed ? 1 : 0);
    nns::sgx::DrawText(textX, textY, m_Text);
}

void SimpleButton::InitializeDefaultColorScheme() NN_NOEXCEPT
{
    if (g_IsDefaultColorInitialized)
    {
        return;
    }

    g_DefaultNormalColor   = { Colors::White(),  Colors::DarkGray(), Colors::DimGray(), Colors::White() };
    g_DefaultDisabledColor = { Colors::Silver(), Colors::Smoke(),    Colors::Smoke(),   Colors::DarkGray() };
    g_DefaultPressColor    = { Colors::Silver(), Colors::Shadow(),   Colors::Smoke(),   Colors::Silver() };
    g_IsDefaultColorInitialized = true;
}

void SimpleButton::SetupColorScheme() NN_NOEXCEPT
{
    InitializeDefaultColorScheme();
    m_NormalColor   = g_DefaultNormalColor;
    m_DisabledColor = g_DefaultDisabledColor;
    m_PressColor    = g_DefaultPressColor;
}

}}}  // nns::sgx::gui
