﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_GuiCursor.h"

namespace nns { namespace sgx { namespace gui {

const int AnimationFrameCount = 60;

const int DecideAnimationFrameCount = 15;

bool Cursor::g_IsDefaultColorInitialized = false;

// 初期化時に設定するので初期値は入れない
CursorColorScheme Cursor::g_DefaultColor = {};

Cursor::Cursor() NN_NOEXCEPT
    : m_State(CursorState_Init)
    , m_Color()
    , m_LineWeight(5.0f)
    , m_Duration(0)
    , m_DecideDuration(0)
    , m_DecideEffectRect()
{
}

void Cursor::StartDecideEffect(const Point2D& offset) NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    m_DecideDuration = DecideAnimationFrameCount;

    GetRenderPosition(&m_DecideEffectRect.position);
    m_DecideEffectRect.position += offset;
    GetSize(&m_DecideEffectRect.size);
}

void Cursor::StartDecideEffect(const DisplayObject& obj, const Point2D& offset) NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    m_DecideDuration = DecideAnimationFrameCount;

    obj.GetRenderPosition(&m_DecideEffectRect.position);
    m_DecideEffectRect.position += offset;
    obj.GetSize(&m_DecideEffectRect.size);
}

void Cursor::StopDecideEffect() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    m_DecideDuration = 0;
}

void Cursor::Update() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    InitializeIfNeeded();

    m_DecideDuration = std::max(m_DecideDuration - 1, 0);

    if (!IsVisible())
    {
        return;
    }

    if (IsEnabled())
    {
        m_Duration = (m_Duration + 1) % AnimationFrameCount;
    }
    else
    {
        m_Duration = 0;
    }
}

void Cursor::Render() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    InitializeIfNeeded();

    if (!IsVisible())
    {
        return;
    }

    Rectangle rect;
    GetRenderPosition(&rect.position);
    GetSize(&rect.size);

    auto opacity = GetDisplayOpacity();

    if (IsEnabled())
    {
        // 有効状態ではアニメーション表示
        const int halfCount = AnimationFrameCount / 2;
        float blendRate;
        if (m_Duration < halfCount)
        {
            blendRate = 1.0f * m_Duration / halfCount;
        }
        else
        {
            blendRate = 1.0f * (AnimationFrameCount - m_Duration) / halfCount;
        }

        Color color = Color::Lerp(m_Color.activeBegin, m_Color.activeEnd, blendRate);
        DrawRectangle(rect, color.BlendAlpha(opacity), m_LineWeight);
    }
    else
    {
        DrawRectangle(rect, m_Color.inactive.BlendAlpha(opacity), m_LineWeight);
    }
}

void Cursor::RenderDecideEffect(const Point2D& originOffset) NN_NOEXCEPT
{
    Point2D originPosition;
    GetRenderPosition(&originPosition);
    SetPosition(originPosition + originOffset);

    auto opacity = GetDisplayOpacity();

    // 決定アニメを再生
    if (m_DecideDuration > 0)
    {
        auto effectRect = m_DecideEffectRect;
        float offset = (DecideAnimationFrameCount - m_DecideDuration) * 16.0f / DecideAnimationFrameCount;
        effectRect.x      -= offset;
        effectRect.y      -= offset;
        effectRect.width  += offset * 2;
        effectRect.height += offset * 2;

        int alpha = 255 * m_DecideDuration / DecideAnimationFrameCount;
        alpha = alpha * opacity / 255;

        Color color = Color::Lerp(m_Color.activeBegin, m_Color.activeEnd, 0.5f);
        DrawRectangle(effectRect, color.BlendAlpha(alpha), m_LineWeight);
    }

    SetPosition(originPosition);
}

void Cursor::InitializeDefaultColorScheme() NN_NOEXCEPT
{
    if (g_IsDefaultColorInitialized)
    {
        return;
    }

    g_DefaultColor = { Colors::RoyalBlue(), Colors::PaleGreen(), Colors::Gray() };
    g_IsDefaultColorInitialized = true;
}

void Cursor::SetupColorScheme() NN_NOEXCEPT
{
    InitializeDefaultColorScheme();
    m_Color = g_DefaultColor;
}

}}}  // nns::sgx::gui
