﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../SimpleGfx_Types.h"

// ボタン押下判定をリリース時にする
//#define NNS_SGX_GUI_PUSH_ON_RELEASE

// 呼び出しを排他する
//#define NNS_SGX_GUI_ENABLE_LOCK

// デバッグ描画を有効にする
//#define NNS_SGX_GUI_ENABLE_DEBUG_DRAW

#if defined(NNS_SGX_GUI_ENABLE_LOCK)
#define NNS_SGX_GUI_SCOPED_LOCK             std::lock_guard<decltype(m_Mutex)> lock(m_Mutex)
#define NNS_SGX_GUI_SCOPED_LOCK_P(mutex)    std::lock_guard<decltype(mutex)> lock(mutex)
#else
#define NNS_SGX_GUI_SCOPED_LOCK
#define NNS_SGX_GUI_SCOPED_LOCK_P(mutex)
#endif

// クライアント領域の枠を描画 (デバッグ描画有効時のみ)
#if defined(NNS_SGX_GUI_ENABLE_DEBUG_DRAW)
#define NNS_SGX_GUI_DRAW_CLIENT_AREA_DEBUG(rect) ::nns::sgx::DrawRectangle((rect), ::nns::sgx::Colors::Red().BlendAlpha(192), 1.0f)
#else
#define NNS_SGX_GUI_DRAW_CLIENT_AREA_DEBUG(rect) static_cast<void>(0)
#endif

// ヒット領域を描画 (デバッグ描画有効時のみ)
#if defined(NNS_SGX_GUI_ENABLE_DEBUG_DRAW)
#define NNS_SGX_GUI_DRAW_HIT_AREA_DEBUG(rect)    ::nns::sgx::FillRectangle((rect), ::nns::sgx::Colors::LimeGreen().BlendAlpha(128))
#else
#define NNS_SGX_GUI_DRAW_HIT_AREA_DEBUG(rect)   static_cast<void>(0)
#endif

/**
 * @brief   GUI イベントハンドラの仮引数を宣言します。
 */
#define NNS_SGX_GUI_EVENT_HANDLER_ARGS(pSender, arg) \
    ::nns::sgx::gui::DisplayObject* (pSender), uintptr_t (arg)

/**
 * @brief   GUI の無名イベントハンドラを定義します。
 */
#define NNS_SGX_GUI_EVENT_HANDLER(pSender, arg) \
    [](::nns::sgx::gui::DisplayObject* (pSender), uintptr_t (arg)) NN_NOEXCEPT

namespace nns { namespace sgx { namespace gui {

class DisplayObject;

/**
 * @brief   GUI のイベント発生時に呼ばれるハンドラの型です。
 */
typedef void(*GuiEventHandler)(DisplayObject* pSender, uintptr_t argument);

/**
 * @brief   方向を示す型です。
 */
enum class Direction
{
    None,       //!< なし
    Up,         //!< 上
    Down,       //!< 下
    Left,       //!< 左
    Right       //!< 右
};

/**
 * @brief   水平整列方向を示す型です。
 */
enum class HorizontalAlignment
{
    Left,       //!< 左
    Center,     //!< 中央
    Right       //!< 右
};

/**
 * @brief   垂直整列方向を示す型です。
 */
enum class VerticalAlignment
{
    Top,        //!< 上
    Middle,     //!< 中央
    Bottom      //!< 下
};

/**
 * @brief   GUI 操作時に呼ばれるハンドラと、渡される引数の組です。
 */
struct GuiEventHandlerType
{
    GuiEventHandler handler;    //!< イベントハンドラ
    uintptr_t       argument;   //!< 引数

    /**
     * @brief   イベントハンドラが登録されているか判定します。
     */
    bool IsValid() const NN_NOEXCEPT
    {
        return handler != nullptr;
    }

    /**
     * @brief   イベントハンドラの登録を解除します。
     */
    void Invalidate() NN_NOEXCEPT
    {
        handler  = nullptr;
        argument = 0;
    }

    /**
     * @brief   登録されているイベントハンドラを実行します。
     *
     * @param[in]   pSender     ハンドラの呼び出し元オブジェクト
     */
    void Invoke(DisplayObject* pSender) NN_NOEXCEPT
    {
        if (IsValid())
        {
            handler(pSender, argument);
        }
    }

    /**
     * @brief   登録されているイベントハンドラを実行します。イベントハンドラの登録は解除されます。
     *
     * @param[in]   pSender     ハンドラの呼び出し元オブジェクト
     */
    void InvokeAndClear(DisplayObject* pSender) NN_NOEXCEPT
    {
        if (IsValid())
        {
            auto savedHandler = handler;
            auto savedArg     = argument;
            Invalidate();
            savedHandler(pSender, savedArg);
        }
    }
};

}}}  // nns::sgx::gui
