﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_GuiCheckBox.h"
#include "SimpleGfx_GuiUtil.h"
#include "../../input/Input.h"

namespace nns { namespace sgx { namespace gui {

namespace
{

// チェックボックスの枠
const char BoxText[] = "□";

// チェックマーク
const char CheckText[] = "\uE14B";

}  // anonymous namespace

CheckBox::CheckBox() NN_NOEXCEPT
    : m_IsKeyFocusable(true)
    , m_IsChecked(false)
    , m_ButtonMaskForPress(nn::hid::NpadButton::A::Mask)
    , m_Text()
    , m_TextSize(40.0f)
    , m_State(CheckBoxState::Idle)
    , m_HandlerForPushEvent()
{
    SetDecideEffectEnabled(true);
}

CheckBoxAppearanceSet CheckBox::GetAppearance() const NN_NOEXCEPT
{
    CheckBoxAppearanceSet appearance;
    appearance.Reset();
    if (IsChecked())
    {
        appearance.Set(CheckBoxAppearance::Checked::Index);
    }

    if (!IsEnabled())
    {
        appearance.Set(CheckBoxAppearance::Enabled::Index);
    }

    if (IsFocused())
    {
        appearance.Set(CheckBoxAppearance::Focused::Index);
    }

    if (IsPressed())
    {
        appearance.Set(CheckBoxAppearance::Pressed::Index);
    }

    return appearance;
}

void CheckBox::SetText(const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);
    NNS_SGX_GUI_SCOPED_LOCK;

    NN_UTIL_SCOPE_EXIT
    {
        UpdateAutoSize();
    };

    // UTF8 -> UTF32 変換
    {
        int lengthUtf32;
        auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&lengthUtf32, text);
        if (result != nn::util::CharacterEncodingResult_Success ||
            lengthUtf32 >= CheckBoxTextLengthMax)
        {
            // 変換不能
            m_Text[0] = 0;
            return;
        }
    }

    auto result = nn::util::ConvertStringUtf8ToUtf32(m_Text, CheckBoxTextLengthMax, text);
    if (result != nn::util::CharacterEncodingResult_Success)
    {
        // 変換失敗
        m_Text[0] = 0;
        return;
    }
}

void CheckBox::SetText(const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);
    NNS_SGX_GUI_SCOPED_LOCK;

    int length = 0;
    for (; length < CheckBoxTextLengthMax - 1 && text[length] != 0; length++)
    {
        m_Text[length] = text[length];
    }

    m_Text[length] = 0;
    UpdateAutoSize();
}

void CheckBox::PerformPress() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsVisible() || !IsEnabled())
    {
        // 押せない
        return;
    }

    SetState(CheckBoxState::Idle);
    m_HandlerForPushEvent.Invoke(this);
}

void CheckBox::Update() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    DisplayObject::Update();

    if (!IsVisible() || !IsEnabled())
    {
        SetState(CheckBoxState::Idle);
        return;
    }
    else if (!IsFocused() && GetState() == CheckBoxState::PressByKey)
    {
        SetState(CheckBoxState::Idle);
        return;
    }
}

bool CheckBox::UpdateKeyInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsFocused())
    {
        return false;
    }

    auto onPush = [](CheckBox* pSelf)
    {
        pSelf->SetChecked(!pSelf->IsChecked());
        PlaySystemSe(SystemSe::Decide);
        //pSelf->m_HandlerForPushEvent.Invoke(pSelf);
        pSelf->ReservePostUpdateEvent(pSelf->m_HandlerForPushEvent);
    };

    // パッド操作による押下判定
    auto controllers = input::GetNpadControllers();
    controllers.push_back(input::NpadManager::GetInstance()->GetDummyController());

    auto state = GetState();
    for (auto pController : controllers)
    {
        if (pController->IsAnyTriggered(m_ButtonMaskForPress))
        {
            if (IsEnabled())
            {
                // ボタン押下開始
                SetState(CheckBoxState::PressByKey);

#if defined(NNS_SGX_GUI_PUSH_ON_RELEASE)
                PlaySystemSe(SystemSe::Press);
#else
                onPush(this);
                return true;
#endif  // if defined(NNS_SGX_GUI_PUSH_ON_RELEASE)
            }
            else
            {
                // 禁止状態
                PlaySystemSe(SystemSe::Buzzer);
            }
            return false;
        }
        else if (pController->IsAnyPressed(m_ButtonMaskForPress))
        {
            // 押下継続
            // カーソル移動によるキャンセルを受け付けるため、入力は処理されていないものとして振る舞う
            return false;
        }
        else if (!pController->IsAnyPressed(m_ButtonMaskForPress) &&
            state == CheckBoxState::PressByKey)
        {
            SetState(CheckBoxState::Idle);

#if defined(NNS_SGX_GUI_PUSH_ON_RELEASE)
            // 押下状態からリリースされたらボタン押し判定
            onPush(this);
            return true;
#else
            return false;
#endif  // if defined(NNS_SGX_GUI_PUSH_ON_RELEASE)
        }
    }

    return false;
}

bool CheckBox::UpdateTouchInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    auto* tp = input::TouchPanel::GetInstance();

    auto rect  = GetHitArea().ToFloat4();
    auto state = GetState();
    if (tp->IsTouchTriggered(rect))
    {
        if (state == CheckBoxState::Idle)
        {
            // ボタン押下開始
            SetState(CheckBoxState::PressByTouch);
            PlaySystemSe(SystemSe::Press);
        }
    }
    else if (!tp->IsTouched(rect) &&
        state == CheckBoxState::PressByTouch)
    {
        SetState(CheckBoxState::Idle);

        // 押下状態からリリースされたらボタン押し判定
        if (tp->IsTouchReleased(rect))
        {
            SetChecked(!IsChecked());
            PlaySystemSe(SystemSe::Decide);
            //m_HandlerForPushEvent.Invoke(this);
            ReservePostUpdateEvent(m_HandlerForPushEvent);
            return true;
        }
    }

    return false;
}

void CheckBox::CancelTouchInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    SetState(CheckBoxState::Idle);
}

void CheckBox::Render() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsVisible())
    {
        return;
    }

    NNS_SGX_GUI_DRAW_HIT_AREA_DEBUG(GetHitArea());

    nns::sgx::ScopedFontContextSaver saver;

    auto opacity = GetDisplayOpacity();

    const auto& baseColor  = Colors::Black().BlendAlpha(opacity);
    const auto& checkColor = Colors::RoyalBlue().BlendAlpha(opacity);

    Rectangle rect;
    GetRenderPosition(&rect.position);
    GetSize(&rect.size);

    Size checkSize;
    nns::sgx::GetTextDrawSize(&checkSize, BoxText);

    nns::sgx::SetTextSize(m_TextSize);
    nns::sgx::SetTextColor(baseColor);

    // テキスト描画
    Size textSize;
    nns::sgx::GetTextDrawSize(&textSize, m_Text);

    float textX = rect.x + 8.0f;
    float textY = rect.y + 8.0f;
    nns::sgx::DrawText(textX, textY, BoxText);
    nns::sgx::DrawText(textX + checkSize.width + 2.0f, textY, m_Text);

    // チェックマーク
    if (IsChecked())
    {
        float size = m_TextSize * 1.3f;
        nns::sgx::SetTextSize(size);
        nns::sgx::SetTextColor(checkColor);
        nns::sgx::DrawText(
            textX +  1.0f * size / 40.0f,
            textY - 11.0f * size / 40.0f,
            CheckText
        );
    }
}

void CheckBox::UpdateAutoSize() NN_NOEXCEPT
{
    nns::sgx::ScopedFontContextSaver saver;

    nns::sgx::SetTextSize(m_TextSize);

    Size textSize;
    nns::sgx::GetTextDrawSize(&textSize, m_Text);

    Size checkSize;
    nns::sgx::GetTextDrawSize(&checkSize, BoxText);

    SetSize(
        textSize.width + checkSize.width + 32.0f,
        std::max(textSize.height, checkSize.height) + 20.0f
    );
}

}}}  // nns::sgx::gui
