﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_GuiButton.h"
#include "SimpleGfx_GuiUtil.h"
#include "../../input/Input.h"

namespace nns { namespace sgx { namespace gui {

ButtonSkin Button::g_DefaultSkin;

void Button::LoadDefaultSkin() NN_NOEXCEPT
{
    if (g_DefaultSkin.m_BaseImage.IsValid())
    {
        return;
    }

    g_DefaultSkin.SetDefault();
    nns::sgx::LoadImage(&g_DefaultSkin.m_BaseImage, "rom:/NormalButton.png");
    g_DefaultSkin.m_NormalTextColor = nns::sgx::Colors::Shadow();
}

Button::Button() NN_NOEXCEPT
    : ButtonBase()
    , m_pSkin(&g_DefaultSkin)
    , m_TextAlignment(nns::sgx::TextAlignment::Center)
    , m_Text()
    , m_TextSize(24.0f)
    , m_HandlerForRenderContents()
{
}

Rectangle Button::GetHitArea() const NN_NOEXCEPT
{
    auto rect = ButtonBase::GetHitArea();
    if (m_pSkin->m_BaseImage.IsValid() && !rect.size.IsEmpty())
    {
        // スキンの補正を適用
        rect.x      += m_pSkin->m_BaseOffset.x;
        rect.y      += m_pSkin->m_BaseOffset.y;
        rect.width  += m_pSkin->m_BaseSizeOffset.width;
        rect.height += m_pSkin->m_BaseSizeOffset.height;
    }

    return rect;
}

Size Button::GetClientAreaSize() const NN_NOEXCEPT
{
    if (!m_pSkin->m_BaseImage.IsValid())
    {
        return Size();
    }

    const auto& padding = m_pSkin->m_ContentsPadding;
    Size size;
    size.width  = GetWidth()  - (padding.left + padding.right);
    size.height = GetHeight() - (padding.top  + padding.bottom);

    return size;
}

Rectangle Button::GetClientArea() const NN_NOEXCEPT
{
    if (!m_pSkin->m_BaseImage.IsValid())
    {
        return Rectangle();
    }

    const auto& padding = m_pSkin->m_ContentsPadding;
    Rectangle rect;
    rect.x    = GetX() + padding.left;
    rect.y    = GetY() + padding.top;
    rect.size = GetClientAreaSize();

    return rect;
}

void Button::SetText(const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);
    NNS_SGX_GUI_SCOPED_LOCK;

    // UTF8 -> UTF32 変換
    {
        int lengthUtf32;
        auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&lengthUtf32, text);
        if (result != nn::util::CharacterEncodingResult_Success ||
            lengthUtf32 >= ButtonTextLengthMax)
        {
            // 変換不能
            m_Text[0] = 0;
            return;
        }
    }

    auto result = nn::util::ConvertStringUtf8ToUtf32(m_Text, ButtonTextLengthMax, text);
    if (result != nn::util::CharacterEncodingResult_Success)
    {
        // 変換失敗
        m_Text[0] = 0;
        return;
    }
}

void Button::SetText(const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);
    NNS_SGX_GUI_SCOPED_LOCK;

    int length = 0;
    for (; length < ButtonTextLengthMax - 1 && text[length] != 0; length++)
    {
        m_Text[length] = text[length];
    }

    m_Text[length] = 0;
}

void Button::Render() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsVisible())
    {
        return;
    }

    auto opacity = GetDisplayOpacity();

    // ベースの描画
    if (m_pSkin->m_BaseImage.IsValid())
    {
        float prevOpacity = nns::sgx::GetImageOpacity();
        nns::sgx::SetImageOpacity(opacity / 255.0f);
        DrawImageButton(
            m_pSkin->m_BaseImage,
            GetX()      + m_pSkin->m_BaseOffset.x,
            GetY()      + m_pSkin->m_BaseOffset.y,
            GetWidth()  + m_pSkin->m_BaseSizeOffset.width,
            GetHeight() + m_pSkin->m_BaseSizeOffset.height,
            GetAppearance());
        nns::sgx::SetImageOpacity(prevOpacity);
    }

    // コンテンツ領域の描画開始
    {
        auto rect = GetClientArea();
        NNS_SGX_GUI_DRAW_CLIENT_AREA_DEBUG(rect);
        nns::sgx::ApplyRenderArea(rect);
    }

    // テキストの描画
    if (m_Text[0] != 0)
    {
        nns::sgx::ScopedFontContextSaver saver;

        // フォントサイズ設定
        nns::sgx::SetTextSize(m_TextSize);

        // 文字色設定
        nns::sgx::SetTextColor(m_pSkin->m_NormalTextColor.BlendAlpha(opacity));

        auto size = GetClientAreaSize();
        nns::sgx::DrawText(0, 0, size.width, size.height, m_TextAlignment, m_Text);
    }

    RenderContents();
    m_HandlerForRenderContents.Invoke(this);

    // コンテンツ領域の描画終了
    nns::sgx::RestoreRenderArea();
}

}}}  // nns::sgx::gui
