﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Macro.h>
#include <nn/os.h>

#include "SceneBase.h"
#include "../sgx/SimpleGfx.h"

namespace nns { namespace hid { namespace scene {

/**
 * @brief   シーンの一覧を扱う型です。
 */
typedef std::vector<SceneBase*> SceneListType;

/**
 * @brief   マネージャによってシーンが終了させられたことを通知するためのコールバック関数です。
 */
typedef void(*TerminateCallback)(SceneBase* pTerminatedScene);

/**
 * @brief   シーン遷移を管理するクラスです。
 */
class SceneManager final
{
    NN_DISALLOW_COPY(SceneManager);
    NN_DISALLOW_MOVE(SceneManager);

public:
    /**
     * @brief   マネージャを初期化します。
     */
    static void Initialize() NN_NOEXCEPT;

    /**
     * @brief   マネージャを終了します。
     */
    static void Finalize() NN_NOEXCEPT;

    /**
     * @brief   マネージャによってシーンが終了させられたことを通知するコールバックを登録します。
     */
    static void RegisterTerminateCallback(TerminateCallback callback) NN_NOEXCEPT
    {
        g_TerminateCallback = callback;
    }

    /**
     * @brief   終了要求を受けたか判定します。
     */
    static bool IsExitRequested() NN_NOEXCEPT;

    /**
     * @brief   アクティブなシーンを取得します。
     */
    static SceneBase* GetActiveScene() NN_NOEXCEPT;

    /**
     * @brief   フレームの更新と描画を行います。
     */
    static void StepFrame() NN_NOEXCEPT
    {
        Update();
        Render();
    }

    /**
     * @brief   アクティブなシーンを更新します。
     */
    static void Update() NN_NOEXCEPT;

    /**
     * @brief   アクティブなシーンを描画します。
     */
    static void Render() NN_NOEXCEPT;

    /**
     * @brief   シーンの背景を描画します。
     */
    static void RenderBackground() NN_NOEXCEPT;

    /**
     * @brief   新たなシーンをスタックに積みつつ、指定されたシーンに遷移します。
     *
     * @param[in]   pNewScene       遷移先のシーン
     */
    static void Push(SceneBase* pNewScene) NN_NOEXCEPT;

    /**
     * @brief   現在スタックに積まれているシーンを全て終了し、指定されたシーンに遷移します。
     */
    static void Goto(SceneBase* pNewScene) NN_NOEXCEPT;

    /**
     * @brief   現在のシーンを終了し、一つ前のシーンに戻ります。
     *
     * @return  終了したシーン
     */
    static SceneBase* Pop() NN_NOEXCEPT;

    /**
     * @brief   指定したシーンに、間のシーンを終了しつつ戻ります。
     */
    static void BackTo(SceneBase* pTargetScene) NN_NOEXCEPT;

    /**
     * @brief   指定したシーンに、間のシーンを終了しつつ戻ります。
     */
    static void BackTo(const char* sceneName) NN_NOEXCEPT;

private:
    typedef void(*SceneProcessFunc)(SceneBase* pScene);

private:
    SceneManager() NN_NOEXCEPT {}

    /**
     * @brief   シーンスタックが空かどうか判定します。
     */
    static bool IsEmpty() NN_NOEXCEPT;

    /**
     * @brief   アクティブシーンに対して指定処理を実行します。
     */
    static SceneBase* ProcessActiveScene(SceneProcessFunc func) NN_NOEXCEPT;

    /**
     * @brief   シーンを終了します。
     */
    static void TerminateScene(SceneBase* pScene) NN_NOEXCEPT;

    /**
     * @brief   シーン数が指定した数になるまで、末尾から順に終了しつつスタックから取り除きます。
     */
    static void TerminateTo(int count) NN_NOEXCEPT;

    /**
     * @brief   シーンの終了判定を行います。
     */
    static void CheckSceneTerminated() NN_NOEXCEPT;

    /**
     * @brief   シーンの描画を停止します。
     */
    static void FreezeRender() NN_NOEXCEPT;

    /**
     * @brief   シーンの描画を再開します。
     */
    static void ResumeRender() NN_NOEXCEPT;

    /**
     * @brief   シーン遷移エフェクトを開始します。
     */
    static void StartTransitionEffect() NN_NOEXCEPT;

    /**
     * @brief   シーン遷移エフェクトを終了します。
     */
    static void EndTransitionEffect() NN_NOEXCEPT;

    /**
     * @brief   シーン遷移エフェクト中か判定します。
     */
    static bool IsTransiting() NN_NOEXCEPT;

    /**
     * @brief   シーン離脱エフェクト中か判定します。
     */
    static bool IsOutingScene() NN_NOEXCEPT;

    /**
     * @brief   シーン遷移エフェクトを更新します。
     */
    static void UpdateTransitionEffect() NN_NOEXCEPT;

private:
    static nn::os::Mutex        g_Mutex;
    static bool                 g_IsInitialized;
    static bool                 g_IsRenderFrozen;
    static SceneListType        g_SceneList;
    static SceneListType        g_TerminatedSceneList;
    static TerminateCallback    g_TerminateCallback;

    static nns::sgx::ImageData  g_BackgroundImage;
    static nns::sgx::ImageData  g_TransitionImage;
    static nns::sgx::Color      g_TransitionColor;
    static int                  g_TransitionFrameCount;
    static int                  g_TransitionDuration;
};

}}}  // nns::hid::scene
