﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <new>

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/oe/oe_HomeButtonControl.h>
#include <nn/hid/debug/hid_FirmwareUpdate.h>
#include <nn/util/util_BytePtr.h>

#include "../sgx/SimpleGfx.h"
#include "../sgx/gui/SimpleGfx_Gui.h"
#include "../sgx/gui/SimpleGfx_GuiUtil.h"
#include "../input/Input.h"
#include "../util/StringTable.h"
#include "../util/util.h"
#include "../util/FirmwareImageEnumerator.h"

#include "FlashWriter.h"

NN_DISABLE_WARNING_ARRAY_DEFAULT_INITIALIZATION_IN_CONSTRUCTOR

namespace nns { namespace hid { namespace scene {

namespace
{

const float ImageButtonWidth  = 640.0f;
const float ImageButtonHeight = 76.0f;

// リストのラベル背景色
const nns::sgx::Color LabelBackgroundColor = nns::sgx::Color(0xF8, 0xF8, 0xF8);

// デバイスタイプ無視切り替え用のキーアサイン
const nn::hid::NpadButtonSet ForceModeButtonMask = nn::hid::NpadButton::StickL::Mask | nn::hid::NpadButton::StickR::Mask;

// 詳細表示切替用のキーアサイン
const nn::hid::NpadButtonSet DetailButtonMask = nn::hid::NpadButton::Plus::Mask | nn::hid::NpadButton::Minus::Mask;

}  // anonymous

FlashWriter::FlashWriter() NN_NOEXCEPT
    : SceneBase("FlashWriter")
    , m_Title()
    , m_KeyHelp()
    , m_FwImageListContainer()
    , m_IsInfoVisible(false)
    , m_IconImage()
    , m_ButtonSkin()
    , m_FwImageButtonCount(0)
    , m_pFwImageButtons(nullptr)
    , m_ForceModeCheckBox()
    , m_InfoView()
    , m_PadView()
    , m_ImageListLabel()
    , m_PadListLabel()
    , m_Dialog()
    , m_ProgressDialog()
{
}

void FlashWriter::Initialize() NN_NOEXCEPT
{
    SceneBase::Initialize();

    // アイコンの読み込み
    NN_ABORT_UNLESS_EQUAL(
        nns::sgx::LoadImage(&m_IconImage, "rom:/MainIcon.bmp"),
        nns::sgx::ResultCode::Success
    );

    auto& enumerator = *util::FirmwareImageEnumerator::GetInstance();
    enumerator.Mount();

    auto& str = nns::hid::util::StringTable::GetInstance();

    m_Title.SetText(str.Get("FlashWriter_Title"));
    m_Title.SetIconImage(&m_IconImage);
    AddChild(&m_Title);

    AddChild(&m_KeyHelp);

    nns::sgx::Size canvasSize;
    nns::sgx::GetCanvasSize(&canvasSize);

    // ファームウェアリスト用のコンテナ
    m_FwImageListContainer.SetPosition(48, 136);
    m_FwImageListContainer.SetSize(ImageButtonWidth + 24, canvasSize.height - 220);
    m_FwImageListContainer.SetDirection(nns::sgx::gui::LayoutDirection::Vertical);
    m_FwImageListContainer.SetAutoWrap(false);
    m_FwImageListContainer.SetItemGap({ { 48, 0 } });
    m_FwImageListContainer.SetVerticalLoopEnabled(true);
    AddChild(&m_FwImageListContainer);

    // ボタンスキンの読み込み
    NN_ABORT_UNLESS_EQUAL(
        m_ButtonSkin.Load("rom:/FlatMenuButton.json"),
        nns::sgx::ResultCode::Success
    );

    // ボタンの作成
    CreateImageButtons();

    // チェックボックスの初期化
    m_ForceModeCheckBox.SetSize(520, 64);
    m_ForceModeCheckBox.SetPosition(748, canvasSize.height - m_ForceModeCheckBox.GetHeight() - 64);
    m_ForceModeCheckBox.SetText(str.Get("FlashWriter_ForceModeCheckBox"));
    m_ForceModeCheckBox.SetTextSize(24);
    m_ForceModeCheckBox.SetButtonMaskForPress(util::DecideButtonMask);
    m_ForceModeCheckBox.SetChecked(util::Config::GetInstance()->IsForceUpdateEnabled());
    m_ForceModeCheckBox.SetKeyFocusable(false);
    m_ForceModeCheckBox.SetPushEventHandler(NNS_SGX_GUI_EVENT_HANDLER(pSender, arg)
    {
        NN_UNUSED(pSender);

        auto* pConfig = util::Config::GetInstance();
        pConfig->SetForceUpdateEnabled(!pConfig->IsForceUpdateEnabled());

        auto& scene = *reinterpret_cast<FlashWriter*>(arg);
        scene.m_ForceModeCheckBox.SetChecked(pConfig->IsForceUpdateEnabled());
    }, reinterpret_cast<uintptr_t>(this));
    AddChild(&m_ForceModeCheckBox);

    // FW イメージ情報の初期化
    m_InfoView.SetPosition(732, 96);
    m_InfoView.SetSize(
        canvasSize.width - m_InfoView.GetX(),
        canvasSize.height - m_ForceModeCheckBox.GetHeight() - 188
    );
    m_InfoView.SetZ(1000);
    m_InfoView.SetOpacity(0);
    AddChild(&m_InfoView);

    // コントローラー一覧の初期化
    m_PadView.Initialize();
    m_PadView.SetPosition(748, 136);
    m_PadView.SetSize(
        520,
        canvasSize.height - m_ForceModeCheckBox.GetHeight() - 228
    );
    m_PadView.SetDecideCallback(PushPadButtonHandler, reinterpret_cast<uintptr_t>(this));
    m_PadView.SetVerticalLoopEnabled(true);
    AddChild(&m_PadView);

    // ラベルの初期化
    m_ImageListLabel.SetPosition(72, 96);
    m_ImageListLabel.SetSize(m_FwImageListContainer.GetWidth() - 56, 24);
    m_ImageListLabel.SetPadding(16, 0);
    m_ImageListLabel.SetBackgroundColor(LabelBackgroundColor);
    m_ImageListLabel.SetText(str.Get("FlashWriter_ImageListLabel"));
    m_ImageListLabel.SetTextSize(18);
    m_ImageListLabel.SetColor(nns::sgx::Colors::Gray());
    AddChild(&m_ImageListLabel);

    m_PadListLabel.SetPosition(756, 96);
    m_PadListLabel.SetSize(m_PadView.GetWidth() - 56, 24);
    m_PadListLabel.SetPadding(m_ImageListLabel.GetPadding());
    m_PadListLabel.SetBackgroundColor(LabelBackgroundColor);
    m_PadListLabel.SetText(str.Get("FlashWriter_PadListLabel"));
    m_PadListLabel.SetTextSize(m_ImageListLabel.GetTextSize());
    m_PadListLabel.SetColor(nns::sgx::Colors::Gray());
    AddChild(&m_PadListLabel);

    // ダイアログの初期化
    m_Dialog.Initialize();
    m_Dialog.SetSize(720, 320);
    m_Dialog.SetButtonSkin(m_ButtonSkin);
    m_Dialog.SetButtonCount(1);
    m_Dialog.SetButtonMaskForPress(0, util::DecideButtonMask);
    m_Dialog.SetButtonMaskForCancel(util::CancelButtonMask);
    AddChild(&m_Dialog);

    m_ProgressDialog.Initialize();
    m_ProgressDialog.SetButtonSkin(m_ButtonSkin);
    AddChild(&m_ProgressDialog);

    // 初期フォーカスの設定
    SetFocusedChild(&m_FwImageListContainer);

    UpdateKeyHelp();

}  // NOLINT(readability/fn_size)

void FlashWriter::Terminate() NN_NOEXCEPT
{
    auto& enumerator = *util::FirmwareImageEnumerator::GetInstance();
    enumerator.Unmount();

    if (m_pFwImageButtons != nullptr)
    {
        delete[] m_pFwImageButtons;
    }

    // 画像の破棄
    nns::sgx::DestroyImage(&m_IconImage);

    // FW イメージバッファの破棄
    if (m_pImageBuffer != nullptr)
    {
        delete[] m_pImageBuffer;
        m_pImageBuffer = nullptr;
    }

    // 念のため HOME 禁止を解除
    nn::oe::EndBlockingHomeButton();

    SceneBase::Terminate();
}

void FlashWriter::Update() NN_NOEXCEPT
{
    auto& enumerator = *util::FirmwareImageEnumerator::GetInstance();
    if (enumerator.TryAutoMount())
    {
        CreateImageButtons();

        // Pad 選択中に自動マウントされたら、念のためファイル一覧に戻す
        if (m_PadView.IsFocused())
        {
            SetFocusedChild(&m_FwImageListContainer);
        }
        return;
    }

    if (m_FwImageListContainer.IsFocused())
    {
        auto& enumerator = *util::FirmwareImageEnumerator::GetInstance();
        if (enumerator.GetFileCount() > 0)
        {
            int activeIndex = GetActiveImageIndex();
            const auto& image = enumerator.GetImage(activeIndex);
            m_InfoView.SetImage(&image);
        }

#if 0
        // B ボタンで戻る
        if (input::IsAnyTriggered<nn::hid::NpadButton::B>())
        {
            SceneManager::Pop();
            return;
        }
#endif

        // +/- ボタンで詳細切り替え
        if (input::IsAnyTriggered(DetailButtonMask))
        {
            m_IsInfoVisible = !m_IsInfoVisible;
            if (m_IsInfoVisible)
            {
                m_InfoView.Appear();
            }
            else
            {
                m_InfoView.Disappear();
            }
            UpdateKeyHelp();
        }

        UpdateForceModeChange();
    }
    else if (m_PadView.IsFocused())
    {
        const auto cancelMask =
#if defined(NNS_HID_ENABLE_CURSOR_DECISION)
            util::CancelButtonMask |
            nn::hid::NpadButton::Left::Mask |
            nn::hid::NpadButton::StickLLeft::Mask |
            nn::hid::NpadButton::StickRLeft::Mask;
#else
            util::CancelButtonMask;
#endif  // if defined(NNS_HID_ENABLE_CURSOR_DECISION)

        // B/ZL ボタンで戻る
        if (input::IsAnyTriggered(cancelMask))
        {
            SetFocusedChild(&m_FwImageListContainer);
            UpdateKeyHelp();
            return;
        }

        UpdateForceModeChange();
    }
    else if (m_ProgressDialog.IsFocused())
    {
        // 完了したらメッセージに切り替え
        if (m_ProgressDialog.IsFinished())
        {
            nn::oe::EndBlockingHomeButton();
            m_ProgressDialog.Disappear();

            auto& str = nns::hid::util::StringTable::GetInstance();
            ShowMessage(str.Get(m_ProgressDialog.IsSucceeded() ? "FlashWriter_UpdateSuccess" : "FlashWriter_UpdateFailed"));
        }
    }

    SceneBase::Update();
}

void FlashWriter::PushImageButtonHandler(NNS_SGX_GUI_EVENT_HANDLER_ARGS(pSender, arg)) NN_NOEXCEPT
{
    NN_UNUSED(pSender);

    auto& scene = *reinterpret_cast<FlashWriter*>(arg);
    if (scene.m_PadView.IsEmpty())
    {
        NN_LOG("No controllers are connected.\n");
        return;
    }

    auto& enumerator = *util::FirmwareImageEnumerator::GetInstance();
    if (enumerator.IsEmpty())
    {
        NN_LOG("No images are detected.\n");
        return;
    }

    // 選択している FW イメージを取得
    int activeIndex = scene.GetActiveImageIndex();
    NN_LOG("Selected image: %d\n", scene.GetActiveImageIndex());

    const auto& image = enumerator.GetImage(activeIndex);
    scene.m_PadView.SetTargetImage(image);
    scene.SetFocusedChild(&scene.m_PadView);

    // 詳細表示を消す
    if (scene.m_IsInfoVisible)
    {
        scene.m_IsInfoVisible = false;
        scene.m_InfoView.Disappear();
    }
    scene.UpdateKeyHelp();
}

void FlashWriter::PushPadButtonHandler(const nn::hid::system::UniquePadId& id, uintptr_t arg) NN_NOEXCEPT
{
    NN_LOG("Selected UniquePad: %lld\n", id._storage);

    auto& scene = *reinterpret_cast<FlashWriter*>(arg);
    scene.StartFirmwareUpdate(id);
}

void FlashWriter::UpdateKeyHelp() NN_NOEXCEPT
{
    auto& str = nns::hid::util::StringTable::GetInstance();
    m_KeyHelp.Clear();

    m_KeyHelp.AddHelp(ForceModeButtonMask, str.Get("FlashWriter_ForceUpdateModeHelp"));
    if (m_FwImageListContainer.IsFocused())
    {
        m_KeyHelp.AddHelp(
            DetailButtonMask,
            str.Get(m_IsInfoVisible ? "FlashWriter_HideDetailHelp" : "FlashWriter_ShowDetailHelp")
        );
    }
    else if (m_PadView.IsFocused())
    {
        m_KeyHelp.AddHelp(util::CancelButtonMask, str.Get("Back"));
    }
    m_KeyHelp.AddHelp(util::DecideButtonMask, str.Get("Decide"));
}

void FlashWriter::UpdateForceModeChange() NN_NOEXCEPT
{
    if (input::IsAnyTriggered(ForceModeButtonMask))
    {
        m_ForceModeCheckBox.PerformPress();
    }
}

void FlashWriter::CreateImageButtons() NN_NOEXCEPT
{
    if (m_pFwImageButtons != nullptr)
    {
        for (int i = 0; i < m_FwImageButtonCount; i++)
        {
            m_FwImageListContainer.RemoveChild(&m_pFwImageButtons[i]);
        }

        delete[] m_pFwImageButtons;
    }

    auto& enumerator = *util::FirmwareImageEnumerator::GetInstance();

    const auto decideMask =
#if defined(NNS_HID_ENABLE_CURSOR_DECISION)
        util::DecideButtonMask |
        nn::hid::NpadButton::Right::Mask |
        nn::hid::NpadButton::StickLRight::Mask |
        nn::hid::NpadButton::StickRRight::Mask;
#else
        util::DecideButtonMask;
#endif  // if defined(NNS_HID_ENABLE_CURSOR_DECISION)

    // ボタンの作成
    m_FwImageButtonCount = std::max(enumerator.GetFileCount(), 1);
    m_pFwImageButtons    = new ui::FirmwareImageButton[m_FwImageButtonCount];
    NN_ABORT_UNLESS_NOT_NULL(m_pFwImageButtons);
    for (int i = 0; i < m_FwImageButtonCount; i++)
    {
        auto& button = *(new (&m_pFwImageButtons[i]) ui::FirmwareImageButton());
        button.SetSkin(m_ButtonSkin);
        button.SetButtonMaskForPress(decideMask);
        button.SetSize(ImageButtonWidth, ImageButtonHeight);
        if (enumerator.GetFileCount() > 0)
        {
            button.SetImage(&enumerator.GetImage(i));
        }
        button.SetPushEventHandler(PushImageButtonHandler, reinterpret_cast<uintptr_t>(this));
        m_FwImageListContainer.AddChild(&button);
    }

    m_FwImageListContainer.SetFocusedChild(&m_pFwImageButtons[0]);
}

int FlashWriter::GetActiveImageIndex() NN_NOEXCEPT
{
    auto* pFocused = m_FwImageListContainer.GetFocusedChild();
    for (int i = 0; i < m_FwImageButtonCount; i++)
    {
        if (&m_pFwImageButtons[i] == pFocused)
        {
            return i;
        }
    }

    return -1;
}

void FlashWriter::StartFirmwareUpdate(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    auto& enumerator = *util::FirmwareImageEnumerator::GetInstance();

    // 選択している FW イメージを取得
    int activeIndex = GetActiveImageIndex();
    const auto& image = enumerator.GetImage(activeIndex);

    if (m_pImageBuffer != nullptr)
    {
        delete[] m_pImageBuffer;
    }

    m_ImageSize    = image.GetFileSize();
    m_pImageBuffer = new char[m_ImageSize + nn::os::MemoryPageSize];
    NN_ABORT_UNLESS_NOT_NULL(m_pImageBuffer);

    // nn::os::MemoryPageSize アラインメントのバッファを使う
    auto bufferPtr = nn::util::BytePtr(m_pImageBuffer);
    auto alignedBuffer = reinterpret_cast<decltype(m_pImageBuffer)>(
        bufferPtr.AlignUp(nn::os::MemoryPageSize).Get()
        );

    auto& str = nns::hid::util::StringTable::GetInstance();
    const char* message = str.Get("FlashWriter_UpdateFailed");
    bool isStarted = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!isStarted)
        {
            delete[] m_pImageBuffer;
            m_pImageBuffer = nullptr;
            ShowMessage(message);
        }
    };

    // FW イメージの読み込み
    if (image.ReadImage(alignedBuffer, m_ImageSize).IsFailure())
    {
        message = str.Get("FlashWriter_ReadImageFailed");
        return;
    }

    // FW 更新を開始
    nn::hid::system::FirmwareUpdateDeviceHandle handle;
    if (nn::hid::debug::StartFirmwareUpdateIndividual(
        &handle,
        id,
        image.GetTargetChip(),
        alignedBuffer,
        m_ImageSize).IsFailure())
    {
        return;
    }

    isStarted = true;

    // FW 更新中は HOME ボタン禁止
    nn::oe::BeginBlockingHomeButton();

    // 進捗表示を開始
    m_ProgressDialog.StartProgressCheck(id, handle);
    m_ProgressDialog.Appear();
    SetFocusedChild(&m_ProgressDialog);
}

void FlashWriter::ShowMessage(const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    auto& str = nns::hid::util::StringTable::GetInstance();

    m_Dialog.SetButtonCount(1);
    m_Dialog.SetButtonText(0, str.Get("OK"));

    auto okHandler = [](NNS_SGX_GUI_EVENT_HANDLER_ARGS(pSender, arg))
    {
        NN_UNUSED(pSender);

        auto& scene = *reinterpret_cast<FlashWriter*>(arg);
        scene.m_Dialog.Disappear();
        scene.SetFocusedChild(&scene.m_FwImageListContainer);
        scene.UpdateKeyHelp();
    };

    m_Dialog.SetButtonPushEventHandler(0, okHandler, reinterpret_cast<uintptr_t>(this));
    m_Dialog.SetCancelHandler(okHandler, reinterpret_cast<uintptr_t>(this));
    m_Dialog.SetRenderTextAreaHandler(NNS_SGX_GUI_EVENT_HANDLER(pSender, arg)
    {
        auto& dialog = *reinterpret_cast<nns::sgx::gui::Dialog*>(pSender);
        auto clientSize = dialog.GetTextAreaSize();

        nns::sgx::ScopedFontContextSaver saver;

        const auto* message = reinterpret_cast<const char*>(arg);
        nns::sgx::SetTextColor(nns::sgx::Colors::Shadow());
        nns::sgx::DrawText(4, 4, clientSize.width - 8, clientSize.height - 8, nns::sgx::TextAlignment::Center, message);
    }, reinterpret_cast<uintptr_t>(text));

    m_Dialog.Appear();
    SetFocusedChild(&m_Dialog);
}

}}}  // nns::hid::scene
