﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "layout_LineChart.h"

namespace nns { namespace hidfw { namespace layout {

    LineChart::LineChart() NN_NOEXCEPT
    {
        SetDefault();
    }

    void LineChart::SetDefault() NN_NOEXCEPT
    {
        static_cast<BaseItem*>(this)->SetDefault();
        SetSize(600.f, 300.f);

        m_MainColor     = nn::util::Color4u8::Black();  // 背景色
        m_SubColor      = nn::util::Color4u8::Black();  // 未使用
        m_TextColor     = nn::util::Color4u8::Blue();   // 未使用
        m_BorderColor   = nn::util::Color4u8::Gray();   // 枠線の色
        m_EffectColor   = nn::util::Color4u8::Green();  // ホバー時に表示されるエフェクト色
        m_IsFrameDraw = false;                          // 非表示時は枠線も描画しません
    }

    void LineChart::Update() NN_NOEXCEPT
    {
        if (m_State.Test<LayoutState::Selected>())
        {
            CallFunc();
            Cancel();
        }
        ++m_FrameCount;
    }

    void LineChart::PrintText() NN_NOEXCEPT
    {

    }

    void LineChart::Draw() NN_NOEXCEPT
    {
        nn::util::Color4u8 frameColor       = m_BorderColor;    // 枠線の色 (異常時には色を変更します)
        nn::util::Color4u8 subBorderColor   = m_BorderColor;    // 中央の横線の色
        subBorderColor.SetA(subBorderColor.GetA() * 0.5f);

        if (IsVisible())
        {
            //　描画が有効な場合
            gDrawer.SetColor(m_MainColor);

            //----------------------------------------------------------------
            //  背景と補助線の描画
            //----------------------------------------------------------------
            // 背景
            gDrawer.Draw2DRect(m_Pos, m_Size);
            gDrawer.SetColor(subBorderColor);
            // 横
            gDrawer.Draw2DLine(
                nn::util::MakeFloat2(m_Pos.x, m_Pos.y + m_Size.y / 4.f),
                nn::util::MakeFloat2(m_Pos.x + m_Size.x, m_Pos.y + m_Size.y / 4.f)
            );
            gDrawer.Draw2DLine(
                nn::util::MakeFloat2(m_Pos.x, m_Pos.y + (m_Size.y / 4.f) * 3.f),
                nn::util::MakeFloat2(m_Pos.x + m_Size.x, m_Pos.y + (m_Size.y / 4.f) * 3.f)
            );
            gDrawer.SetColor(m_BorderColor);
            // 横
            gDrawer.Draw2DLine(
                nn::util::MakeFloat2(m_Pos.x, m_Pos.y + m_Size.y / 2.f),
                nn::util::MakeFloat2(m_Pos.x + m_Size.x, m_Pos.y + m_Size.y / 2.f)
            );
            //----------------------------------------------------------------
            //  要素毎の描画処理
            //----------------------------------------------------------------
            for (
                std::vector<Element>::iterator it = m_Elements.begin();
                it != m_Elements.end();
                ++it
                )
            {
                if (((*it).Visible == true) && ((*it).Value.size() >= 2))
                {
                    // 要素の表示が有効かつ2つ以上の値を保持している時のみ描画を行います
                    size_t              valueIndex = 0;
                    float               valueSize = (*it).MaxValue - (*it).MinValue;
                    nn::util::Float2    pos[MaxValueLength];

                    for (size_t i = 0; i < (*it).Value.size(); ++i)
                    {
                        valueIndex = ((*it).ValueIndex + i) % (*it).Value.size();
                        pos[i].x = m_Pos.x + (float)static_cast<int>((static_cast<float>(m_Size.x * i) / static_cast<float>((*it).Value.size() - 1)) + 0.5f);
                        pos[i].y = m_Pos.y + (float)static_cast<int>(m_Size.y * (1.f - (((*it).Value.at(valueIndex) - (*it).MinValue) / valueSize)) + 0.5f);
                        if (((*it).Value.at(valueIndex) > std::max((*it).MaxValue, (*it).MinValue)) ||
                            ((*it).Value.at(valueIndex) < std::min((*it).MaxValue, (*it).MinValue)))
                        {
                            // 上限・下限 を超過した場合枠を赤くして警告します
                            frameColor = nn::util::Color4u8::Red();
                        }
                    }
                    const auto width = gDrawer.GetLineWidth();
                    gDrawer.SetColor((*it).Color);
                    gDrawer.SetLineWidth((*it).LineWidth);
                    gDrawer.Draw2DLine((*it).Value.size(), pos);
                    gDrawer.SetLineWidth(width);
                }
            }
        }

        if (IsVisible() || m_IsFrameDraw)
        {
            // 表示が有効か、外枠のみ表示設定になっている場合
            if (m_State.Test<LayoutState::Hover>())
            {
                // ホバー時のエフェクトの色
                const nn::util::Color4u8 transparenceEffectColor = nn::util::Color4u8(m_EffectColor.v[0], m_EffectColor.v[1], m_EffectColor.v[2], 0);
                float effect = nn::util::SinEst(nn::util::DegreeToRadian((m_FrameCount % 120) * 3)) * 1.f + 5.f;
                gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, m_EffectColor, transparenceEffectColor);
                gDrawer.Draw2DFrame(m_Pos, m_Size, effect);
            }
            gDrawer.SetColor(frameColor);
            gDrawer.Draw2DFrame(m_Pos, m_Size, 1);
        }
    }

    void LineChart::AddValue(int index, float value) NN_NOEXCEPT
    {
        m_Elements.at(index).ValueIndex %= m_Elements.at(index).Value.size();
        m_Elements.at(index).Value.at(m_Elements.at(index).ValueIndex) = value;
        m_Elements.at(index).ValueIndex = (m_Elements.at(index).ValueIndex + 1) % m_Elements.at(index).Value.size();
    }

    int LineChart::AddElement(const Element& data) NN_NOEXCEPT
    {
        m_Elements.push_back(data);
        return m_Elements.size();
    }

    const std::vector<LineChart::Element>& LineChart::GetElements() const
    {
        return m_Elements;
    }

    bool LineChart::VisibleElement(int index, bool enable) NN_NOEXCEPT
    {
        if (index < static_cast<int>(m_Elements.size()))
        {
            m_Elements.at(index).Visible = enable;
            return true;
        }
        return false;
    }

    LineChart::Element LineChart::CreateElement(nn::util::Color4u8 color, const char* name, int valueCount, float maxValue, float minValue, float lineWidth) NN_NOEXCEPT
    {
        Element outData;
        outData.Color       = color;
        outData.Name        = name;
        outData.MaxValue    = maxValue;
        outData.MinValue    = minValue;
        outData.Value.clear();
        outData.Value.resize(valueCount);
        outData.Visible     = true;
        outData.LineWidth   = lineWidth;
        return outData;
    }

}}}
