﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       縦棒グラフ作成のためのAPI宣言
 */

#pragma once

#include "layout_Base.h"

namespace nns { namespace hidfw { namespace layout {

    //----------------------------------------------------------------
    //! @brief 縦棒グラフを作成するためのクラスです
    //----------------------------------------------------------------
    class BarChart : public BaseItem
    {
    public:
        //----------------------------------------------------------------
        //! @brief グラフの描画モード
        //----------------------------------------------------------------
        enum ViewerMode
        {
            ViewerMode_Default,                                     //!< 標準的な描画を行います
            ViewerMode_Music,                                       //!< オーディオ用に描画を強化します
        };
        //----------------------------------------------------------------
        //! @brief グラフの要素
        //----------------------------------------------------------------
        struct Element
        {
            float HorizontalValue;                                  //!< 横軸の値
            float VerticalValue;                                    //!< 縦軸の値
            float DivisionNumber;                                   //!< 分割番号
            Element() NN_NOEXCEPT :
                HorizontalValue(0.f),
                VerticalValue(0.f)
            {
            }
            Element(float horizontal, float vertical) NN_NOEXCEPT  :
                HorizontalValue(horizontal),
                VerticalValue(vertical)
            {
            }
        };

    public:
        static const int        MaxDivisionCount = 256;             //!< 縦棒グラフが許容する最大の分割数

    public:

        BarChart() NN_NOEXCEPT;
        virtual ~BarChart() NN_NOEXCEPT {}
        virtual void Update() NN_NOEXCEPT;
        virtual void PrintText() NN_NOEXCEPT;
        virtual void Draw() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief  デフォルト値を設定
        //----------------------------------------------------------------
        virtual void SetDefault() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      枠の表示を残す
        //! @details    非表示設定時も枠の描画のみ有効化します
        //! @param[in]  enable trueの時、枠の表示を残す
        //----------------------------------------------------------------
        void LeaveFrameDrawing(bool enable) NN_NOEXCEPT { m_IsFrameDraw = enable; }

        //----------------------------------------------------------------
        //! @brief      縦軸と横軸の値を代入する
        //! @param[in]  horizontal  横軸
        //! @param[in]  vertical    縦軸
        //----------------------------------------------------------------
        void AddValue(float horizontal, float vertical) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      要素をクリアします
        //! @details    m_ViewerMode = ViewerMode_Music の場合、Update内で呼び出されます
        //----------------------------------------------------------------
        void Clear() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      描画モードを切り替えます
        //! @details    オーディオモードの時は、演出が強化されます
        //! @param[in]  mode 描画モード
        //----------------------------------------------------------------
        void SetViewerMode(ViewerMode mode) NN_NOEXCEPT { m_ViewerMode = mode; };

        //----------------------------------------------------------------
        //! @brief      グラフの分割数を設定します
        //! @param[in]  div 分割数
        //----------------------------------------------------------------
        void SetDivisionCount(int div) NN_NOEXCEPT { m_DivisionCount = div; }

        //----------------------------------------------------------------
        //! @brief      縦軸の最大値を設定します
        //! @param[in]  maxValue 最大値
        //----------------------------------------------------------------
        void SetMaxVerticalValue(float maxValue) NN_NOEXCEPT { m_MaxVerticalValue = maxValue; }

        //----------------------------------------------------------------
        //! @brief      縦軸の最小値を設定します
        //! @param[in]  minValue 最小値
        //----------------------------------------------------------------
        void SetMinVerticalValue(float minValue) NN_NOEXCEPT { m_MinVerticalValue = minValue; }

        //----------------------------------------------------------------
        //! @brief      縦軸の最大値を設定します
        //! @param[in]  maxValue 最大値
        //----------------------------------------------------------------
        void SetMaxHorizontalValue(float maxValue) NN_NOEXCEPT { m_MaxHorizontalValue = maxValue; }

        //----------------------------------------------------------------
        //! @brief      縦軸の最小値を設定します
        //! @param[in]  minValue 最小値
        //----------------------------------------------------------------
        void SetMinHorizontalValue(float minValue) NN_NOEXCEPT { m_MinHorizontalValue = minValue; }

    protected:
        bool                    m_IsFrameDraw;                      //!< true の場合、Visibleがfalseであっても枠のみ描画する
        ViewerMode              m_ViewerMode;                       //!< 描画モード
        int                     m_DivisionCount;                    //!< 分割数
        uint64_t                m_EffectCount[MaxDivisionCount];    //!< エフェクトの実行時間 (カウントアップ)
        int                     m_ElementsCount;                    //!< エレメントの個数
        float                   m_EffectValues[MaxDivisionCount];   //!< エフェクトの値
        float                   m_MaxVerticalValue;                 //!< 縦軸の最大値
        float                   m_MinVerticalValue;                 //!< 縦軸の最小値
        float                   m_MaxHorizontalValue;               //!< 横軸の最大値
        float                   m_MinHorizontalValue;               //!< 横軸の最小値
        std::vector<Element>    m_Elements;                         //!< 要素のリスト
        std::vector<Element>    m_OldElements;                      //!< アップデート直前の要素リストのコピー
    };

}}}
