﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <nn/vi.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
#include <nvnTool/nvnTool_GlslcInterface.h>
#endif
#endif
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

#define SHADER_SOURCE( ... ) #__VA_ARGS__

namespace nns { namespace hidfw { namespace gfx {

     /**
     * @brief       NintendoSDKが提供するグラフィックス処理を使用するためのクラスです
     * @details     線や図形、テキストの描画は本クラスを用いてGraphicsDrawer、DebugFontTextWriter
                    が使用します
                    例外として初期化、パフォーマンス計測等は、本クラスを直接用います
     */
    class GraphicsSystem
    {
        NN_DISALLOW_COPY(GraphicsSystem);
        NN_DISALLOW_MOVE(GraphicsSystem);
    public:
        GraphicsSystem() NN_NOEXCEPT;
        ~GraphicsSystem() NN_NOEXCEPT;

        void SetApplicationHeap(nn::mem::StandardAllocator* pValue) NN_NOEXCEPT;
        nn::gfx::Device& GetDevice() NN_NOEXCEPT;
        nn::vi::NativeWindowHandle GetNativeWindowHandle() NN_NOEXCEPT;
        nn::gfx::CommandBuffer& GetCommandBuffer() NN_NOEXCEPT;
        nn::gfx::util::DebugFontTextWriter& GetDebugFont() NN_NOEXCEPT;
        nns::gfx::PrimitiveRenderer::Renderer& GetPrimitiveRenderer() NN_NOEXCEPT;

        nn::gfx::ColorTargetView* GetColorTargetView() NN_NOEXCEPT { return &m_ColorTargetView; }
        nn::gfx::Texture* GetColorTargetTexture() NN_NOEXCEPT;
        nn::gfx::Shader* GetFrostedTGlassShader();

        uint64_t GetFrameCount() NN_NOEXCEPT;

        void Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        float GetScreenWidth() NN_NOEXCEPT;
        float GetScreenHeight() NN_NOEXCEPT;

        float SetScreenWidth() NN_NOEXCEPT;
        float SetScreenHeight() NN_NOEXCEPT;

        bool IsEnablePerfMeter() NN_NOEXCEPT;
        void SetEnablePerfMeter(bool enable) NN_NOEXCEPT;

        nn::gfx::DescriptorPool* GetTextureDescriptorPool() NN_NOEXCEPT { return &m_TextureDescriptorPool; }
        nn::gfx::DescriptorPool* GetSamplerDescriptorPool() NN_NOEXCEPT { return &m_SamplerDescriptorPool; }

        nn::gfx::DepthStencilView* GetDepthStencilView() NN_NOEXCEPT { return &m_DepthStencilView; }

        nn::gfx::ColorTargetView* GetDefaultRenderTarget() NN_NOEXCEPT { return &m_ColorTargetView; }
        void SetRenderTarget(nn::gfx::ColorTargetView* pColorTargetView) NN_NOEXCEPT;
        void SetRenderTarget(nn::gfx::ColorTargetView* pColorTargetView, nn::gfx::DepthStencilView* pDepthStencilView) NN_NOEXCEPT;

        void BeginPerfFrame() NN_NOEXCEPT;
        void EndPerfFrame() NN_NOEXCEPT;
        void BeginDraw(const nn::util::Color4u8& clearColor = nn::util::Color4u8::Black()) NN_NOEXCEPT;
        void EndDraw() NN_NOEXCEPT;
        void DrawDone() NN_NOEXCEPT;
        void Synchronize(const nn::TimeSpan& timeout) NN_NOEXCEPT;

        void AllocateTexture(nn::gfx::Texture* pTexture,
             const nn::gfx::Texture::InfoType* pInfo
             ) NN_NOEXCEPT;
        void AllocateBuffer(nn::gfx::Buffer* pBuffer,
            const nn::gfx::Buffer::InfoType* pInfo
            ) NN_NOEXCEPT;
        void FreeTexture(nn::gfx::Texture* pTexture
            ) NN_NOEXCEPT;
        void FreeBuffer(nn::gfx::Buffer* pBuffer
            ) NN_NOEXCEPT;

        void RegisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutValue,
            const nn::gfx::TextureView& textureView
            ) NN_NOEXCEPT;
        void RegisterSamplerSlot(nn::gfx::DescriptorSlot* pOutValue,
            const nn::gfx::Sampler& sampler) NN_NOEXCEPT;
        void UnregisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutValue,
            const nn::gfx::TextureView& textureView
            ) NN_NOEXCEPT;
        void UnregisterSamplerSlot(nn::gfx::DescriptorSlot* pOutValue,
            const nn::gfx::Sampler& sampler) NN_NOEXCEPT;

        void RegisterDebugFontTextWriter(nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter) NN_NOEXCEPT;
        void UnregisterDebugFontTextWriter(nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter) NN_NOEXCEPT;

    private:
        static void OutOfCommandMemoryEventCallback(
            nn::gfx::CommandBuffer* pCommandBuffer,
            const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT;
        static void OutOfControlMemoryEventCallback(
            nn::gfx::CommandBuffer* pCommandBuffer,
            const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT;

        static void* AllocateFunction(size_t size,
            size_t alignment,
            void* pUserData) NN_NOEXCEPT;
        static void FreeFunction(void* ptr, void* pUserData) NN_NOEXCEPT;

        void InitializeDisplay() NN_NOEXCEPT;
        void InitializeDevice() NN_NOEXCEPT;
        void InitializeMemoryPool() NN_NOEXCEPT;
        void InitializeMemoryPoolInvisible() NN_NOEXCEPT;
        void InitializeSwapChain() NN_NOEXCEPT;
        void InitializeQueue() NN_NOEXCEPT;
        void InitializeFence() NN_NOEXCEPT;
        void InitializeColorTargetTexture() NN_NOEXCEPT;
        void InitializeColorTargetView() NN_NOEXCEPT;
        void InitializeDepthStencilTexture() NN_NOEXCEPT;
        void InitializeDepthStencilView() NN_NOEXCEPT;
        void InitializeTextureDescriptorPool() NN_NOEXCEPT;
        void InitializeSamplerDescriptorPool() NN_NOEXCEPT;
        void InitializeCommandBuffer() NN_NOEXCEPT;
        void InitializeViewportScissorState() NN_NOEXCEPT;
        void InitializeBlendState() NN_NOEXCEPT;
        void InitializeDepthStencilState() NN_NOEXCEPT;
        void InitializeRasterizerState() NN_NOEXCEPT;
        void InitializeDebugFont() NN_NOEXCEPT;
        void InitializeLoadMeter() NN_NOEXCEPT;
        void InitializePrimitiveRenderer() NN_NOEXCEPT;
        void InitializeBlurBuffer() NN_NOEXCEPT;
        void InitializeShader() NN_NOEXCEPT;
        void FinalizeMemoryPool() NN_NOEXCEPT;
        void FinalizeMemoryPoolInvisible() NN_NOEXCEPT;
        void FinalizeViewportScissorState() NN_NOEXCEPT;
        void FinalizeBlendState() NN_NOEXCEPT;
        void FinalizeDebugFont() NN_NOEXCEPT;
        void FinalizeLoadMeter() NN_NOEXCEPT;
        void FinalizePrimitiveRenderer() NN_NOEXCEPT;
        void ResetCommandBuffer() NN_NOEXCEPT;
        void AddCommandMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
            ) NN_NOEXCEPT;
        void AddControlMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
            ) NN_NOEXCEPT;

    private:
        nn::mem::StandardAllocator* m_pApplicationHeap;

        nn::gfx::MemoryPool m_MemoryPool;
        void* m_pMemoryPool;
        ptrdiff_t m_pMemoryPoolOffset;
        ptrdiff_t m_pMemoryPoolCommandOffset;
        void*  m_pPerfMeterMemory;

        nn::gfx::MemoryPool m_MemoryPoolInvisible;
        void* m_pMemoryPoolInvisible;
        ptrdiff_t m_pMemoryPoolInvisibleOffset;

        nn::gfx::DescriptorPool m_TextureDescriptorPool;
        int m_TextureSlotCount;

        nn::gfx::DescriptorPool m_SamplerDescriptorPool;
        int m_SamplerSlotCount;

        bool m_ScanStart;

        nn::vi::Display* m_pDisplay;
        int m_RenderWidth;
        int m_RenderHeight;
        nn::vi::Layer* m_pLayer;
        nn::vi::NativeWindowHandle m_NativeWindowHandle;
        nn::gfx::Device m_Device;
        nn::gfx::SwapChain m_SwapChain;
        nn::gfx::Queue m_Queue;
        nn::gfx::Fence m_Fence[2];
        nn::gfx::Fence m_DisplayFence;
        nn::gfx::Semaphore m_DisplaySemaphore;
        nn::gfx::Texture m_ColorTargetTexture;
        nn::gfx::ColorTargetView m_ColorTargetView;
        nn::gfx::Texture m_DepthStencilTexture;
        nn::gfx::DepthStencilView m_DepthStencilView;
        nn::gfx::CommandBuffer m_CommandBuffer;
        nn::gfx::ViewportScissorState m_ViewportScissorState;
        nn::gfx::BlendState m_BlendState;
        nn::gfx::DepthStencilState m_DepthStencilState;
        nn::gfx::RasterizerState m_RasterizerState;

        nn::util::BytePtr m_DebugFontHeap;
        nn::gfx::util::DebugFontTextWriter m_DebugFont;
        nns::gfx::PrimitiveRenderer::Renderer* m_pPrimitiveRenderer;

        std::vector<void*> m_CommandBufferControlMemoryChunks;

        nn::gfx::Shader m_PSGrassShader;
        int m_PSGrassSlot;

        nns::gfx::PrimitiveRenderer::MeterDrawer m_MeterDrawer;

        uint64_t m_FrameCount;

        nn::gfx::Texture* m_pScanBuffer[2];
        nn::gfx::ColorTargetView* m_pScanBufferView[2];

        bool m_IsEnablePerfMeter;
    };

    void* Allocate(size_t size) NN_NOEXCEPT;
    void Deallocate(void* addr, size_t size) NN_NOEXCEPT;
    void* Reallocate(void* addr, size_t newSize, void*) NN_NOEXCEPT;

}}}
