﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <memory>

#include <nn/nn_Common.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/applet/applet.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#include <nv_Scheduler.h>

namespace
{
    const size_t GraphicsSystemMemorySize = 64 * 1024;
    char GraphicsSystemMemory[GraphicsSystemMemorySize] NN_ALIGNAS(4096);

    enum
    {
        RunListId_None = 0,
        RunListId_Overlay = 1,
        RunListId_System = 2,
        RunListId_Application = 3,
    };

    const char* RunListNames[]
    {
        "None",
        "Overlay",
        "System",
        "Application",
    };
}

extern "C" void nninitStartup()
{
}

extern "C" void nnMain()
{
    nn::Result result;

    auto argc = nn::os::GetHostArgc();
    auto argv = nn::os::GetHostArgv();

    NN_LOG("[GpuOverrunNotifier] CommandLineOptions: ");
    for (auto i = 0; i < argc; i++)
    {
        NN_LOG("%s ", argv[i]);
    }
    NN_LOG("\n");

    uint64_t targetRunlistId = RunListId_None;
    uint64_t debtThreashold = 0;
    uint64_t timeout = 0;

    while (argv[0] != NULL)
    {
        auto arg = argv[0];

        if (!std::strcmp(arg, "--help"))
        {
            NN_LOG(
                "[GpuOverrunNotifier] Usage: GpuOverrunNotifier.nsp \n"
                "[GpuOverrunNotifier]   --target-runlist-id <runlist-id> (1: Overlay, 2: System, 3: Application)\n"
                "[GpuOverrunNotifier]   --debt-threashold <threashold> (unit: nsec)\n"
                "[GpuOverrunNotifier]   --timeout <timeout> (unit: sec, 0 means never finished.)\n"
            );

            return;
        }
        if (!std::strcmp(arg, "--target-runlist-id"))
        {
            argv++;
            targetRunlistId = std::atoi(argv[0]);
        }
        if (!std::strcmp(arg, "--debt-threashold"))
        {
            argv++;
            debtThreashold = std::atoi(argv[0]);
        }
        if (!std::strcmp(arg, "--timeout"))
        {
            argv++;
            timeout = std::atoi(argv[0]);
        }

        argv++;
    }

    NN_LOG(
        "[GpuOverrunNotifier] Settings:\n"
        "[GpuOverrunNotifier]   TargetRunlistId: %lld (%s)\n"
        "[GpuOverrunNotifier]   DebtThreashold: %llu nsec\n"
        "[GpuOverrunNotifier]   Timeout: %llu sec\n"
        , targetRunlistId, RunListNames[targetRunlistId]
        , debtThreashold
        , timeout
    );

    nv::SetGraphicsServiceName("nvdrv:s");
    nv::InitializeGraphics(GraphicsSystemMemory, sizeof(GraphicsSystemMemory));

    nv::sched::Client client;
    client.Open();
    NN_UTIL_SCOPE_EXIT
    {
        client.EnableOverrunEvents(false);
        client.Close();
        NN_LOG("[GpuOverrunNotifier] Finished.\n");
    };

    client.EnableOverrunEvents(false);

    nn::os::SystemEventType overrunEvent;
    client.GetOverrunEventHandle(&overrunEvent);

    // cleanup previous overrun data.
    nv::sched::Client::OverrunEventType overrunData;
    while (client.HasOverrunData())
    {
        result = client.GetOverrunData(&overrunData);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    auto current = nn::os::GetSystemTick().ToTimeSpan();
    auto end = current + nn::TimeSpan::FromSeconds(timeout);

    client.EnableOverrunEvents(true);

    NN_LOG("[GpuOverrunNotifier] Started.\n");

    while (timeout == 0 || current < end)
    {
        nn::os::TimedWaitSystemEvent(&overrunEvent, nn::TimeSpan::FromSeconds(1));

        while (client.HasOverrunData())
        {
            result = client.GetOverrunData(&overrunData);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            auto debt = overrunData.debt_total - overrunData.debt_from_powergating;

            if (overrunData.runlist_id != targetRunlistId)
                break;

            if (debt < debtThreashold)
                break;

            NN_LOG(
                "[GpuOverrunNotifier] ********* Overrun happens *********\n"
                "[GpuOverrunNotifier]           RunlistId: %llu (%s)\n"
                "[GpuOverrunNotifier]           Timeslice: %llu nsec\n"
                "[GpuOverrunNotifier]           DebtTotal: %llu nsec\n"
                "[GpuOverrunNotifier]                Debt: %llu nsec\n"
                "[GpuOverrunNotifier] DebtFromPowergating: %llu nsec\n"
                , overrunData.runlist_id, RunListNames[overrunData.runlist_id]
                , overrunData.timeslice
                , overrunData.debt_total
                , debt
                , overrunData.debt_from_powergating
            );

        }

        current = nn::os::GetSystemTick().ToTimeSpan();
    }
} // NOLINT(impl/function_size)
