﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <string>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/account/account_Types.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/util/util_ScopeExit.h>
#include "DevQuestMenu_Log.h"
#include "DevQuestMenu_RidFileIo.h"
#include "DevQuestMenu_RidSaveDataIo.h"

namespace nn { namespace devquestmenu {

    bool RidSaveDataIo::Read(nn::Bit8* pBuffer, int bufferSize, const char* saveDataName) NN_NOEXCEPT
    {
        //! 本体セーブデータへのマウント
        const std::string mountName = "deviceSaveData";
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountDeviceSaveData(mountName.c_str()));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(mountName.c_str());
        };

        //! 本体セーブデータディレクトリのオープン
        nn::fs::DirectoryHandle directoryHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, (mountName + ":/").c_str(), nn::fs::OpenDirectoryMode::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(directoryHandle);
        };

        //! セーブデータ読み込み
        return RidFileIo::ReadFile(pBuffer, bufferSize, (mountName + ":/" + saveDataName).c_str());
    };

    void RidSaveDataIo::Write(nn::Bit8* pBuffer, int bufferSize, const char* saveDataName) NN_NOEXCEPT
    {
        //! 本体セーブデータへのマウント
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::EnsureSaveData(nn::account::InvalidUid));
        const std::string mountName = "deviceSaveData";
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountDeviceSaveData(mountName.c_str()));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(mountName.c_str());
        };

        {
            //! 本体セーブデータディレクトリのオープン
            nn::fs::DirectoryHandle directoryHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, (mountName + ":/").c_str(), nn::fs::OpenDirectoryMode::OpenDirectoryMode_All));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseDirectory(directoryHandle);
            };

            nn::fs::FileHandle fileHandle;

            //! ファイルを Open する
            const std::string filePath = mountName + ":/" + saveDataName;
            nn::Result result = nn::fs::OpenFile(&fileHandle, filePath.c_str(), nn::fs::OpenMode::OpenMode_Write);

            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                //! ファイルが存在しない場合は作成する
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateFile(filePath.c_str(), bufferSize));
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, filePath.c_str(), nn::fs::OpenMode::OpenMode_Write));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            //! ファイル書き込み
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(fileHandle, 0, pBuffer, bufferSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CommitSaveData(mountName.c_str()));
    };
}}
