﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Color.h>
#include "ui_Types.h"
#include "ui_ItemAlignment.h"

namespace ui{

    enum BackgroundMode
    {
        BackgroundMode_Color,
        BackgroundMode_Transparent,
        BackgroundMode_InnerItem,
    };

    enum SizeMode
    {
        SizeMode_Auto,
        SizeMode_Fixed,
    };

    enum BorderType
    {
        BorderType_None = 0,
        BorderType_Solid,
    };

    enum ScrollbarMode
    {
        ScrollbarMode_None = 0,
        ScrollbarMode_Always,
    };

    class BackgroundStyle
    {
    public:
        BackgroundStyle& SetMode(BackgroundMode value) NN_NOEXCEPT
        {
            m_Mode = value;
            return *this;
        }

        BackgroundStyle& SetColor(const nn::util::Color4f& value) NN_NOEXCEPT
        {
            m_Color = value;
            return *this;
        }

        BackgroundMode GetMode() const NN_NOEXCEPT { return m_Mode; }
        nn::util::Color4f GetColor() const NN_NOEXCEPT { return m_Color; }

    private:
        BackgroundMode m_Mode = BackgroundMode_Color;
        nn::util::Color4f m_Color = {0, 0, 0, 1};
    };

    class SizeStyle
    {
    public:
        SizeStyle& SetAutoSize() NN_NOEXCEPT
        {
            m_Mode = SizeMode_Auto;
            m_Size = {0, 0};
            return *this;
        }

        SizeStyle& SetAutoSize(const Size& sizeMin) NN_NOEXCEPT
        {
            m_Mode = SizeMode_Auto;
            m_Size = sizeMin;
            return *this;
        }

        SizeStyle& SetFixedSize(const Size& size) NN_NOEXCEPT
        {
            m_Mode = SizeMode_Fixed;
            m_Size = size;
            return *this;
        }

        SizeMode GetMode() const NN_NOEXCEPT { return m_Mode; }
        Size GetSize() const NN_NOEXCEPT { return m_Size; }

    private:
        SizeMode m_Mode = SizeMode_Auto;
        Size m_Size = {0, 0};
    };

    class TextStyle
    {
    public:
        TextStyle& SetFontSize(float value) NN_NOEXCEPT
        {
            m_FontSize = value;
            return *this;
        }

        TextStyle& SetFontColor(const nn::util::Color4f& value) NN_NOEXCEPT
        {
            m_FontColor = value;
            return *this;
        }

        float GetFontSize() const NN_NOEXCEPT { return m_FontSize; }
        nn::util::Color4f GetFontColor() const NN_NOEXCEPT { return m_FontColor; }

    private:
        float              m_FontSize = { 10 };
        nn::util::Color4f  m_FontColor = { 1, 1, 1 };
    };

    class IconStyle
    {
    public:
        IconStyle& SetIconSize(float value) NN_NOEXCEPT
        {
            m_IconSize = value;
            return *this;
        }

        IconStyle& SetIconColor(const nn::util::Color4f& value) NN_NOEXCEPT
        {
            m_IconColor = value;
            return *this;
        }

        float GetIconSize() const NN_NOEXCEPT { return m_IconSize; }
        nn::util::Color4f GetIconColor() const NN_NOEXCEPT { return m_IconColor; }

    private:
        float              m_IconSize = { 10 };
        nn::util::Color4f  m_IconColor = { 1, 1, 1 };
    };

    class BorderStyle
    {
    public:
        BorderStyle& SetType(BorderType value) NN_NOEXCEPT
        {
            m_Type = value;
            return *this;
        }

        BorderStyle& SetColor(const nn::util::Color4f& value) NN_NOEXCEPT
        {
            m_Color = value;
            return *this;
        }

        BorderStyle& SetOuterWidth(int top, int right, int bottom, int left) NN_NOEXCEPT
        {
            m_TopWidth = top;
            m_RightWidth = right;
            m_BottomWidth = bottom;
            m_LeftWidth = left;
            return *this;
        }

        BorderStyle& SetOuterWidth(int w) NN_NOEXCEPT
        {
            return SetOuterWidth(w, w, w, w);
        }

        BorderStyle& SetInnerWidth(int horizontal, int vertical) NN_NOEXCEPT
        {
            m_HorizontalWidth = horizontal;
            m_VerticalWidth = vertical;
            return *this;
        }

        BorderStyle& SetInnerWidth(int w) NN_NOEXCEPT
        {
            return SetInnerWidth(w, w);
        }

        BorderType GetType() const NN_NOEXCEPT { return m_Type; }
        nn::util::Color4f GetColor() const NN_NOEXCEPT { return m_Color; }
        int GetTopWidth() const NN_NOEXCEPT { return m_TopWidth; }
        int GetRightWidth() const NN_NOEXCEPT { return m_RightWidth; }
        int GetBottomWidth() const NN_NOEXCEPT { return m_BottomWidth; }
        int GetLeftWidth() const NN_NOEXCEPT { return m_LeftWidth; }
        int GetHorizontalWidth() const NN_NOEXCEPT { return m_HorizontalWidth; }
        int GetVerticalWidth() const NN_NOEXCEPT { return m_VerticalWidth; }

    private:
        BorderType m_Type = BorderType_Solid;
        nn::util::Color4f m_Color = nn::util::Color4f(1, 1, 1, 1);

        int m_TopWidth    = 0;
        int m_RightWidth  = 0;
        int m_BottomWidth = 0;
        int m_LeftWidth   = 0;

        int m_HorizontalWidth = 0;
        int m_VerticalWidth   = 0;
    };

    class ScrollbarStyle
    {
    public:
        ScrollbarStyle& SetMode(ScrollbarMode value) NN_NOEXCEPT
        {
            m_Mode = value;
            return *this;
        }

        ScrollbarStyle& SetWidth(int value) NN_NOEXCEPT
        {
            m_Width = value;
            return *this;
        }

        ScrollbarStyle& SetBarColor(const nn::util::Color4f& value) NN_NOEXCEPT
        {
            m_BarColor = value;
            return *this;
        }

        ScrollbarStyle& SetKnobColor(const nn::util::Color4f& value) NN_NOEXCEPT
        {
            m_KnobColor = value;
            return *this;
        }

        ScrollbarMode GetMode() const NN_NOEXCEPT { return m_Mode; }
        int GetWidth() const NN_NOEXCEPT { return m_Width; }
        nn::util::Color4f GetBarColor() const NN_NOEXCEPT { return m_BarColor; }
        nn::util::Color4f GetKnobColor() const NN_NOEXCEPT { return m_KnobColor; }

    private:
        ScrollbarMode m_Mode = ScrollbarMode_None;
        int m_Width = 8;
        nn::util::Color4f m_BarColor = {0, 0, 0, 1};
        nn::util::Color4f m_KnobColor = {1, 1, 1, 1};
    };

    class CheckMarkStyle
    {
    public:
        CheckMarkStyle& SetSize(int width, int height) NN_NOEXCEPT
        {
            m_Width = width;
            m_Height = height;
            return *this;
        }

        CheckMarkStyle& SetCheckMarkSize(float value) NN_NOEXCEPT
        {
            m_CheckMarkSize = value;
            return *this;
        }

        CheckMarkStyle& SetCheckMarkColor(const nn::util::Color4f& value) NN_NOEXCEPT
        {
            m_CheckMarkColor = value;
            return *this;
        }

        CheckMarkStyle& SetBorderWidth(int value) NN_NOEXCEPT
        {
            m_BorderWidth = value;
            return *this;
        }

        CheckMarkStyle& SetBackgroundColor(const nn::util::Color4f& value) NN_NOEXCEPT
        {
            m_BackgroundColor = value;
            return *this;
        }

        CheckMarkStyle& SetBorderColor(const nn::util::Color4f& value) NN_NOEXCEPT
        {
            m_BorderColor = value;
            return *this;
        }

        int GetWidth() const NN_NOEXCEPT { return m_Width; }
        int GetHeight() const NN_NOEXCEPT { return m_Height; }
        int GetBorderWidth() const NN_NOEXCEPT { return m_BorderWidth; }
        float GetCheckMarkSize() const NN_NOEXCEPT { return m_CheckMarkSize; }
        nn::util::Color4f GetCheckMarkColor() const NN_NOEXCEPT { return m_CheckMarkColor; }
        nn::util::Color4f GetBackgroundColor() const NN_NOEXCEPT { return m_BackgroundColor; }
        nn::util::Color4f GetBorderColor() const NN_NOEXCEPT { return m_BorderColor; }

    private:
        int m_Width = 32;
        int m_Height = 32;
        int m_BorderWidth = 1;
        float              m_CheckMarkSize   = { 10 };
        nn::util::Color4f  m_CheckMarkColor  = { 1, 1, 1 };
        nn::util::Color4f  m_BackgroundColor = { 0, 0, 0, };
        nn::util::Color4f  m_BorderColor     = { 1, 1, 1 };
    };



}
