﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ui_MenuItemUserCustom.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "util/ui_CalculateItemSize.h"
#include "util/ui_CalculateAlignedPosition.h"
#include "util/ui_CalculateBackgroundVisibility.h"

namespace ui{

    MenuItemUserCustom::MenuItemUserCustom(const MenuItemUserCustomStyle& style) NN_NOEXCEPT
        : m_Style(style)
    {
        m_pContainer = std::make_shared<panel::PanelContainer>();
    }

    std::shared_ptr<MenuItemUserCustom> MenuItemUserCustom::SetFocusAcceptable(bool value) NN_NOEXCEPT
    {
        m_IsFocusAcceptable = value;
        return this->shared_from_this();
    }

    std::shared_ptr<MenuItemUserCustom> MenuItemUserCustom::SetTouchAcceptable(bool value) NN_NOEXCEPT
    {
        m_IsTouchAcceptable = value;
        return this->shared_from_this();
    }

    std::shared_ptr<MenuItemUserCustom> MenuItemUserCustom::SetExternalPanel(const std::shared_ptr<panel::IPanel>& value) NN_NOEXCEPT
    {
        m_pExternalPanel = value;
        return this->shared_from_this();
    }

    std::shared_ptr<MenuItemUserCustom> MenuItemUserCustom::SetUpdateLayoutCallbackFunction(const std::function<ui::Size ()>& value) NN_NOEXCEPT
    {
        m_UpdateLayoutCallbackFunction = value;
        return this->shared_from_this();
    }

    std::shared_ptr<MenuItemUserCustom> MenuItemUserCustom::SetUpdatePanelCallbackFunction(const std::function<void (const ui::Position&)>& value) NN_NOEXCEPT
    {
        m_UpdatePanelCallbackFunction = value;
        return this->shared_from_this();
    }

    void MenuItemUserCustom::UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        Size innerSize = {};
        if(m_UpdateLayoutCallbackFunction)
        {
            innerSize = m_UpdateLayoutCallbackFunction();
        }

        auto margin = m_Style.GetMargin();
        Size layoutSize = {
            innerSize.width + margin.left + margin.right,
            innerSize.height + margin.top + margin.bottom
        };

        auto outerSize = util::CalculateItemSize::Calculate([&](){ return layoutSize; }, m_Style.GetSizeStyle());

        auto layoutPosition = util::CalculateAlignedPosition::Calculate(layoutSize, outerSize, m_Style.GetAlignment());

        SetLayoutSize(outerSize);
        m_ExternalPanelPosition = {layoutPosition.x + margin.left, layoutPosition.y + margin.top};
    }

    void MenuItemUserCustom::UpdatePanel() NN_NOEXCEPT
    {
        auto pos = GetLayoutPosition();
        auto size = GetLayoutSize();
        auto vis = util::CalculateBackgroundVisibility::Calculate(m_Style.GetBackgroundStyle(), m_pExternalPanel);
        m_pContainer->SetVisibility(vis.visibility);
        m_pContainer->SetColor(vis.color);
        m_pContainer->SetPosition(pos.x, pos.y);
        m_pContainer->SetSize(size.width, size.height);

        if(m_UpdatePanelCallbackFunction)
        {
            m_UpdatePanelCallbackFunction(m_ExternalPanelPosition);
        }

        std::vector<std::shared_ptr<panel::IPanel>> panelList;
        if(m_pExternalPanel)
        {
            panelList.push_back(m_pExternalPanel);
        }
        m_pContainer->SetChildren(panelList);
    }

    std::shared_ptr<panel::IPanel> MenuItemUserCustom::GetPanel() NN_NOEXCEPT
    {
        return m_pContainer;
    }

}
