﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ui_MenuItemScrollBox.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "util/ui_MenuItemTree.h"
#include "util/ui_MenuItemGeometry.h"
#include "util/ui_CalculateBackgroundVisibility.h"
#include "util/ui_FindFocusAcceptingItem.h"

namespace ui{

    MenuItemScrollBox::MenuItemScrollBox(const MenuItemScrollBoxStyle& style) NN_NOEXCEPT
    {
        m_pPanel = std::make_shared<panel::PanelContainer>();
        m_pViewAreaPanel = std::make_shared<panel::PanelContainer>();
        m_pHorizontalBar  = std::make_shared<panel::PanelText>();
        m_pHorizontalKnob = std::make_shared<panel::PanelText>();
        m_pVerticalBar  = std::make_shared<panel::PanelText>();
        m_pVerticalKnob = std::make_shared<panel::PanelText>();

        m_pPanel->AddChild(m_pViewAreaPanel);
        m_pPanel->AddChild(m_pHorizontalBar);
        m_pPanel->AddChild(m_pVerticalBar);
        m_pPanel->AddChild(m_pHorizontalKnob);
        m_pPanel->AddChild(m_pVerticalKnob);

        m_Style = style;
    }

    MenuButtonResult MenuItemScrollBox::HandleUpButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT
    {
        if(PassFocusToInnerItemImpl(context))
        {
            return MenuButtonResult::GetHandled();
        }

        // きっちり端までスクロールするまでは次のアイテムにフォーカスを移動しない
        int scrollDistance = m_Style.GetArrowButtonScrollDistance();
        if( m_Layout.m_IsVerticallyScrolling &&
            scrollDistance > 0 &&
            m_Layout.m_InnerPosition.y < 0)
        {
            if(auto pPage = context.GetPage())
            {
                m_RequestedInnerPositionUpdate.y += scrollDistance;
                pPage->UpdateLayout();
                return MenuButtonResult::GetHandled();
            }
        }

        return MenuButtonResult::GetNotHandled();
    }

    MenuButtonResult MenuItemScrollBox::HandleDownButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT
    {
        if(PassFocusToInnerItemImpl(context))
        {
            return MenuButtonResult::GetHandled();
        }

        // きっちり端までスクロールするまでは次のアイテムにフォーカスを移動しない
        int scrollDistance = m_Style.GetArrowButtonScrollDistance();
        if( m_Layout.m_IsVerticallyScrolling &&
            scrollDistance > 0 &&
            (-m_Layout.m_InnerPosition.y + m_Layout.m_ViewSize.height < m_Layout.m_InnerLayout.m_Size.height))
        {
            if(auto pPage = context.GetPage())
            {
                m_RequestedInnerPositionUpdate.y -= scrollDistance;
                pPage->UpdateLayout();
                return MenuButtonResult::GetHandled();
            }
        }

        return MenuButtonResult::GetNotHandled();
    }

    MenuButtonResult MenuItemScrollBox::HandleLeftButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT
    {
        if(PassFocusToInnerItemImpl(context))
        {
            return MenuButtonResult::GetHandled();
        }

        // きっちり端までスクロールするまでは次のアイテムにフォーカスを移動しない
        int scrollDistance = m_Style.GetArrowButtonScrollDistance();
        if( m_Layout.m_IsHorizontallyScrolling &&
            scrollDistance > 0 &&
            m_Layout.m_InnerPosition.x < 0)
        {
            if(auto pPage = context.GetPage())
            {
                m_RequestedInnerPositionUpdate.x += scrollDistance;
                pPage->UpdateLayout();
                return MenuButtonResult::GetHandled();
            }
        }

        return MenuButtonResult::GetNotHandled();
    }

    MenuButtonResult MenuItemScrollBox::HandleRightButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT
    {
        if(PassFocusToInnerItemImpl(context))
        {
            return MenuButtonResult::GetHandled();
        }

        // きっちり端までスクロールするまでは次のアイテムにフォーカスを移動しない
        int scrollDistance = m_Style.GetArrowButtonScrollDistance();
        if( m_Layout.m_IsHorizontallyScrolling &&
            scrollDistance > 0 &&
            (-m_Layout.m_InnerPosition.x + m_Layout.m_ViewSize.width < m_Layout.m_InnerLayout.m_Size.width))
        {
            if(auto pPage = context.GetPage())
            {
                m_RequestedInnerPositionUpdate.x -= scrollDistance;
                pPage->UpdateLayout();
                return MenuButtonResult::GetHandled();
            }
        }

        return MenuButtonResult::GetNotHandled();
    }

    bool MenuItemScrollBox::PassFocusToInnerItemImpl(const MenuButtonHandleContext& context) NN_NOEXCEPT
    {
        // 自分がフォーカスを持っているときに方向ボタンを入力されたら子にフォーカスを渡す。
        if(IsFocused())
        {
            if(auto pToBeFocused = util::FindFocusAcceptingItem::Find(m_pInnerItem, context.GetCurrentFocusedItem(), util::FindFocusAcceptingItem::Distance))
            {
                if(auto pPage = context.GetPage())
                {
                    pPage->ChangeFocus(pToBeFocused);
                    return true;
                }
            }
        }
        return false;
    }

    MenuButtonResult MenuItemScrollBox::HandleTouchDown(const MenuButtonHandleContext& context, const nn::util::Vector3f& value) NN_NOEXCEPT
    {
        m_TouchState.m_IsHeld = true;
        m_TouchState.m_PrevPosition = {static_cast<int>(value.GetX()), static_cast<int>(value.GetY())};
        return MenuButtonResult::GetHandled();
    }

    MenuButtonResult MenuItemScrollBox::HandleTouchMove(const MenuButtonHandleContext& context, const nn::util::Vector3f& value) NN_NOEXCEPT
    {
        if(!m_TouchState.m_IsHeld)
        {
            return MenuButtonResult::GetNotHandled();
        }

        if(auto pPage = context.GetPage())
        {
            Position cur = {static_cast<int>(value.GetX()), static_cast<int>(value.GetY())};
            auto delta = cur - m_TouchState.m_PrevPosition;

            m_RequestedInnerPositionUpdate += delta;
            m_TouchState.m_PrevPosition = cur;

            pPage->UpdateLayout();
        }

        return MenuButtonResult::GetHandled();
    }

    MenuButtonResult MenuItemScrollBox::HandleTouchUp(const MenuButtonHandleContext&, const nn::util::Vector3f&) NN_NOEXCEPT
    {
        m_TouchState.m_IsHeld = false;
        m_TouchState.m_PrevPosition = {};
        return MenuButtonResult::GetHandled();
    }

    MenuButtonResult MenuItemScrollBox::HandleTouchCanceled() NN_NOEXCEPT
    {
        m_TouchState.m_IsHeld = false;
        m_TouchState.m_PrevPosition = {};
        return MenuButtonResult::GetHandled();
    }

    MenuButtonResult MenuItemScrollBox::HandleRightAnalogStickInput(const MenuButtonHandleContext& context, const nn::util::Vector3f& value) NN_NOEXCEPT
    {
        int distanceMax = m_Style.GetAnalogStickScrollDistanceMax();
        if(distanceMax <= 0)
        {
            return MenuButtonResult::GetNotHandled();
        }

        int deltaX = distanceMax * value.GetX();
        int deltaY = distanceMax * value.GetY();

        if(context.GetPage() && (deltaX != 0 || deltaY != 0))
        {
            m_RequestedInnerPositionUpdate.x -= deltaX;
            m_RequestedInnerPositionUpdate.y -= deltaY;
            context.GetPage()->UpdateLayout();
        }

        return MenuButtonResult::GetHandled();
    }

    void MenuItemScrollBox::NotifyFocusedItemChanged(const MenuFocusedItemChangedContext& context) NN_NOEXCEPT
    {
        if(util::MenuItemTree::IsAncestor(this->shared_from_this(), context.GetNewFocusedItem()))
        {
            if(auto pPage = context.GetPage())
            {
                m_pFocusTargetItem = context.GetNewFocusedItem();
                pPage->UpdateLayout();
            }
        }
    }

    //---------------------------------------

    void MenuItemScrollBox::UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT
    {
        if(m_pInnerItem)
        {
            m_pInnerItem->UpdateLayoutRecursively(context);
        }

        const auto hbarStyle = m_Style.GetHorizontalScrollbarStyle();
        const auto vbarStyle = m_Style.GetVerticalScrollbarStyle();

        auto innerLayout = CalculateInnerLayoutImpl(m_pInnerItem, m_Style, context);

        Size outerSize = {};
        Size viewSize = {};
        CalculateViewAreaSizeImpl(&outerSize, &viewSize, innerLayout.m_Size, m_Style);

        bool isHScrolling = (hbarStyle.GetMode() == ScrollbarMode_Always) && (viewSize.width  < innerLayout.m_Size.width);
        bool isVScrolling = (vbarStyle.GetMode() == ScrollbarMode_Always) && (viewSize.height < innerLayout.m_Size.height);

        Position prevPosition = m_Layout.m_InnerPosition;
        // 移動要求を処理
        if(auto pFocusTarget = m_pFocusTargetItem.lock())
        {
            prevPosition = CalculateInnerPositionToFocusTarget(prevPosition, viewSize, this->shared_from_this(), pFocusTarget);
        }
        else
        {
            prevPosition += m_RequestedInnerPositionUpdate;
        }

        auto innerPosition = CalculateInnerPositionImpl(prevPosition, innerLayout, viewSize, isHScrolling, isVScrolling, m_Style, context);

        //NN_SDK_LOG("outer-size    : (%d,%d)\n", outerSize.width, outerSize.height);
        //NN_SDK_LOG("view-size     : (%d,%d)\n", viewSize.width, viewSize.height);
        //NN_SDK_LOG("inner-size    : (%d,%d)\n", innerLayout.m_Size.width, innerLayout.m_Size.height);
        //NN_SDK_LOG("inner-position: (%d,%d)\n", innerPosition.x, innerPosition.y);
        //NN_SDK_LOG("item-position : (%d,%d)\n", innerLayout.m_ItemPosition.x, innerLayout.m_ItemPosition.y);

        if(m_pInnerItem)
        {
            m_pInnerItem->SetLayoutPosition({
                innerPosition.x + innerLayout.m_ItemPosition.x,
                innerPosition.y + innerLayout.m_ItemPosition.y,
            });
        }
        this->SetLayoutSize(outerSize);
        m_Layout.m_ViewSize = viewSize;
        m_Layout.m_InnerLayout = innerLayout;
        m_Layout.m_InnerPosition = innerPosition;
        m_Layout.m_IsHorizontallyScrolling = isHScrolling;
        m_Layout.m_IsVerticallyScrolling   = isVScrolling;

        // フォーカスを ScrollBox 自身が受け取るか決める
        {
            auto crop = Aabb::FromRectangle(util::MenuItemGeometry::GetGlobalRectangle(this->shared_from_this()));
            auto pFocusAcceptable = util::FindFocusAcceptingItem::Find(m_pInnerItem, nullptr, crop, util::FindFocusAcceptingItem::Distance);
            m_IsFocusAcceptable = (pFocusAcceptable == nullptr);
        }

        m_pFocusTargetItem.reset();
        m_RequestedInnerPositionUpdate = {};
    }

    // マージン込の InnerItem のレイアウトを計算する。
    // pInnerItem に対して UpdateLayout を予め呼んでおくこと
    MenuItemScrollBox::InnerLayout MenuItemScrollBox::CalculateInnerLayoutImpl(
        const std::shared_ptr<const IMenuItem>& pInnerItem,
        const MenuItemScrollBoxStyle& style,
        const MenuLayoutUpdateContext& context
    ) NN_NOEXCEPT
    {
        auto margin = style.GetMargin();

        Size innerSize = {};
        if(pInnerItem)
        {
            innerSize = pInnerItem->GetLayoutSize();
        }

        InnerLayout v = {};
        v.m_ItemPosition = Position{margin.left, margin.top};
        v.m_Size  = Size{
            innerSize.width  + margin.left + margin.right,
            innerSize.height + margin.top + margin.bottom
        };
        return v;

    }

    void MenuItemScrollBox::CalculateViewAreaSizeImpl(
        Size* pOutOuterSize,
        Size* pOutViewAreaSize,
        const Size& innerLayoutSize,
        const MenuItemScrollBoxStyle& style
    ) NN_NOEXCEPT
    {
        auto hbarStyle = style.GetHorizontalScrollbarStyle();
        auto vbarStyle = style.GetVerticalScrollbarStyle();
        auto sizeStyle = style.GetSizeStyle();

        Size outerSize = {};
        Size viewSize = {};

        int hbarWidth = (hbarStyle.GetMode() == ScrollbarMode_Always) ? hbarStyle.GetWidth() : 0;
        int vbarWidth = (vbarStyle.GetMode() == ScrollbarMode_Always) ? vbarStyle.GetWidth() : 0;

        // 横方向のスクロールバー
        if(hbarStyle.GetMode() == ScrollbarMode_Always ||
            sizeStyle.GetMode() == SizeMode_Fixed)
        {
            outerSize.width = sizeStyle.GetSize().width;
        }
        else // ScrollbarMode_None && SizeMode_Auto
        {
            outerSize.width = std::max(innerLayoutSize.width + vbarWidth, sizeStyle.GetSize().width);
        }

        // 縦方向のスクロールバー
        if(vbarStyle.GetMode() == ScrollbarMode_Always ||
            sizeStyle.GetMode() == SizeMode_Fixed)
        {
            outerSize.height = sizeStyle.GetSize().height;
        }
        else // ScrollbarMode_None && SizeMode_Auto
        {
            outerSize.height = std::max(innerLayoutSize.height + hbarWidth, sizeStyle.GetSize().height);
        }

        viewSize.width  = outerSize.width  - vbarWidth;
        viewSize.height = outerSize.height - hbarWidth;

        *pOutOuterSize = outerSize;
        *pOutViewAreaSize = viewSize;
    }

    Position MenuItemScrollBox::CalculateInnerPositionToFocusTarget(
        const Position& prevPosition,
        const Size& viewSize,
        const std::shared_ptr<const IMenuItem>& pSelf,
        const std::shared_ptr<const IMenuItem>& pTarget
    ) NN_NOEXCEPT
    {
        if(!util::MenuItemTree::IsAncestorOrSelf(pSelf, pTarget))
        {
            return prevPosition;
        }

        const auto myAabb = Aabb::FromRectangle({util::MenuItemGeometry::GetGlobalPosition(pSelf), viewSize});
        const auto tgAabb = Aabb::FromRectangle(util::MenuItemGeometry::GetGlobalRectangle(pTarget));

        //NN_DEVOVL_MENU_LOG_INFO("myAABB (%d_%d;%d_%d) tgAABB (%d_%d;%d_%d)\n",
        //    myAabb.xBeg, myAabb.xEnd, myAabb.yBeg, myAabb.yEnd,
        //    tgAabb.xBeg, tgAabb.xEnd, tgAabb.yBeg, tgAabb.yEnd
        //);

        Position position = prevPosition;

        // x 方向
        if(tgAabb.xBeg < myAabb.xBeg)
        {
            position.x += myAabb.xBeg - tgAabb.xBeg;
        }
        else if(tgAabb.xEnd > myAabb.xEnd)
        {
            position.x -= tgAabb.xEnd - myAabb.xEnd;
        }

        // y 方向
        if(tgAabb.yBeg < myAabb.yBeg)
        {
            position.y += myAabb.yBeg - tgAabb.yBeg;
        }
        else if(tgAabb.yEnd > myAabb.yEnd)
        {
            position.y -= tgAabb.yEnd - myAabb.yEnd;
        }

        return position;
    }


    Position MenuItemScrollBox::CalculateInnerPositionImpl(
        const Position& prevPosition,
        const InnerLayout& innerLayout,
        const Size& viewSize,
        bool isHScrolling,
        bool isVScrolling,
        const MenuItemScrollBoxStyle& style,
        const MenuLayoutUpdateContext& context
    ) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        auto alignment = style.GetAlignment();
        auto layoutSize = innerLayout.m_Size;

        Position innerPosition = prevPosition;

        // スクロールしない方向の位置を決定
        {
            auto pos = util::CalculateAlignedPosition::Calculate(layoutSize, viewSize, alignment);

            if(!isHScrolling)
            {
                innerPosition.x = pos.x;
            }

            if(!isVScrolling)
            {
                innerPosition.y = pos.y;
            }
        }

        // スクロールする方向の位置を所定の範囲に収める
        {
            if(isHScrolling)
            {
                if(innerPosition.x > 0)
                {
                    innerPosition.x = 0;
                }
                else if(innerPosition.x < viewSize.width - layoutSize.width)
                {
                    innerPosition.x = viewSize.width - layoutSize.width;
                }
            }

            if(isVScrolling)
            {
                if(innerPosition.y > 0)
                {
                    innerPosition.y = 0;
                }
                else if(innerPosition.y < viewSize.height - layoutSize.height)
                {
                    innerPosition.y = viewSize.height - layoutSize.height;
                }
            }
        }

        return innerPosition;
    }

    //---------------------------------------

    void MenuItemScrollBox::UpdatePanel() NN_NOEXCEPT
    {
        {
            auto v = util::CalculateBackgroundVisibility::Calculate(m_Style.GetBackgroundStyle(), m_pInnerItem);
            m_pPanel->SetVisibility(v.visibility);
            m_pPanel->SetColor(v.color);

            auto pos = this->GetLayoutPosition();
            auto size = this->GetLayoutSize();
            m_pPanel->SetPosition(pos.x, pos.y);
            m_pPanel->SetSize(size.width, size.height);

            m_pViewAreaPanel->SetVisibility(v.visibility);
            m_pViewAreaPanel->SetColor(v.color);
            m_pViewAreaPanel->SetPosition(0, 0);
            m_pViewAreaPanel->SetSize(m_Layout.m_ViewSize.width, m_Layout.m_ViewSize.height);
        }

        UpdateHScrollbarPanelImpl();
        UpdateVScrollbarPanelImpl();
    }

    namespace {

        std::pair<int, int> CalculateScrollbarKnobPosAndSize(
            int viewPos,
            int viewSize,
            int layoutSize,
            int barLength
        ) NN_NOEXCEPT
        {
            int viewBeg = viewPos;
            int viewEnd = viewPos + viewSize;

            int barBeg = static_cast<int>(std::floor(static_cast<float>(barLength) * viewBeg / layoutSize));
            int barEnd = static_cast<int>(std::ceil(static_cast<float>(barLength) * viewEnd / layoutSize));

            barBeg = std::max(barBeg, 0);
            barEnd = std::min(barEnd, barLength);

            if(barEnd <= barBeg)
            {
                return {};
            }

            return std::make_pair(barBeg, barEnd - barBeg);
        }
    }

    void MenuItemScrollBox::UpdateHScrollbarPanelImpl() NN_NOEXCEPT
    {
        const auto style = m_Style.GetHorizontalScrollbarStyle();
        if(style.GetMode() == ScrollbarMode_None)
        {
            m_pHorizontalBar->SetVisibility(panel::PanelVisibility::Invisible);
            m_pHorizontalKnob->SetVisibility(panel::PanelVisibility::Invisible);
            return;
        }

        const auto& layout = m_Layout;

        Position bpos = {};
        bpos.x = 0;
        bpos.y = layout.m_ViewSize.height;
        Size bsize = {};
        bsize.width = layout.m_ViewSize.width;
        bsize.height = style.GetWidth();

        m_pHorizontalBar->SetVisibility(panel::PanelVisibility::Visible);
        m_pHorizontalBar->SetPosition(bpos.x, bpos.y);
        m_pHorizontalBar->SetSize(bsize.width, bsize.height);
        m_pHorizontalBar->SetColor(style.GetBarColor());

        if(layout.m_IsHorizontallyScrolling)
        {
            int viewLength = layout.m_ViewSize.width;
            auto range = CalculateScrollbarKnobPosAndSize(-layout.m_InnerPosition.x, viewLength, layout.m_InnerLayout.m_Size.width, viewLength);

            m_pHorizontalKnob->SetVisibility(panel::PanelVisibility::Visible);
            m_pHorizontalKnob->SetPosition(range.first, bpos.y);
            m_pHorizontalKnob->SetSize(range.second, bsize.height);
            m_pHorizontalKnob->SetColor(style.GetKnobColor());
        }
        else
        {
            m_pHorizontalKnob->SetVisibility(panel::PanelVisibility::Invisible);
        }
    }

    void MenuItemScrollBox::UpdateVScrollbarPanelImpl() NN_NOEXCEPT
    {
        const auto style = m_Style.GetVerticalScrollbarStyle();
        if(style.GetMode() == ScrollbarMode_None)
        {
            m_pVerticalBar->SetVisibility(panel::PanelVisibility::Invisible);
            m_pVerticalKnob->SetVisibility(panel::PanelVisibility::Invisible);
            return;
        }

        const auto& layout = m_Layout;

        Position bpos = {};
        bpos.x = layout.m_ViewSize.width;
        bpos.y = 0;
        Size bsize = {};
        bsize.width = style.GetWidth();
        bsize.height = layout.m_ViewSize.height;

        m_pVerticalBar->SetVisibility(panel::PanelVisibility::Visible);
        m_pVerticalBar->SetPosition(bpos.x, bpos.y);
        m_pVerticalBar->SetSize(bsize.width, bsize.height);
        m_pVerticalBar->SetColor(style.GetBarColor());

        if(layout.m_IsVerticallyScrolling)
        {
            int viewLength = layout.m_ViewSize.height;
            auto range = CalculateScrollbarKnobPosAndSize(-layout.m_InnerPosition.y, viewLength, layout.m_InnerLayout.m_Size.height, viewLength);

            m_pVerticalKnob->SetVisibility(panel::PanelVisibility::Visible);
            m_pVerticalKnob->SetPosition(bpos.x, range.first);
            m_pVerticalKnob->SetSize(bsize.width, range.second);
            m_pVerticalKnob->SetColor(style.GetKnobColor());
        }
        else
        {
            m_pVerticalKnob->SetVisibility(panel::PanelVisibility::Invisible);
        }
    }

}

