﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ui_MenuItemIcon.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../framework/Framework.h"
#include "util/ui_CalculateItemSize.h"
#include "util/ui_CalculateTextSize.h"
#include "util/ui_CalculateAlignedPosition.h"

namespace ui{

    namespace {

        const char* GetIconText(Icon icon) NN_NOEXCEPT
        {
            switch(icon)
            {
            case Icon_None:
                return "";
            case Icon_LeftStick:
                return "\uE0C1";
            case Icon_RightStick:
                return "\uE0C2";
            case Icon_RightStickUpDown:
                return "\uE0C6\uE0C2\uE0C7";
            case Icon_ArrowRight:
                return "\uE090";
            case Icon_ArrowLeft:
                return "\uE091";
            case Icon_ArrowUp:
                return "\uE092";
            case Icon_ArrowDown:
                return "\uE093";
            case Icon_ArrowUpRight:
                return "\uE094";
            case Icon_ArrowDownRight:
                return "\uE095";
            case Icon_ArrowDownLeft:
                return "\uE096";
            case Icon_ArrowUpLeft:
                return "\uE097";
            case Icon_DirectionUp:
                return "\uE147";
            case Icon_DirectionDown:
                return "\uE148";
            case Icon_DirectionLeft:
                return "\uE149";
            case Icon_DirectionRight:
                return "\uE14A";
            default:
                return "?";
            }
        }

    }

    MenuItemIcon::MenuItemIcon(const MenuItemIconStyle& style, Icon icon) NN_NOEXCEPT
    {
        m_pPanel = std::make_shared<panel::PanelText>();
        m_pPanel->SetText(GetIconText(icon));

        m_Style = style;
    }

    void MenuItemIcon::UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        const auto& text = m_pPanel->GetText();
        auto margin = m_Style.GetMargin();
        Size textSize = util::CalculateTextSize::Calculate(text.c_str(), m_Style.GetIconStyle(IsEnabled(), IsFocused()).GetIconSize(), framework::TextWriterUsage_Icon);
        Size layoutSize = {
            textSize.width + margin.left + margin.right,
            textSize.height + margin.top + margin.bottom
        };

        auto outerSize = util::CalculateItemSize::Calculate([&](){ return layoutSize; }, m_Style.GetSizeStyle());

        auto layoutPosition = util::CalculateAlignedPosition::Calculate(layoutSize, outerSize, m_Style.GetAlignment());

        SetLayoutSize(outerSize);
        m_IconPosition = {layoutPosition.x + margin.left, layoutPosition.y + margin.top};
    }


    bool MenuItemIcon::IsFocusAcceptable() const NN_NOEXCEPT
    {
        return IsEnabled() && this->HasButtonCallbackFunction();
    }

    bool MenuItemIcon::IsTouchAcceptable() const NN_NOEXCEPT
    {
        return IsEnabled() && this->HasButtonCallbackFunction();
    }

    std::shared_ptr<panel::IPanel> MenuItemIcon::GetPanel() NN_NOEXCEPT
    {
        return m_pPanel;
    }

    void MenuItemIcon::UpdatePanel() NN_NOEXCEPT
    {
        auto pos = this->GetLayoutPosition();
        auto size = this->GetLayoutSize();
        m_pPanel->SetVisibility(panel::PanelVisibility::Visible);
        m_pPanel->SetColor(m_Style.GetBackgroundColor(IsEnabled(), IsFocused()));
        m_pPanel->SetPosition(pos.x, pos.y);
        m_pPanel->SetSize(size.width, size.height);

        auto iconStyle = m_Style.GetIconStyle(IsEnabled(), IsFocused());
        m_pPanel->SetTextPosition(m_IconPosition.x, m_IconPosition.y);
        m_pPanel->SetTextColor(iconStyle.GetIconColor());
        m_pPanel->SetTextSize(iconStyle.GetIconSize());
    }

}

