﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ui_MenuItemColumn.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "util/ui_FindFocusAcceptingItem.h"

namespace ui{

    MenuItemColumn::MenuItemColumn(const MenuItemTableStyle& style) NN_NOEXCEPT
        : MenuItemTableBase(style)
    {
    }

    MenuItemColumn::~MenuItemColumn() NN_NOEXCEPT
    {
    }

    void MenuItemColumn::UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT
    {
        // 子を更新しながら最大の幅を計算
        int cellWidthMax = 0;
        for(size_t i = 0; i < m_CellList.size(); i++)
        {
            UpdateCellLayout1(i, context);
            auto size = GetExpectedCellSize(i);
            cellWidthMax = std::max(cellWidthMax, size.width);
        }

        // 子の位置を設定
        int curPosY = m_Style.GetBorderStyle().GetTopWidth();
        const int posX = m_Style.GetBorderStyle().GetLeftWidth();
        for(size_t i = 0; i < m_CellList.size(); i++)
        {
            auto minSize = GetExpectedCellSize(i);
            Size cellSize = {cellWidthMax, minSize.height};
            UpdateCellLayout2(i, cellSize, context);

            m_CellList[i]->SetLayoutPosition({posX, curPosY});
            curPosY += cellSize.height + m_Style.GetBorderStyle().GetHorizontalWidth();
        }

        int width  = cellWidthMax + m_Style.GetBorderStyle().GetLeftWidth() + m_Style.GetBorderStyle().GetRightWidth();
        int height = curPosY - m_Style.GetBorderStyle().GetHorizontalWidth() + m_Style.GetBorderStyle().GetBottomWidth();

        SetLayoutSize({width, height});
    }

    MenuButtonResult MenuItemColumn::HandleUpButtonPressed(const MenuButtonHandleContext& context)    NN_NOEXCEPT
    {
        auto pChild = context.GetPreviousHandlerItem();
        if(pChild == nullptr)
        {
            return MenuButtonResult::GetNotHandled();
        }

        auto info = pChild->GetParentInfo();
        NN_SDK_ASSERT_EQUAL(this, info.pParent.lock().get());
        NN_SDK_ASSERT_RANGE(info.index, 0u, m_CellList.size());
        NN_SDK_ASSERT_EQUAL(m_CellList[info.index], pChild);

        for(int index = info.index - 1; index >= 0; index--)
        {
            if(auto pToFocus = util::FindFocusAcceptingItem::Find(m_CellList[index], context.GetCurrentFocusedItem(), util::FindFocusAcceptingItem::Distance))
            {
                context.GetPage()->ChangeFocus(pToFocus);
                return MenuButtonResult::GetHandled();
            }
        }

        // 範囲外になる場合は NotHandled
        return MenuButtonResult::GetNotHandled();
    }

    MenuButtonResult MenuItemColumn::HandleDownButtonPressed(const MenuButtonHandleContext& context)  NN_NOEXCEPT
    {
        auto pChild = context.GetPreviousHandlerItem();
        if(pChild == nullptr)
        {
            return MenuButtonResult::GetNotHandled();
        }

        auto info = pChild->GetParentInfo();
        NN_SDK_ASSERT_EQUAL(this, info.pParent.lock().get());
        NN_SDK_ASSERT_RANGE(info.index, 0u, m_CellList.size());
        NN_SDK_ASSERT_EQUAL(m_CellList[info.index], pChild);

        for(int index = info.index + 1; index < m_CellList.size(); index++)
        {
            if(auto pToFocus = util::FindFocusAcceptingItem::Find(m_CellList[index], context.GetCurrentFocusedItem(), util::FindFocusAcceptingItem::Distance))
            {
                context.GetPage()->ChangeFocus(pToFocus);
                return MenuButtonResult::GetHandled();
            }
        }

        // 範囲外になる場合は NotHandled
        return MenuButtonResult::GetNotHandled();
    }

}

