﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "../../panel/panel_PanelContainer.h"
#include "../ui_Macro.h"
#include "../ui_MenuItemTableCell.h"
#include "ui_MenuItemAttributeNoFocus.h"
#include "ui_MenuItemStyleProperty.h"

namespace ui{

    class MenuItemTableStyle
        : public
            base::MenuItemStylePropertyBorderStyle<base::StateNone,
            base::MenuItemStyleBase<MenuItemTableStyle>
            >
    {
    public:
        MenuItemTableStyle& SetDefaultCellStyle(const MenuItemTableCellStyle& value) NN_NOEXCEPT
        {
            m_DefaultCellStyle = value;
            return *this;
        }

        const MenuItemTableCellStyle& GetDefaultCellStyle() const NN_NOEXCEPT { return m_DefaultCellStyle; }

    private:
        MenuItemTableCellStyle m_DefaultCellStyle = {};
    };

}

namespace ui{ namespace base{

    template<typename TBaseType>
    class MenuItemTableBase
        : public base::MenuItemAttributeNoFocus<TBaseType>
    {
    public:
        typedef typename TBaseType::ConcreteType ConcreteType;

    public:
        explicit MenuItemTableBase(const MenuItemTableStyle& style) NN_NOEXCEPT
        {
            m_pPanel = std::make_shared<panel::PanelContainer>();
            SetStyleImpl(style);
        }

        virtual ~MenuItemTableBase() NN_NOEXCEPT
        {
            RemoveAllCell();
        }

        virtual int GetChildCount() const NN_NOEXCEPT UI_FINAL
        {
            return static_cast<int>(m_CellList.size());
        }

        virtual std::shared_ptr<IMenuItem> GetChild(int index) NN_NOEXCEPT UI_FINAL
        {
            if(index < 0 || index >= m_CellList.size())
            {
                return nullptr;
            }

            return m_CellList[index];
        }

    //@name Touch
    //@{

        virtual bool IsTouchAcceptable() const NN_NOEXCEPT UI_FINAL
        {
            return false;
        }

    //@}

    //@name Rendering
    //@{

        virtual std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT UI_FINAL
        {
            return m_pPanel;
        }

        virtual void UpdatePanel() NN_NOEXCEPT UI_FINAL
        {
            auto pos = this->GetLayoutPosition();
            auto size = this->GetLayoutSize();
            m_pPanel->SetPosition(pos.x, pos.y);
            m_pPanel->SetSize(size.width, size.height);

            switch(this->m_Style.GetBorderStyle().GetType())
            {
            case BorderType_None:
                m_pPanel->SetVisibility(panel::PanelVisibility::Transparent);
                break;
            default:
                // BorderType_Solid
                m_pPanel->SetVisibility(panel::PanelVisibility::Visible);
                m_pPanel->SetColor(this->m_Style.GetBorderStyle().GetColor());
            }
        }

    //@}

    //@name Chindren
    //@{

        std::shared_ptr<ConcreteType> AddCell(
            const MenuItemTableCellStyle& style,
            const std::shared_ptr<IMenuItem>& item
        ) NN_NOEXCEPT
        {
            auto cell = std::make_shared<MenuItemTableCell>(style);
            if(item)
            {
                cell->SetInnerItem(item);
            }

            size_t index = m_CellList.size();
            m_CellList.push_back(cell);
            cell->OnParentItemChanged(this->shared_from_this(), index);

            m_pPanel->AddChild(cell->GetPanel());
            return this->shared_from_this();
        }

        std::shared_ptr<ConcreteType> AddCell(
            const std::shared_ptr<IMenuItem>& item
        ) NN_NOEXCEPT
        {
            return AddCell(m_Style.GetDefaultCellStyle(), item);
        }

        void RemoveAllCell() NN_NOEXCEPT
        {
            for(auto& cell : m_CellList)
            {
                cell->OnParentItemChanged(nullptr, 0);
            }
            m_CellList.clear();

            m_pPanel->ClearChildren();
        }

    //@}

    protected:
        void UpdateCellLayout1(size_t index, const MenuLayoutUpdateContext& context) NN_NOEXCEPT
        {
            m_CellList[index]->UpdateInnerLayout(context);
        }

        Size GetExpectedCellSize(size_t index) const NN_NOEXCEPT
        {
            return m_CellList[index]->GetExpectedOuterSize();
        }

        void UpdateCellLayout2(size_t index, const Size& size, const MenuLayoutUpdateContext& context) NN_NOEXCEPT
        {
            m_CellList[index]->UpdateOuterLayout(size, context);
        }


    private:
        void SetStyleImpl(const MenuItemTableStyle& style) NN_NOEXCEPT
        {
            m_Style = style;
        }



    protected:
        std::vector<std::shared_ptr<MenuItemTableCell>> m_CellList;
        std::shared_ptr<panel::PanelContainer> m_pPanel;
        MenuItemTableStyle m_Style;
    };

}}
