﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneSandboxMenuPage.h"

#include <sstream>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../../ui/ui.h"

#include "../SceneIndex.h"
#include "../../ResourceUsage.h"

namespace scene{ namespace menu{

#ifdef NN_DEVOVL_ENABLE_SANDBOX_PAGE
    namespace {

        std::shared_ptr<ui::MenuItemColumn> CreateSandboxMenuItems() NN_NOEXCEPT
        {
            auto labelStyle = ui::MenuItemLabelStyle()
                //.SetSize({100, 30, 0})
                .SetBackgroundColor(ui::ItemStateFilter_Any, {0, 0, 0, 1})
                .SetBackgroundColor(ui::ItemStateFilter_InFocus, {1, 0, 0, 1})
                .SetTextStyle(ui::ItemStateFilter_Any, ui::TextStyle()
                    .SetFontColor({1, 1, 1, 1})
                    .SetFontSize(16)
                )
            ;

            auto tableStyle = ui::MenuItemTableStyle()
                .SetBorderStyle(ui::BorderStyle()
                    .SetColor({1, 1, 1})
                    .SetOuterWidth(5, 5, 5, 5)
                    .SetInnerWidth(10, 10)
                )
            ;

            auto cellStyle = ui::MenuItemTableCellStyle()
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor({0, 0, 0, 1})
                )
            ;

            auto pRoot = std::make_shared<ui::MenuItemColumn>(tableStyle)
                ->AddCell(cellStyle,
                    std::make_shared<ui::MenuItemLabel>(labelStyle, "item1")
                    ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>&)
                    {
                        NN_DEVOVL_MENU_LOG_INFO("item1 selected\n");
                    })
                )
                ->AddCell(
                    cellStyle,
                    std::make_shared<ui::MenuItemLabel>(labelStyle, "sleep")
                    ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                    {
                        NN_DEVOVL_MENU_LOG_INFO("sleeping system\n");
                        pPage->SleepSystem();
                    })
                )
                ->AddCell(
                    cellStyle,
                    std::make_shared<ui::MenuItemLabel>(labelStyle, "shutdown")
                    ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                    {
                        NN_DEVOVL_MENU_LOG_INFO("shutting down system\n");
                        pPage->ShutdownSystem();
                    })
                )
                ->AddCell(
                    cellStyle,
                    std::make_shared<ui::MenuItemLabel>(labelStyle, "close page")
                    ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                    {
                        NN_DEVOVL_MENU_LOG_INFO("closing page\n");
                        pPage->ClosePage();
                    })
                )
                ->AddCell(
                    cellStyle,
                    std::make_shared<ui::MenuItemLabel>(labelStyle, "close menu")
                    ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                    {
                        NN_DEVOVL_MENU_LOG_INFO("closing menu\n");
                        pPage->CloseMenu();
                    })
                )
                ->AddCell(
                    cellStyle,
                    std::make_shared<ui::MenuItemLabel>(labelStyle, "print resource")
                    ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                    {
                        DumpResourceUsage();
                    })
                )
            ;

            return pRoot;
        }

        std::shared_ptr<ui::MenuItemColumn> CreateRadioButtonCheckBoxItems() NN_NOEXCEPT
        {
            ui::TextStyle textStyle = {};
            textStyle
                .SetFontColor({1, 1, 1})
                .SetFontSize(16);

            ui::TextStyle textStyleSelected = {};
            textStyleSelected
                .SetFontColor({1, 0, 0})
                .SetFontSize(16);

            ui::CheckMarkStyle checkStyle = {};
            checkStyle
                .SetCheckMarkColor({.1f, .1f, .1f, 1})
                .SetCheckMarkSize(20)
                .SetBorderWidth(1);

            ui::CheckMarkStyle checkStyleSelected = {};
            checkStyleSelected
                .SetCheckMarkColor({0, 1, 0, 1})
                .SetCheckMarkSize(20)
                .SetBorderWidth(1);

            nn::util::Color4f bgColor = {0, 0, 0};
            nn::util::Color4f bgColorFocused = {.3f, .3f, .3f};

            ui::MenuItemTableStyle tableStyle = {};
            tableStyle
                .SetBorderStyle(ui::BorderStyle()
                    .SetColor(nn::util::Color4f(.5f, .5f, .5f))
                    .SetOuterWidth(1, 1, 1, 1)
                    .SetInnerWidth(1, 1)
                )
            ;
            auto cellStyle = ui::MenuItemTableCellStyle()
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor({0, 0, 0, 1})
                )
            ;
            auto pRoot = std::make_shared<ui::MenuItemColumn>(tableStyle);

            // RadioButton
            {
                ui::MenuItemRadioButtonStyle style = {};
                style
                    .SetSizeStyle(ui::SizeStyle().SetAutoSize({100, 30}))
                    .SetBackgroundColor(ui::ItemStateFilter_Any, bgColor)
                    .SetBackgroundColor(ui::ItemStateFilter_InFocus, bgColorFocused)
                    .SetTextStyle(ui::ItemStateFilter_Any, textStyle)
                    .SetTextStyle(ui::ItemStateFilter_Selected, textStyleSelected);


                {
                    auto group = std::make_shared<ui::RadioButtonGroup>();
                    for(int i = 0; i < 4; i++)
                    {
                        std::stringstream ss;
                        ss << "Radio0-" << i;
                        std::string name = ss.str();
                        pRoot->AddCell(cellStyle,
                            std::make_shared<ui::MenuItemRadioButton>(style, group, ss.str())
                            ->SetValueChangedCallbackFunction([name](bool v, auto p){
                                NN_DEVOVL_MENU_LOG_INFO("%s: change selection -> %s\n", name.c_str(), v ? "true" : "false");
                            })
                        );
                    }
                }

                {
                    auto group = std::make_shared<ui::RadioButtonGroup>();
                    auto pValue = std::make_shared<int>(0);
                    for(int i = 0; i < 4; i++)
                    {
                        std::stringstream ss;
                        ss << "Radio1-" << i;
                        std::string name = ss.str();
                        pRoot->AddCell(cellStyle,
                            std::make_shared<ui::MenuItemRadioButton>(style, group, ss.str())
                            ->SetValueAccessor(
                                [pValue, i](){ return *pValue == i; },
                                [pValue, i](bool v){ if(v){ *pValue = i; } }
                            )
                            ->SetValueChangedCallbackFunction([name](bool v, auto p){
                                NN_DEVOVL_MENU_LOG_INFO("%s: change selection -> %s\n", name.c_str(), v ? "true" : "false");
                            })
                        );
                    }
                }
            }

            // CheckBox
            {
                auto style = ui::MenuItemCheckBoxStyle()
                    .SetBackgroundColor(bgColor)
                    .SetBackgroundColor(ui::ItemStateFilter_InFocus, bgColorFocused)
                    .SetTextStyle(textStyle)
                    .SetCheckMarkStyle(checkStyle)
                    .SetCheckMarkStyle(ui::ItemStateFilter_Selected, checkStyleSelected)
                    .SetAlignment(ui::ItemAlignment_BottomRight)
                    .SetMargin({4, 8, 4, 8})
                    .SetSizeStyle(ui::SizeStyle()
                        .SetAutoSize({})
                    )
                ;
                pRoot
                    ->AddCell(cellStyle, std::make_shared<ui::MenuItemCheckBox>(style, "Check0"))
                    ->AddCell(cellStyle, std::make_shared<ui::MenuItemCheckBox>(style, "Check1"))
                ;
            }
            return pRoot;
        }

        std::shared_ptr<ui::MenuItemColumn> CreateDropDownListItems() NN_NOEXCEPT
        {
            auto style = ui::MenuItemDropDownListStyle()
                .SetSizeStyle(ui::SizeStyle()
                    .SetAutoSize({100, 40})
                )
                .SetMargin({4, 4, 4, 4})
                .SetAlignment(ui::ItemAlignment_MiddleLeft)
                .SetBackgroundColor(ui::ItemStateFilter_Any, {0, 0, 0, 1})
                .SetBackgroundColor(ui::ItemStateFilter_InFocus, {.3f, .3f, .3f, 1})
                .SetTextStyle(ui::ItemStateFilter_Any, ui::TextStyle()
                    .SetFontColor({1, 1, 1, 1})
                    .SetFontSize(16)
                )
                .SetDropDownPopupStyle(ui::MenuItemDropDownPopupStyle()
                    .SetRegion({{ScreenWidth / 4, ScreenHeight / 4}, {ScreenWidth / 2, ScreenHeight / 2}})
                    .SetVerticalScrollbarStyle(ui::ScrollbarStyle()
                        .SetMode(ui::ScrollbarMode_Always)
                        .SetBarColor({1, 1, 1, 1})
                        .SetKnobColor({0, 0, 1, 1})
                    )
                    .SetArrowButtonScrollDistance(32)
                    .SetAnalogStickScrollDistanceMax(16)
                )
            ;

            auto tableStyle = ui::MenuItemTableStyle()
                .SetBorderStyle(ui::BorderStyle()
                    .SetColor(nn::util::Color4f(.5f, .5f, .5f))
                    .SetOuterWidth(1, 1, 1, 1)
                    .SetInnerWidth(1, 1)
                )
            ;

            auto cellStyle = ui::MenuItemTableCellStyle()
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor({0, 0, 0, 1})
                )
            ;

            auto pRoot = std::make_shared<ui::MenuItemColumn>(tableStyle);

            // 2 個
            pRoot->AddCell(cellStyle, std::make_shared<ui::MenuItemDropDownList>(style)
                ->AddDropDownItem("drop1", {}, {}, {})
                ->AddDropDownItem("drop2", {}, {}, {})
            );

            // いっぱい
            {
                auto p = std::make_shared<ui::MenuItemDropDownList>(style);
                for(int i = 0; i < 40; i++)
                {
                    std::stringstream ss;
                    ss << "drop" << i;
                    p->AddDropDownItem(ss.str(), {}, {}, {});
                }
                pRoot->AddCell(cellStyle, p);
            }

            // value changed つき
            {
                auto p = std::make_shared<ui::MenuItemDropDownList>(style);
                for(int i = 0; i < 10; i++)
                {
                    std::stringstream ss;
                    ss << "drop" << i;
                    auto name = ss.str();
                    p->AddDropDownItem(ss.str(), {}, {}, [name](bool v, auto p){
                        NN_DEVOVL_MENU_LOG_INFO("%s: change selection -> %s\n", name.c_str(), v ? "true" : "false");
                    });
                }
                pRoot->AddCell(cellStyle, p);
            }


            return pRoot;
        }


        // とりあえず InnerItem を大量に作る
        std::shared_ptr<ui::MenuItemScrollBox> CreateScrollBoxItem1() NN_NOEXCEPT
        {
            auto style = ui::MenuItemScrollBoxStyle()
                .SetSizeStyle(ui::SizeStyle()
                    .SetAutoSize({150, 100})
                )
                .SetAlignment(ui::ItemAlignment_BottomRight)
                .SetHorizontalScrollbarStyle(ui::ScrollbarStyle()
                    .SetMode(ui::ScrollbarMode_Always)
                    //.SetMode(ui::ScrollbarMode_None)
                    .SetBarColor({1, .8f, .8f, 1})
                    .SetKnobColor({1, 0, 0, 1})
                )
                .SetVerticalScrollbarStyle(ui::ScrollbarStyle()
                    .SetMode(ui::ScrollbarMode_Always)
                    //.SetMode(ui::ScrollbarMode_None)
                    .SetBarColor({.8f, 1, .8f, 1})
                    .SetKnobColor({0, 1, 0, 1})
                )
                .SetArrowButtonScrollDistance(32)
                .SetAnalogStickScrollDistanceMax(16)
            ;

            auto pBox = std::make_shared<ui::MenuItemScrollBox>(style);

            auto pRow = std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle());
            for(int j = 0; j < 3; j++)
            {
                auto pColumn = std::make_shared<ui::MenuItemColumn>(ui::MenuItemTableStyle());
                for(int i = 0; i < 30; i++)
                {
                    std::stringstream ss;
                    ss << "label" << i << "-" << j;
                    auto str = ss.str();
                    auto p = std::make_shared<ui::MenuItemLabel>(ui::MenuItemLabelStyle()
                        .SetSizeStyle(ui::SizeStyle()
                            .SetFixedSize({100, 20})
                        )
                        .SetBackgroundColor({0, 0, 0, 1})
                        .SetBackgroundColor(ui::ItemStateFilter_InFocus, {.3f, 0, 0, 1})
                        .SetTextStyle(ui::TextStyle()
                            .SetFontSize(16)
                        ),
                        str
                    );
                    p->SetButtonCallbackFunction([str](const std::shared_ptr<ui::IMenuPage>&){
                        NN_DEVOVL_MENU_LOG_INFO("pressed %s\n", str.c_str());
                    });
                    pColumn->AddCell(p);
                }
                pRow->AddCell(pColumn);
            }

            pBox->SetInnerItem(pRow);
            return pBox;
        }

        // フォーカスのチェック用
        std::shared_ptr<ui::MenuItemScrollBox> CreateScrollBoxItem2() NN_NOEXCEPT
        {
            auto style = ui::MenuItemScrollBoxStyle()
                .SetSizeStyle(ui::SizeStyle()
                    .SetAutoSize({150, 100})
                )
                .SetAlignment(ui::ItemAlignment_BottomRight)
                .SetHorizontalScrollbarStyle(ui::ScrollbarStyle()
                    .SetMode(ui::ScrollbarMode_Always)
                    //.SetMode(ui::ScrollbarMode_None)
                    .SetBarColor({1, .8f, .8f, 1})
                    .SetKnobColor({1, 0, 0, 1})
                )
                .SetVerticalScrollbarStyle(ui::ScrollbarStyle()
                    .SetMode(ui::ScrollbarMode_Always)
                    //.SetMode(ui::ScrollbarMode_None)
                    .SetBarColor({.8f, 1, .8f, 1})
                    .SetKnobColor({0, 1, 0, 1})
                )
                .SetArrowButtonScrollDistance(32)
                .SetAnalogStickScrollDistanceMax(16)
            ;

            auto pBox = std::make_shared<ui::MenuItemScrollBox>(style);

            auto pRow = std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle());
            for(int j = 0; j < 3; j++)
            {
                auto pColumn = std::make_shared<ui::MenuItemColumn>(ui::MenuItemTableStyle());
                for(int i = 0; i < 3; i++)
                {
                    std::stringstream ss;
                    ss << "label" << i << "-" << j;
                    auto str = ss.str();

                    ui::Size size = {100, 20};
                    bool isFocusAcceptable = true;
                    // 真ん中は大きい
                    if(i == 1)
                    {
                        size.height = 200;
                    }
                    if(j == 1)
                    {
                        size.width = 200;
                    }

                    if(i == 1 && j == 1)
                    {
                        isFocusAcceptable = false;
                    }

                    nn::util::Color4f color = {.5f, .5f, .5f, 1};
                    if((i + j) % 2 != 0)
                    {
                        color = {.5f, .5f, 1, 1};
                    }

                    auto p = std::make_shared<ui::MenuItemLabel>(ui::MenuItemLabelStyle()
                        .SetSizeStyle(ui::SizeStyle()
                            .SetFixedSize(size)
                        )
                        .SetAlignment(ui::ItemAlignment_MiddleCenter)
                        .SetBackgroundColor(color)
                        .SetBackgroundColor(ui::ItemStateFilter_InFocus, {.3f, 0, 0, 1})
                        .SetTextStyle(ui::TextStyle()
                            .SetFontSize(16)
                        ),
                        str
                    );
                    if(isFocusAcceptable)
                    {
                        p->SetButtonCallbackFunction([str](const std::shared_ptr<ui::IMenuPage>&){
                            NN_DEVOVL_MENU_LOG_INFO("pressed %s\n", str.c_str());
                        });
                    }
                    pColumn->AddCell(
                        ui::MenuItemTableCellStyle()
                            .SetAlignment(ui::ItemAlignment_MiddleCenter)
                        ,
                        p
                    );
                }
                pRow->AddCell(pColumn);
            }

            pBox->SetInnerItem(pRow);
            return pBox;
        }

        // フォーカスのチェック用
        std::shared_ptr<ui::MenuItemScrollBox> CreateScrollBoxItem3() NN_NOEXCEPT
        {
            auto style = ui::MenuItemScrollBoxStyle()
                .SetSizeStyle(ui::SizeStyle()
                    .SetAutoSize({150, 100})
                )
                .SetAlignment(ui::ItemAlignment_BottomRight)
                .SetHorizontalScrollbarStyle(ui::ScrollbarStyle()
                    .SetMode(ui::ScrollbarMode_Always)
                    //.SetMode(ui::ScrollbarMode_None)
                    .SetBarColor({1, .8f, .8f, 1})
                    .SetKnobColor({1, 0, 0, 1})
                )
                .SetVerticalScrollbarStyle(ui::ScrollbarStyle()
                    .SetMode(ui::ScrollbarMode_Always)
                    //.SetMode(ui::ScrollbarMode_None)
                    .SetBarColor({.8f, 1, .8f, 1})
                    .SetKnobColor({0, 1, 0, 1})
                )
                .SetArrowButtonScrollDistance(32)
                .SetAnalogStickScrollDistanceMax(16)
            ;

            auto pBox = std::make_shared<ui::MenuItemScrollBox>(style);

            {
                std::stringstream ss;
                ss << "no-focused-inner-item";
                auto str = ss.str();

                ui::Size size = {250, 200};
                nn::util::Color4f color = {.5f, .5f, .5f, 1};

                auto p = std::make_shared<ui::MenuItemLabel>(ui::MenuItemLabelStyle()
                    .SetSizeStyle(ui::SizeStyle()
                        .SetFixedSize(size)
                    )
                    .SetAlignment(ui::ItemAlignment_MiddleCenter)
                    .SetBackgroundColor(color)
                    .SetBackgroundColor(ui::ItemStateFilter_InFocus, {.3f, 0, 0, 1})
                    .SetTextStyle(ui::TextStyle()
                        .SetFontSize(16)
                    ),
                    str
                );

                pBox->SetInnerItem(p);
            }

            return pBox;
        }

    }// anonymous namespace
#endif

    //------------------------------------------------

    SceneSandboxMenuPage::SceneSandboxMenuPage(const std::shared_ptr<void>& arg) NN_NOEXCEPT
        : SceneMenuPageBase(arg)
    {
#ifdef NN_DEVOVL_ENABLE_SANDBOX_PAGE
        SetPanelName("sbxmenu");

        auto tableStyle = ui::MenuItemTableStyle()
            .SetBorderStyle(ui::BorderStyle()
                .SetColor({1, 1, 1})
                .SetOuterWidth(5, 5, 5, 5)
                .SetInnerWidth(10, 10)
            )
        ;

        auto cellStyle = ui::MenuItemTableCellStyle()
            .SetBackgroundStyle(ui::BackgroundStyle()
                .SetColor({.3f, .3f, .3f, 1})
            )
        ;

        auto pRoot = std::make_shared<ui::MenuItemRow>(tableStyle);
        pRoot
            ->AddCell(cellStyle, CreateSandboxMenuItems())
            ->AddCell(cellStyle, CreateRadioButtonCheckBoxItems())
            ->AddCell(cellStyle, CreateDropDownListItems())
            ->AddCell(cellStyle, CreateScrollBoxItem1())
            ->AddCell(cellStyle, CreateScrollBoxItem2())
            ->AddCell(cellStyle, CreateScrollBoxItem3())
        ;

        SetMenuItem(pRoot);
#endif
    };

    void SceneSandboxMenuPage::OnEnteringPage(const std::shared_ptr<ui::IMenuPage>& pPage) NN_NOEXCEPT
    {
        pPage->LoadValue();
    }

}}

