﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneQuickMenuPage.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../../Config.h"
#include "../SceneIndex.h"
#include "SceneSandboxMenuPage.h"
#include "../../config/QuickMenuStyle.h"
#include "../debug/debug_SceneDebug.h"
#include "../debug/debug_LogViewer.h"

namespace scene{ namespace menu{

    SceneQuickMenuPage::SceneQuickMenuPage(const std::shared_ptr<void>& arg) NN_NOEXCEPT
        : SceneMenuPageBase(arg, ui::ItemAlignment_TopRight)
    {
        const Style style;
        SetPanelName("quick");

        auto pRow = std::make_shared<ui::MenuItemRow>(
            ui::MenuItemTableStyle()
            .SetBorderStyle(ui::BorderStyle()
                .SetType(ui::BorderType_None)
            )
            .SetDefaultCellStyle(ui::MenuItemTableCellStyle()
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetMode(ui::BackgroundMode_InnerItem)
                )
            )
        );

        pRow->AddCell(CreateLogManipulatorImpl())
            ->AddCell(CreateMenuColumnImpl());

        SetMenuItem(pRow);
    };

    void SceneQuickMenuPage::OnEnteringPage(const std::shared_ptr<ui::IMenuPage>&) NN_NOEXCEPT
    {
        if(auto pLogViewr = debug::g_SceneDebug.GetLogViewer())
        {
            pLogViewr->SetScrollbarVisibility(true);
        }
    }

    void SceneQuickMenuPage::OnClosingPage(const std::shared_ptr<ui::IMenuPage>&) NN_NOEXCEPT
    {
        if(auto pLogViewer = debug::g_SceneDebug.GetLogViewer())
        {
            pLogViewer->SetScrollbarVisibility(false);
            pLogViewer->MoveToBottom();
        }
    }

    std::shared_ptr<ui::IMenuItem> SceneQuickMenuPage::CreateMenuColumnImpl() NN_NOEXCEPT
    {
        const Style style;
        int bodyHeight = style.m_MenuHeight - style.m_MenuHeaderHeight - style.m_MenuFooterHeight;

        auto pTable = std::make_shared<ui::MenuItemColumn>(ui::MenuItemTableStyle())
            ->AddCell(
                ui::MenuItemTableCellStyle()
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetColor(style.m_BackgroundColor)
                    )
                    .SetSizeStyle(ui::SizeStyle()
                        .SetFixedSize({style.m_MenuWidth, style.m_MenuHeaderHeight})
                    )
                    .SetAlignment(ui::ItemAlignment_MiddleLeft)
                    .SetMargin(style.m_MenuHeaderMargin)
                ,
                CreateMenuColumnHeaderImpl()
            )
            ->AddCell(
                ui::MenuItemTableCellStyle()
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetColor(style.m_BackgroundColor)
                    )
                    .SetSizeStyle(ui::SizeStyle()
                        .SetFixedSize({style.m_MenuWidth, bodyHeight})
                    )
                    .SetMargin(style.m_MenuBodyMargin)
                ,
                CreateMenuColumnBodyImpl()
            )
            ->AddCell(
                ui::MenuItemTableCellStyle()
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetColor(style.m_BackgroundColor)
                    )
                    .SetSizeStyle(ui::SizeStyle()
                        .SetFixedSize({style.m_MenuWidth, style.m_MenuFooterHeight})
                    )
                    .SetMargin(style.m_MenuFooterMargin)
                ,
                CreateMenuColumnFooterImpl()
            )
        ;

        // スティック入力を LogViewer に送る
        auto pBox = std::make_shared<ui::MenuItemBox>(
                ui::MenuItemBoxStyle()
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor(style.m_BackgroundColor)
                )
            )
            ->SetInnerItem(pTable)
            ->SetRightAnalogStickInputCallbackFunction([](const nn::util::Vector3f& value)
            {
                int n = static_cast<int>(value.GetY() * 4);
                if(n > 0)
                {
                    if(auto pLogViewer = scene::debug::g_SceneDebug.GetLogViewer())
                    {
                        pLogViewer->ScrollDown(n);
                    }
                }
                else if(n < 0)
                {
                    if(auto pLogViewer = scene::debug::g_SceneDebug.GetLogViewer())
                    {
                        pLogViewer->ScrollUp(-n);
                    }
                }
            })
        ;
        return pBox;
    }

    std::shared_ptr<ui::IMenuItem> SceneQuickMenuPage::CreateMenuColumnHeaderImpl() NN_NOEXCEPT
    {
        const Style style;
        return std::make_shared<ui::MenuItemLabel>(style.MakeTitleStyle(), "QuickMenu");
    }

    std::shared_ptr<ui::IMenuItem> SceneQuickMenuPage::CreateMenuColumnFooterImpl() NN_NOEXCEPT
    {
        const Style style;
        auto p =
            std::make_shared<ui::MenuItemColumn>(
                ui::MenuItemTableStyle()
                .SetBorderStyle(ui::BorderStyle()
                    .SetType(ui::BorderType_None)
                )
            )
            // 罫線
            ->AddCell(
                ui::MenuItemTableCellStyle()
                    .SetAlignment(ui::ItemAlignment_MiddleCenter)
                    .SetSizeStyle(ui::SizeStyle()
                        .SetFixedSize({style.m_MenuWidth, style.m_MenuFooterLineSize.height})
                    )
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetColor(style.m_BackgroundColor)
                    )
                ,
                std::make_shared<ui::MenuItemBox>(ui::MenuItemBoxStyle()
                    .SetSizeStyle(ui::SizeStyle()
                        .SetFixedSize(style.m_MenuFooterLineSize)
                    )
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetColor(style.m_MenuFooterLineColor)
                    )
                )
            )
            ->AddCell(
                ui::MenuItemTableCellStyle()
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetColor(style.m_BackgroundColor)
                    )
                    .SetMargin(style.m_MenuFooterContentMargin)
                ,
                std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle())
                ->AddCell(
                    ui::MenuItemTableCellStyle()
                        .SetAlignment(ui::ItemAlignment_MiddleLeft)
                        .SetBackgroundStyle(ui::BackgroundStyle()
                            .SetMode(ui::BackgroundMode_InnerItem)
                        )
                    ,
                    std::make_shared<ui::MenuItemIcon>(style.MakeFooterIconStyle(), ui::Icon_RightStickUpDown)
                )
                ->AddCell(
                    ui::MenuItemTableCellStyle()
                        .SetAlignment(ui::ItemAlignment_MiddleLeft)
                        .SetBackgroundStyle(ui::BackgroundStyle()
                            .SetMode(ui::BackgroundMode_InnerItem)
                        )
                    ,
                    std::make_shared<ui::MenuItemLabel>(style.MakeFooterLabelStyle(), ": Scroll Log")
                )
            )
        ;

        return p;
    }

    std::shared_ptr<ui::IMenuItem> SceneQuickMenuPage::CreateMenuColumnBodyImpl() NN_NOEXCEPT
    {
        const Style style;
        auto pTable = std::make_shared<ui::MenuItemColumn>(ui::MenuItemTableStyle()
                .SetDefaultCellStyle(ui::MenuItemTableCellStyle()
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetMode(ui::BackgroundMode_InnerItem)
                    )
                )
            )
            ->AddCell(
                std::make_shared<ui::MenuItemLabel>(style.MakeItemStyle(), "Debug...")
                ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                {
                    NN_DEVOVL_MENU_LOG_INFO("opening debug menu\n");
                    pPage->OpenPage(SceneIndex_DebugMenuPage);
                })
            )
            ->AddCell(
                std::make_shared<ui::MenuItemLabel>(style.MakeItemStyle(), "Sleep")
                ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                {
                    NN_DEVOVL_MENU_LOG_INFO("sleep selected\n");
                    pPage->SleepSystem();
                })
            )
            ->AddCell(
                std::make_shared<ui::MenuItemLabel>(style.MakeItemStyle(), "Shutdown")
                ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                {
                    NN_DEVOVL_MENU_LOG_INFO("shutdown selected\n");
                    pPage->ShutdownSystem();
                })
            )
            ->AddCell(
                std::make_shared<ui::MenuItemLabel>(style.MakeItemStyle(), "Close")
                ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                {
                    NN_DEVOVL_MENU_LOG_INFO("closing quick menu\n");
                    pPage->CloseMenu();
                })
            )

#ifdef NN_DEVOVL_ENABLE_SANDBOX_PAGE
            ->AddCell(
                std::make_shared<ui::MenuItemLabel>(style.MakeItemStyle(), "(sandbox)")
                ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                {
                    NN_DEVOVL_MENU_LOG_INFO("opening sandbox\n");
                    auto pArg = std::make_shared<SceneSandboxMenuParameter>();
                    pArg->text = "text";
                    pPage->OpenPage(SceneIndex_SandboxMenuPage, pArg);
                })
            )
#endif
        ;
        return pTable;
    }

    //---------------------------------------------------

    std::shared_ptr<ui::IMenuItem> SceneQuickMenuPage::CreateLogManipulatorImpl() NN_NOEXCEPT
    {
        const Style style;
        auto p = std::make_shared<ui::MenuItemBox>(style.MakeLogManipulatorStyle())
            ->SetTouchHandlingCallbackFunctions(
                [this](const nn::util::Vector3f& position) -> void
                {
                    m_LogManipulator.Start(position);
                },
                [this](const nn::util::Vector3f&) -> void
                {
                    m_LogManipulator.End();
                },
                [this](const nn::util::Vector3f& position) -> void
                {
                    m_LogManipulator.Move(position);
                },
                [this]() -> void
                {
                    m_LogManipulator.End();
                }
            )
        ;
        return p;
    }

    //-----------------------------------

    void SceneQuickMenuPage::LogManipulator::Start(const nn::util::Vector3f& position) NN_NOEXCEPT
    {
        m_IsActive = true;
        m_PrevPosition = position;
    }

    void SceneQuickMenuPage::LogManipulator::End() NN_NOEXCEPT
    {
        m_IsActive = false;
        m_PrevPosition = {};
    }

    void SceneQuickMenuPage::LogManipulator::Move(const nn::util::Vector3f& newPosition) NN_NOEXCEPT
    {
        if(!m_IsActive)
        {
            return;
        }

        auto pLogViewer = debug::g_SceneDebug.GetLogViewer();
        if(!pLogViewer)
        {
            return;
        }

        auto delta = newPosition - m_PrevPosition;

        float deltaY = delta.GetY();
        float lineHeight = pLogViewer->GetLineHeight();

        float remainY = 0;
        if(deltaY >= 0)
        {
            int scrollLineCount = static_cast<int>(deltaY / lineHeight);
            remainY = deltaY - lineHeight * scrollLineCount;

            pLogViewer->ScrollUp(scrollLineCount);
        }
        else
        {
            int scrollLineCount = static_cast<int>(-deltaY / lineHeight);
            remainY = deltaY + lineHeight * scrollLineCount;

            pLogViewer->ScrollDown(scrollLineCount);
        }

        m_PrevPosition = newPosition - nn::util::Vector3f{0, remainY, 0};
    }


}}

