﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneDebugMenuPage.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../SceneIndex.h"
#include "../../config/DebugMenuStyle.h"
#include "../debug/debug_SceneDebug.h"
#include "../debug/debug_Activity.h"
#include "../debug/debug_LayoutEditorController.h"
#include "../debug/debug_ReadWriteActivity.h"
#include "DebugModuleConfigBuilder.h"

#include "../../Config.h"
#define NN_DEVOVL_DEBUGMENU_LOG_DEV(...) //NN_DEVOVL_MENU_LOG_ROOT("[dev]" __VA_ARGS__)

namespace scene{ namespace menu{

    SceneDebugMenuPage::SceneDebugMenuPage(const std::shared_ptr<void>& arg) NN_NOEXCEPT
        : SceneMenuPageBase(arg)
    {
        const StyleType style;
        SetPanelName("dbgmenu");

        // 先に LayoutEditor を設定する
        m_pActivity = std::make_shared<debug::Activity>(scene::debug::g_SceneDebug.GetCurrentActivity());
        m_pLayoutEditor = std::make_shared<debug::LayoutEditor>();
        //m_pIsSaveToSdCardRequested = std::make_shared<bool>(scene::debug::g_IsSavingActivityEnabled);
        m_pIsSaveToSdCardRequested = std::make_shared<bool>(false);

        SetupLayoutEditorImpl();

        // 大体こんなレイアウト
        //
        // +-------------------------------------------+
        // |Header                                     |
        // +-------------------------------------------+
        // |Body                                       |
        // |                                           |
        // |                                           |
        // |                                           |
        // |                                           |
        // +-------------------------------------------+
        // |Footer                                     |
        // +-------------------------------------------+

        auto pTable = std::make_shared<ui::MenuItemColumn>(style.MakePageTableStyle())
            ->AddCell(ui::MenuItemTableCellStyle()
                .SetAlignment(ui::ItemAlignment_MiddleLeft)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor(style.m_BackgroundColor)
                )
                .SetSizeStyle(ui::SizeStyle()
                    .SetFixedSize({style.PageHeaderWidth, style.PageHeaderHeight})
                ),
                MakePageHeaderImpl()
            )
            ->AddCell(ui::MenuItemTableCellStyle()
                .SetAlignment(ui::ItemAlignment_TopLeft)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor(style.m_BackgroundColor)
                )
                .SetSizeStyle(ui::SizeStyle()
                    .SetFixedSize({style.PageBodyWidth, style.PageBodyHeight})
                ),
                MakePageBodyImpl()
            )
            ->AddCell(ui::MenuItemTableCellStyle()
                .SetAlignment(ui::ItemAlignment_MiddleRight)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor(style.m_BackgroundColor)
                )
                .SetSizeStyle(ui::SizeStyle()
                    .SetFixedSize({style.PageFooterWidth, style.PageFooterHeight})
                ),
                MakePageFooterImpl()
            )
        ;

        auto pRoot = std::make_shared<ui::MenuItemBox>(ui::MenuItemBoxStyle()
            .SetAlignment(ui::ItemAlignment_MiddleCenter)
            .SetBackgroundStyle(ui::BackgroundStyle()
                .SetMode(ui::BackgroundMode_Color)
                .SetColor(style.m_BackgroundColor)
            )
            .SetSizeStyle(ui::SizeStyle()
                .SetFixedSize({ScreenWidth, ScreenHeight})
            )
        );
        pRoot->SetInnerItem(pTable);
        SetMenuItem(pRoot);
    }

    void SceneDebugMenuPage::OnEnteringPage(const std::shared_ptr<ui::IMenuPage>& pPage) NN_NOEXCEPT
    {
        NN_DEVOVL_DEBUGMENU_LOG_DEV("entering debug-menu\n");
        pPage->LoadValue();
    }

    void SceneDebugMenuPage::OnLeavingPage(const std::shared_ptr<ui::IMenuPage>& pPage) NN_NOEXCEPT
    {
        NN_DEVOVL_DEBUGMENU_LOG_DEV("leaving debug-menu\n");
        pPage->StoreValue();
    }

    void SceneDebugMenuPage::OnClosingPage(const std::shared_ptr<ui::IMenuPage>&) NN_NOEXCEPT
    {
        NN_DEVOVL_DEBUGMENU_LOG_DEV("closing debug-menu\n");
        if(m_pActivity)
        {
            scene::debug::g_SceneDebug.SubmitActivity(*m_pActivity);
        }

        if(m_pActivity && m_pIsSaveToSdCardRequested && *m_pIsSaveToSdCardRequested)
        {
            scene::debug::ReadWriteActivity::WriteToSdCard(*m_pActivity);
            scene::debug::g_IsSavingActivityEnabled = true;
        }
        else
        {
            scene::debug::g_IsSavingActivityEnabled = false;
        }
    }

    void SceneDebugMenuPage::SetupLayoutEditorImpl() NN_NOEXCEPT
    {
        typedef config::DebugModuleLayoutEditorStyle EditorStyle;
        const EditorStyle editorStyle;

        auto act = m_pActivity;
        m_pLayoutEditor->SetCanvasSize({EditorStyle::CanvasWidth, EditorStyle::CanvasHeight});
        m_pLayoutEditor->SetViewSize({EditorStyle::ViewWidth, EditorStyle::ViewHeight});
        m_pLayoutEditor->SetCanvasColor(editorStyle.m_CanvasColor);
        m_pLayoutEditor->SetModuleColorInactive(editorStyle.m_InactiveModuleColor);
        m_pLayoutEditor->SetModuleColorActive(editorStyle.m_ActiveModuleColor);

    #define NN_DEVOVL_DEBUGPAGE_REGISTERMODULE(name, data)  \
        m_pLayoutEditor->RegisterModule(std::shared_ptr<debug::Activity##name>(act, &act->activity##name));

        NN_DEVOVL_FOREACH_DEBUGMODULE(NN_DEVOVL_DEBUGPAGE_REGISTERMODULE, _);

    #undef NN_DEVOVL_DEBUGPAGE_REGISTERMODULE
    }

    std::shared_ptr<ui::IMenuItem> SceneDebugMenuPage::MakePageHeaderImpl() NN_NOEXCEPT
    {
        StyleType style;
        return std::make_shared<ui::MenuItemLabel>(style.MakeTitleStyle(), "DebugMenu");
    }

    std::shared_ptr<ui::IMenuItem> SceneDebugMenuPage::MakePageFooterImpl() NN_NOEXCEPT
    {
        const StyleType style;

        auto pActivity = m_pActivity;
        auto p = std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle());

        auto pOk = std::make_shared<ui::MenuItemBox>(style.MakeOkButtonBorderStyle())
            ->SetInnerItem(std::make_shared<ui::MenuItemLabel>(style.MakeOkButtonStyle(), "OK")
                ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                {
                    pPage->ClosePage();
                })
            )
        ;

        auto pSaveFlag = m_pIsSaveToSdCardRequested;
        auto pSave = std::make_shared<ui::MenuItemCheckBox>(style.MakeSaveActivityCheckBoxStyle(), "Save debug settings to SD card")
            ->SetValueAccessor(
                [pSaveFlag](){ return *pSaveFlag; },
                [pSaveFlag](bool v){ *pSaveFlag = v; }
            )
        ;

        p->AddCell(style.MakeFooterTableCellStyle(), pSave);
        p->AddCell(style.MakeFooterTableCellStyle(), pOk);
        return p;
    }

    std::shared_ptr<ui::IMenuItem> SceneDebugMenuPage::MakePageBodyImpl() NN_NOEXCEPT
    {
        const StyleType style;

        // Body の中は左右に分割
        //
        // +------------------------+-------------+
        // |                        |ConfigList   |
        // |                        |             |
        // |     LayoutEditor       |             |
        // |                        |             |
        // |                        |             |
        // +------------------------+-------------+

        auto pTable = std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle())
            ->AddCell(ui::MenuItemTableCellStyle()
                .SetAlignment(ui::ItemAlignment_MiddleCenter)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetMode(ui::BackgroundMode_InnerItem)
                )
                .SetSizeStyle(ui::SizeStyle()
                    .SetAutoSize({})
                )
                ,
                MakePageBodyLayoutEditorImpl()
            )
            ->AddCell(ui::MenuItemTableCellStyle()
                .SetAlignment(ui::ItemAlignment_TopLeft)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetMode(ui::BackgroundMode_InnerItem)
                )
                .SetSizeStyle(ui::SizeStyle()
                    .SetAutoSize({})
                )
                ,
                MakePageBodyConfigListImpl()
            )
        ;
        return pTable;
    }

    std::shared_ptr<ui::IMenuItem> SceneDebugMenuPage::MakePageBodyLayoutEditorImpl() NN_NOEXCEPT
    {
        const StyleType style;
        auto p = std::make_shared<ui::MenuItemUserCustom>(ui::MenuItemUserCustomStyle()
            .SetAlignment(ui::ItemAlignment_MiddleCenter)
            .SetBackgroundStyle(ui::BackgroundStyle()
                .SetColor(style.m_BackgroundColor)
            )
            .SetSizeStyle(ui::SizeStyle()
                .SetFixedSize({style.PageBodyLayoutEditorWidth, style.PageBodyLayoutEditorHeight})
            )
        );

        auto pLayoutEditor = m_pLayoutEditor;
        p->SetExternalPanel(m_pLayoutEditor->GetPanel());
        p->SetUpdateLayoutCallbackFunction([pLayoutEditor](){ return pLayoutEditor->UpdateLayout(); });
        p->SetUpdatePanelCallbackFunction([pLayoutEditor](const ui::Position& pos){ pLayoutEditor->UpdatePanel(pos); });
        return p;
    }

    std::shared_ptr<ui::IMenuItem> SceneDebugMenuPage::MakePageBodyConfigListImpl() NN_NOEXCEPT
    {
        const StyleType style;
        auto act = m_pActivity;
        auto pTable = std::make_shared<ui::MenuItemColumn>(ui::MenuItemTableStyle());
        // スクロールバーをつける
        auto pScroll = std::make_shared<ui::MenuItemScrollBox>(style.MakeOptionListScrollBoxStyle());
        pScroll->SetInnerItem(pTable);

        // ----------------------------------------------------
        // モジュールや設定項目を足す場合はここに追加する。
        // ----------------------------------------------------

        // Clock
        DebugModuleConfigBuilder(pTable, std::shared_ptr<debug::ActivityCommon>(act, &act->activityClock.common), m_pLayoutEditor)
            .AddOptionDropDownList("Type", std::shared_ptr<debug::ActivityClock::ClockType>(act, &act->activityClock.clockType),
                std::make_pair("SystemClock", debug::ActivityClock::ClockType_SystemClock),
                std::make_pair("SystemTick", debug::ActivityClock::ClockType_SystemTick)
            )
        ;

        // LogViewer
        DebugModuleConfigBuilder(pTable, std::shared_ptr<debug::ActivityCommon>(act, &act->activityLogViewer.common), m_pLayoutEditor)
            .AddOptionDropDownList("Mode", std::shared_ptr<uint32_t>(act, &act->activityLogViewer.isBackground),
                std::make_pair("Show", 0),
                std::make_pair("Background", 1)
            )
            .AddOptionDropDownList("Height", std::shared_ptr<int32_t>(act, &act->activityLogViewer.height),
                std::make_pair("24px", 24),     // TORIAEZU: 今のところ 1 行あたり 24px らしい
                std::make_pair("72px", 72),
                std::make_pair("120px", 120),
                std::make_pair("240px", 240),
                std::make_pair("360px", 360),
                std::make_pair("720px", 720)
            )
        ;

        // HidInputViewer
        DebugModuleConfigBuilder(pTable, std::shared_ptr<debug::ActivityCommon>(act, &act->activityHidInputViewer.common), m_pLayoutEditor)
            .AddOptionDropDownList("Scale", std::shared_ptr<debug::ActivityHidInputViewer::Scale>(act, &act->activityHidInputViewer.scale),
                std::make_pair("Small", debug::ActivityHidInputViewer::Scale_Small),
                std::make_pair("Medium", debug::ActivityHidInputViewer::Scale_Medium),
                std::make_pair("Large", debug::ActivityHidInputViewer::Scale_Large)
            )
            .AddOptionDropDownList("#Slot", std::shared_ptr<int32_t>(act, &act->activityHidInputViewer.slotCount),
                std::make_pair("1", 1),
                std::make_pair("2", 2),
                std::make_pair("3", 3),
                std::make_pair("4", 4),
                std::make_pair("5", 5)
            )
        ;

        // PowerConsumptionMeter
        DebugModuleConfigBuilder(pTable, std::shared_ptr<debug::ActivityCommon>(act, &act->activityPowerConsumptionMeter.common), m_pLayoutEditor)
        ;

#ifdef NN_DEVOVL_IS_SYSTEM
        // SystemSharedTextureViewer
        DebugModuleConfigBuilder(pTable, std::shared_ptr<debug::ActivityCommon>(act, &act->activitySystemSharedTextureViewer.common), m_pLayoutEditor)
            .AddOptionDropDownList("Scale", std::shared_ptr<debug::ActivitySystemSharedTextureViewer::Scale>(act, &act->activitySystemSharedTextureViewer.scale),
                std::make_pair("1/4", debug::ActivitySystemSharedTextureViewer::Scale_Div4),
                std::make_pair("1/8", debug::ActivitySystemSharedTextureViewer::Scale_Div8)
            )
        ;
#endif

        // ------ ここまで --------------------------------------

        return pScroll;
    }

}}

