﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneDebugMenuLayoutEditorPage.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace scene{ namespace menu{

    SceneDebugMenuLayoutEditorPage::SceneDebugMenuLayoutEditorPage(const std::shared_ptr<void>& arg) NN_NOEXCEPT
        : SceneMenuPageBase(arg)
    {
        auto pController = GetUserArgument<debug::LayoutEditorController>();
        NN_SDK_ASSERT_NOT_NULL(pController);

        auto p = MakePage(pController);
        SetMenuItem(p);
        SetPanelName("lytedit");
    }

    std::shared_ptr<ui::IMenuItem> SceneDebugMenuLayoutEditorPage::MakePage(const std::shared_ptr<debug::LayoutEditorController>& pController) NN_NOEXCEPT
    {
        const StyleType style;

        auto pScroll = std::make_shared<ui::MenuItemScrollBox>(style.MakeOptionListScrollBoxStyle());
        pScroll->SetInnerItem(MakeLayoutControlTable(pController));

        auto pOkButton = std::make_shared<ui::MenuItemBox>(style.MakeOkButtonBorderStyle())
            ->SetInnerItem(std::make_shared<ui::MenuItemLabel>(style.MakeOkButtonStyle(), "OK")
                ->SetButtonCallbackFunction([](const std::shared_ptr<ui::IMenuPage>& pPage)
                {
                    pPage->ClosePage();
                })
            )
        ;

        // LayoutEditor の部分は下に居る DebugMenu のものを表示する。
        // DebugMenu にぴったり重ねる。
        auto pRoot = std::make_shared<ui::MenuItemBox>(ui::MenuItemBoxStyle()
                .SetAlignment(ui::ItemAlignment_MiddleCenter)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetMode(ui::BackgroundMode_Transparent)
                )
                .SetSizeStyle(ui::SizeStyle()
                    .SetFixedSize({ScreenWidth, ScreenHeight})
                )
            )
            ->SetLeftAnalogStickInputCallbackFunction([pController](const nn::util::Vector3f& v){
                ui::Position dpos = { static_cast<int>(v.GetX() * StyleType::LayoutEditMoveStickDistance), static_cast<int>(v.GetY() * StyleType::LayoutEditMoveStickDistance) };
                pController->Move(dpos);
                pController->UpdatePanel();
            })
            ->SetInnerItem(
                std::make_shared<ui::MenuItemColumn>(ui::MenuItemTableStyle()
                    .SetBorderStyle(ui::BorderStyle()
                        .SetType(ui::BorderType_None)
                    )
                )
                // Header + 罫線
                ->AddCell(ui::MenuItemTableCellStyle()
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetMode(ui::BackgroundMode_Transparent)
                    )
                    .SetSizeStyle(ui::SizeStyle()
                        .SetFixedSize({StyleType::PageWidth, StyleType::PageHeaderHeight + StyleType::PageBorderLineWidth})
                    ),
                    nullptr
                )
                // Body
                ->AddCell(ui::MenuItemTableCellStyle()
                    .SetAlignment(ui::ItemAlignment_MiddleRight)
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetMode(ui::BackgroundMode_Transparent)
                    )
                    .SetSizeStyle(ui::SizeStyle()
                        .SetFixedSize({StyleType::PageWidth, StyleType::PageBodyHeight})
                    ),
                    pScroll
                )
                // 罫線
                ->AddCell(ui::MenuItemTableCellStyle()
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetMode(ui::BackgroundMode_Transparent)
                    )
                    .SetSizeStyle(ui::SizeStyle()
                        .SetFixedSize({StyleType::PageWidth, StyleType::PageBorderLineWidth})
                    ),
                    nullptr
                )
                // Footer
                ->AddCell(ui::MenuItemTableCellStyle()
                    .SetAlignment(ui::ItemAlignment_MiddleRight)
                    .SetBackgroundStyle(ui::BackgroundStyle()
                        .SetColor(style.m_BackgroundColor) // Footer は上書きする
                    )
                    .SetSizeStyle(ui::SizeStyle()
                        .SetFixedSize({StyleType::PageWidth, StyleType::PageFooterHeight})
                    ),
                    pOkButton
                )
            )
        ;
        return pRoot;
    }

    namespace {

        void AddFitModuleToCorner(const std::shared_ptr<ui::MenuItemColumn>& pTable, const std::shared_ptr<debug::LayoutEditorController>& pController) NN_NOEXCEPT
        {
            const config::DebugMenuStyle style;

            pTable->AddCell(
                style.MakeOptionModuleNameCellStyle(),
                std::make_shared<ui::MenuItemLabel>(style.MakeModuleNameLabelStyle(), "FitToCorner")
            );

            pTable->AddCell(
                ui::MenuItemTableCellStyle()
                .SetAlignment(ui::ItemAlignment_MiddleCenter)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor(style.m_BackgroundColor)
                ),
                std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle())
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowUpLeft)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                pController->SetPosition({0, 0}); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowUpRight)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto size = pController->GetSize();
                                auto csize = pController->GetCanvasSize();
                                pController->SetPosition({csize.width - size.width, 0}); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
            );
            pTable->AddCell(
                ui::MenuItemTableCellStyle()
                .SetAlignment(ui::ItemAlignment_MiddleCenter)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor(style.m_BackgroundColor)
                ),
                std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle())
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowDownLeft)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto size = pController->GetSize();
                                auto csize = pController->GetCanvasSize();
                                pController->SetPosition({0, csize.height - size.height}); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowDownRight)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto size = pController->GetSize();
                                auto csize = pController->GetCanvasSize();
                                pController->SetPosition({csize.width - size.width, csize.height - size.height}); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
            );
        }

        void AddMoveModule(const std::shared_ptr<ui::MenuItemColumn>& pTable, const std::shared_ptr<debug::LayoutEditorController>& pController) NN_NOEXCEPT
        {
            typedef config::DebugMenuStyle StyleType;
            const StyleType style;

            pTable->AddCell(
                style.MakeOptionModuleNameCellStyle(),
                std::make_shared<ui::MenuItemLabel>(style.MakeModuleNameLabelStyle(), "Move")
            );

            pTable->AddCell(
                ui::MenuItemTableCellStyle()
                .SetAlignment(ui::ItemAlignment_MiddleCenter)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor(style.m_BackgroundColor)
                ),
                std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle())
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowUpLeft)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto pos = pController->GetPosition();
                                pos.x -= StyleType::LayoutEditMoveButtonDistance;
                                pos.y -= StyleType::LayoutEditMoveButtonDistance;
                                pController->SetPosition(pos); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowUp)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto pos = pController->GetPosition();
                                pos.y -= StyleType::LayoutEditMoveButtonDistance;
                                pController->SetPosition(pos); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowUpRight)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto pos = pController->GetPosition();
                                pos.x += StyleType::LayoutEditMoveButtonDistance;
                                pos.y -= StyleType::LayoutEditMoveButtonDistance;
                                pController->SetPosition(pos); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
            );
            pTable->AddCell(
                ui::MenuItemTableCellStyle()
                .SetAlignment(ui::ItemAlignment_MiddleCenter)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor(style.m_BackgroundColor)
                ),
                std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle())
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowLeft)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto pos = pController->GetPosition();
                                pos.x -= StyleType::LayoutEditMoveButtonDistance;
                                pController->SetPosition(pos); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(),
                        std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditStickIconStyle(), ui::Icon_LeftStick)
                    )
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowRight)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto pos = pController->GetPosition();
                                pos.x += StyleType::LayoutEditMoveButtonDistance;
                                pController->SetPosition(pos); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
            );
            pTable->AddCell(
                ui::MenuItemTableCellStyle()
                .SetAlignment(ui::ItemAlignment_MiddleCenter)
                .SetBackgroundStyle(ui::BackgroundStyle()
                    .SetColor(style.m_BackgroundColor)
                ),
                std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle())
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowDownLeft)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto pos = pController->GetPosition();
                                pos.x -= StyleType::LayoutEditMoveButtonDistance;
                                pos.y += StyleType::LayoutEditMoveButtonDistance;
                                pController->SetPosition(pos); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowDown)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto pos = pController->GetPosition();
                                pos.y += StyleType::LayoutEditMoveButtonDistance;
                                pController->SetPosition(pos); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
                    ->AddCell(style.MakeLayoutEditArrowButtonCellStyle(), std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(std::make_shared<ui::MenuItemIcon>(style.MakeLayoutEditArrowButtonStyle(), ui::Icon_ArrowDownRight)
                            ->SetButtonCallbackFunction([pController](auto p)
                            {
                                auto pos = pController->GetPosition();
                                pos.x += StyleType::LayoutEditMoveButtonDistance;
                                pos.y += StyleType::LayoutEditMoveButtonDistance;
                                pController->SetPosition(pos); // Activity を直接書き換えるので StoreValue 不要
                                p->UpdatePanel();
                            })
                        )
                    )
            );
        }// NOLINT(impl/function_size)

    }

    std::shared_ptr<ui::IMenuItem> SceneDebugMenuLayoutEditorPage::MakeLayoutControlTable(const std::shared_ptr<debug::LayoutEditorController>& pController) NN_NOEXCEPT
    {
        const StyleType style;

        auto pTable = std::make_shared<ui::MenuItemColumn>(ui::MenuItemTableStyle());

        AddMoveModule(pTable, pController);
        AddFitModuleToCorner(pTable, pController);


        return pTable;
    }

}}
