﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "debug_LayoutEditor.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include "../../Config.h"

#define NN_DEVOVL_LAYOUTEDITOR_HANDLE_TO_ENTRY(varEntry, handle, result) \
    if(handle.value < 0 || handle.value >= m_EntryList.size())  \
    {                                                           \
        return result;                                          \
    }                                                           \
    auto& varEntry = m_EntryList[handle.value];

namespace scene{ namespace debug{

    LayoutEditor::LayoutEditor() NN_NOEXCEPT
    {
        m_pContainer = std::make_shared<panel::PanelContainer>();
        m_pContainer->SetPanelName("dbglyt");
    }

    void LayoutEditor::SetViewSize(const ui::Size& size) NN_NOEXCEPT
    {
        m_ViewSize = size;
    }

    void LayoutEditor::SetCanvasSize(const ui::Size& size) NN_NOEXCEPT
    {
        m_CanvasSize = size;
    }

    void LayoutEditor::SetCanvasColor(const nn::util::Color4f& value) NN_NOEXCEPT
    {
        m_CanvasColor = value;
    }

    void LayoutEditor::SetModuleColorInactive(const nn::util::Color4f& value) NN_NOEXCEPT
    {
        m_ModuleColorInactive = value;
    }

    void LayoutEditor::SetModuleColorActive(const nn::util::Color4f& value) NN_NOEXCEPT
    {
        m_ModuleColorActive = value;
    }

    ui::Size LayoutEditor::GetCanvasSize() const NN_NOEXCEPT
    {
        return m_CanvasSize;
    }

    LayoutEditor::ModuleHandle LayoutEditor::RegisterModule(
        const std::shared_ptr<ActivityCommon>& pCommon,
        const std::function<ui::Size ()>& sizeGetterFunction
    ) NN_NOEXCEPT
    {
        size_t index = m_EntryList.size();

        Entry e = {};
        e.pCommon = pCommon;
        e.sizeGetterFunction = sizeGetterFunction;
        e.pPanel = std::make_shared<panel::PanelText>();
        m_EntryList.push_back(e);

        return {static_cast<int64_t>(index)};
    }

    LayoutEditor::ModuleHandle LayoutEditor::FindModule(const char* name) const NN_NOEXCEPT
    {
        for(size_t i = 0 ; i < m_EntryList.size(); i++)
        {
            auto& e = m_EntryList[i];
            if(0 == nn::util::Strncmp(e.pCommon->name.value, name, DebugModuleNameSize))
            {
                return { static_cast<int64_t>(i) };
            }
        }

        return {};
    }

    ui::Size LayoutEditor::GetModuleSize(ModuleHandle h) const NN_NOEXCEPT
    {
        NN_DEVOVL_LAYOUTEDITOR_HANDLE_TO_ENTRY(e, h, {});
        return e.sizeGetterFunction();
    }

    ui::Position LayoutEditor::GetModulePosition(ModuleHandle h) const NN_NOEXCEPT
    {
        NN_DEVOVL_LAYOUTEDITOR_HANDLE_TO_ENTRY(e, h, {});
        return {static_cast<int>(e.pCommon->positionX), static_cast<int>(e.pCommon->positionY)};
    }

    void LayoutEditor::SetModulePosition(ModuleHandle h, const ui::Position& pos) NN_NOEXCEPT
    {
        NN_DEVOVL_LAYOUTEDITOR_HANDLE_TO_ENTRY(e, h,;);
        e.pCommon->positionX = pos.x;
        e.pCommon->positionY = pos.y;
    }

    void LayoutEditor::MoveModule(ModuleHandle h, const ui::Position& dpos) NN_NOEXCEPT
    {
        NN_DEVOVL_LAYOUTEDITOR_HANDLE_TO_ENTRY(e, h,;);

        // TORIAEZU: 単に足す
        e.pCommon->positionX += dpos.x;
        e.pCommon->positionY += dpos.y;
    }

    void LayoutEditor::SetModuleActivity(ModuleHandle h, bool value) NN_NOEXCEPT
    {
        NN_DEVOVL_LAYOUTEDITOR_HANDLE_TO_ENTRY(e, h,;);
        e.isActive = value;
    }

    ui::Size LayoutEditor::UpdateLayout() NN_NOEXCEPT
    {
        return m_ViewSize;
    }

    void LayoutEditor::UpdatePanel(const ui::Position& position) NN_NOEXCEPT
    {
        UpdateInternalPanel();
        m_pContainer->SetPosition(position.x, position.y);
    }

    void LayoutEditor::UpdateInternalPanel() NN_NOEXCEPT
    {
        std::vector<std::shared_ptr<panel::IPanel>> panelList;

        auto pushModule = [&](Entry& e)
        {
            // 無効なら無視
            if(!e.pCommon->isEnabled)
            {
                return;
            }

            // サイズ 0 なら無視
            auto size = e.sizeGetterFunction();
            if(size.width <= 0 || size.height <= 0)
            {
                return;
            }
            ui::Position pos = {e.pCommon->positionX, e.pCommon->positionY};

            // 表示サイズにスケーリング
            ui::Position vpos  = {
                pos.x * m_ViewSize.width  / m_CanvasSize.width,
                pos.y * m_ViewSize.height / m_CanvasSize.height,
            };
            ui::Size     vsize = {
                size.width  * m_ViewSize.width  / m_CanvasSize.width,
                size.height * m_ViewSize.height / m_CanvasSize.height,
            };

            e.pPanel->SetVisibility(panel::PanelVisibility::Visible);
            e.pPanel->SetPosition(vpos.x, vpos.y);
            e.pPanel->SetSize(vsize.width, vsize.height);
            e.pPanel->SetColor(e.isActive ? m_ModuleColorActive : m_ModuleColorInactive);

            NN_DEVOVL_DEBUG_LOG_DEV_LAYOUTEDITOR("c(%d,%d;%dx%d) -> v(%d,%d;%dx%d) %s\n", pos.x, pos.y, size.width, size.height, vpos.x, vpos.y, vsize.width, vsize.height, e.pCommon->name);

            panelList.push_back(e.pPanel);
        };

        // 先に Inactive
        for(auto& e : m_EntryList)
        {
            if(!e.isActive)
            {
                pushModule(e);
            }
        }

        // 次に Active
        for(auto& e : m_EntryList)
        {
            if(e.isActive)
            {
                pushModule(e);
            }
        }

        m_pContainer->SetChildren(panelList);
        m_pContainer->SetVisibility(panel::PanelVisibility::Visible);
        m_pContainer->SetColor(m_CanvasColor);
        m_pContainer->SetSize(m_ViewSize.width, m_ViewSize.height);
    }

    std::shared_ptr<panel::IPanel> LayoutEditor::GetPanel() NN_NOEXCEPT
    {
        return m_pContainer;
    }


}}

