﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/util/util_Vector.h>

namespace panel
{

    struct PanelRectangle
    {
    public:
        PanelRectangle() NN_NOEXCEPT
        {
            this->x = 0;
            this->y = 0;
            this->width = 0;
            this->height = 0;
        }

        PanelRectangle(int x, int y, int w, int h) NN_NOEXCEPT
        {
            this->x = x;
            this->y = y;
            this->width = w;
            this->height = h;
        }

        bool operator==(const PanelRectangle& v) const NN_NOEXCEPT
        {
            return
                this->x == v.x &&
                this->y == v.y &&
                this->width == v.width &&
                this->height == v.height;
        }

        bool operator!=(const PanelRectangle& v) const NN_NOEXCEPT
        {
            return !(*this == v);
        }

        // 空集合かを取得します。
        bool IsEmpty() const NN_NOEXCEPT
        {
            return (this->width <= 0 || this->height <= 0);
        }

        // 積集合を取得します。
        PanelRectangle Product(const PanelRectangle& rect) const NN_NOEXCEPT
        {
            PanelRectangle result = {};
            {
                result.x = std::max(this->x, rect.x);
                result.width = std::min(this->x + this->width, rect.x + rect.width) - result.x;
                if(result.width < 0)
                {
                    result.width = 0;
                }
            }
            {
                result.y = std::max(this->y, rect.y);
                result.height = std::min(this->y + this->height, rect.y + rect.height) - result.y;
                if(result.height < 0)
                {
                    result.height = 0;
                }
            }
            return result;
        }


        bool Contains(const nn::util::Vector3f& point) const NN_NOEXCEPT
        {
            int x = point.GetX();
            int y = point.GetY();
            return (x >= this->x) && (x <= this->x + this->width) && (y >= this->y) && (y <= this->y + this->height);
        }

    public:
        int x;
        int y;
        int width;
        int height;
    };

}
