﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#define NN_DEVOVL_HID_FOREACH_DEBUGPAD_BUTTON(macro)    \
    macro(A)        \
    macro(B)        \
    macro(X)        \
    macro(Y)        \
    macro(L)        \
    macro(R)        \
    macro(ZL)       \
    macro(ZR)       \
    macro(Left)     \
    macro(Up)       \
    macro(Right)    \
    macro(Down)     \
//

#define NN_DEVOVL_HID_FOREACH_JOY_BUTTON(macro)    \
    NN_DEVOVL_HID_FOREACH_DEBUGPAD_BUTTON(macro)    \
    macro(Plus)     \
    macro(Minus)    \
    macro(StickL)   \
    macro(StickR)   \
    macro(LeftSL)   \
    macro(LeftSR)   \
    macro(RightSL)  \
    macro(RightSR)  \
//


#define NN_DEVOVL_HID_FOREACH_BUTTON(macro) \
    NN_DEVOVL_HID_FOREACH_JOY_BUTTON(macro)
//

static const int HidNpadCountMax = 9;

enum HidNpadStyle
{
    HidNpadStyle_None = 0, // コントローラが無効
    HidNpadStyle_FullKey,               // ProController
    HidNpadStyle_Handheld,              // 携帯機モード

    HidNpadStyle_JoyDual,               // JoyCon 2 本持ち
    HidNpadStyle_JoyDualLeftOnly,       // JoyCon 2 本持ちの左のみ繋がっている状態
    HidNpadStyle_JoyDualRightOnly,      // JoyCon 2 本持ちの右のみ繋がっている状態

    HidNpadStyle_JoyLeft,       // 左 JoyCon
    HidNpadStyle_JoyRight,      // 右 JoyCon
};


template<typename ValueType>
struct HidButtons
{
#define NN_DEVOVL_HID_BUTTON_MEMBER(btn) ValueType btn;

    NN_DEVOVL_HID_FOREACH_BUTTON(NN_DEVOVL_HID_BUTTON_MEMBER)

#undef NN_DEVOVL_HID_BUTTON_MEMBER
};

struct HidAnalogStickValue
{
    float x;
    float y;
};

struct HidAnalogSticks
{
    HidAnalogStickValue stickL;
    HidAnalogStickValue stickR;
};

struct HidNpadState
{
    HidNpadStyle m_Style = HidNpadStyle_None;
    HidButtons<int> m_Button = {};
    HidAnalogSticks m_AnalogStick = {};
    nn::Bit8 m_LedPattern = {};
};


struct HidButtonState
{
    // 押されている間 true になります。
    HidButtons<bool> isPressed;
    // 押されはじめのフレームと押されている間一定フレーム間隔で true になります。
    HidButtons<bool> isTriggered;
    // 押されはじめたフレームのみ true になります。
    HidButtons<bool> isDown;
    // 離されたフレームのみ true になります。
    HidButtons<bool> isUp;

    // アナログスティック入力状態です。
    //
    // 入力がない場合 0
    // x ... 右に倒したときに正
    // y ... 下に倒したときに正(nn::hid で取得できる値とは逆)
    //
    // 最大まで倒した際に絶対値が 1 になります。
    // 複数のコントローラから入力があった場合、すべての入力値の合計を返します。
    // ただし、合計した入力 (x, y) について x^2 + y^2 > 1 となる場合、 x^2 + y^2 == 1 になるように正規化されます。
    HidAnalogSticks analogSticks;
};

struct HidButtonHandleParameter
{
    // 長押しによる連打判定を開始するフレーム数
    // 0 以下の場合、連打判定しない。
    int repeatTriggerFrameBeginCount;

    // 長押しによる連打のフレーム間隔。
    // 1 のとき毎フレーム連打になる。
    int repeatTriggerFrameInterval;
};

struct HidTouchState
{
    int touchPositionX = 0;
    int touchPositionY = 0;
    bool isHeld = false;
    bool isPressed = false;
    bool isReleased = false;
};

class Hid
{
public:
    static void Initialize() NN_NOEXCEPT;
    static void Finalize() NN_NOEXCEPT;
    static void SetParameter(const HidButtonHandleParameter& param) NN_NOEXCEPT;
    static void Update() NN_NOEXCEPT;

    static HidButtonState GetButtonState() NN_NOEXCEPT;
    static HidTouchState GetTouchState() NN_NOEXCEPT;

    static int GetNpadCount() NN_NOEXCEPT;
    static HidNpadState GetNpadState(int index) NN_NOEXCEPT;
};

