﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <queue>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>

class ThreadMessageResultReporter;
class ThreadMessageResultReceiver;

typedef uint64_t ThreadMessageType;

enum ThreadMessageResult
{
    ThreadMessageResult_Success,
    ThreadMessageResult_Failure,
    ThreadMessageResult_Aborted,
};

class ThreadMessageResultReceiver
{
public:
    friend ThreadMessageResultReporter;
public:
    explicit ThreadMessageResultReceiver(const std::shared_ptr<void>& pUserData) NN_NOEXCEPT;
    virtual ~ThreadMessageResultReceiver() NN_NOEXCEPT;

    bool TryWaitComplete() NN_NOEXCEPT;
    void WaitCompletion() NN_NOEXCEPT;
    void InitializeCompletionMultiWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

    ThreadMessageResult GetResult() const NN_NOEXCEPT;
    std::shared_ptr<void> GetUserData() const NN_NOEXCEPT;

private:
    void NotifyCompletion(ThreadMessageResult result) NN_NOEXCEPT;

private:
    bool m_IsComplete;
    ThreadMessageResult m_Result;
    nn::os::EventType m_CompleteEvent;
    std::shared_ptr<void> m_pUserData;
};

class ThreadMessageResultReporter
{
public:
    explicit ThreadMessageResultReporter(const std::shared_ptr<ThreadMessageResultReceiver>& pReceiver) NN_NOEXCEPT;
    ~ThreadMessageResultReporter() NN_NOEXCEPT;

    std::shared_ptr<void> GetUserData() const NN_NOEXCEPT;
    void NotifyCompletion(ThreadMessageResult result) NN_NOEXCEPT;

private:
    bool m_IsResultNotified;
    std::shared_ptr<ThreadMessageResultReceiver> m_pReceiver;
};

class ThreadMessageChannel
{
public:
    explicit ThreadMessageChannel(int messageCountMax) NN_NOEXCEPT;
    ~ThreadMessageChannel() NN_NOEXCEPT;

    // メッセージを送信する。送信するまでブロックする。
    std::shared_ptr<ThreadMessageResultReceiver> SendMessage(ThreadMessageType message, const std::shared_ptr<void>& pUserData) NN_NOEXCEPT;

    // メッセージを受信する。受信するまでブロックする。
    void ReceiveMessage(ThreadMessageType* pOutMessage, std::shared_ptr<ThreadMessageResultReporter>* pOutReporter) NN_NOEXCEPT;

    // メッセージを受信する。なにもなければ false が返る
    bool TryReceiveMessage(ThreadMessageType* pOutMessage, std::shared_ptr<ThreadMessageResultReporter>* pOutReporter) NN_NOEXCEPT;


    void IntializeReceivableEventHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

private:
    bool IsMessageQueueEmpty() const NN_NOEXCEPT;
    bool IsMessageQueueFull() const NN_NOEXCEPT;

private:
    struct MessageEntry
    {
        ThreadMessageType message;
        std::shared_ptr<ThreadMessageResultReporter> pReporter;
    };

    int m_MessageCountMax;
    std::queue<MessageEntry> m_MessageQueue;

    nn::os::MutexType m_Mutex;
    nn::os::ConditionVariableType m_Condition;
    nn::os::EventType m_ReceivableEvent;
};
