﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AppletMessageThread.h"

#include <nn/ae.h>
#include <nn/nn_SdkLog.h>
#include <nn/ae/ae_OverlayAppletApi.h>
#include "Config.h"
#include "scene/SceneIndex.h"
#include "framework/HidHandlingModeManager.h"

namespace {
    static const uintptr_t MultiWaitTag_AppletMessage = 1;
    static const uintptr_t MultiWaitTag_MenuComplete = 2;
}

namespace {
    class ResultWaiter
    {
    public:
        // multiWaitHolderInitFunciton(&holder)
        template<typename F>
        void Activate(
            nn::os::MultiWaitType* pMultiWait,
            uintptr_t multiWaitTag,
            std::shared_ptr<ThreadMessageResultReceiver> pReceiver,
            F multiWaitHolderInitFunction
        ) NN_NOEXCEPT
        {
            m_MultiWaitTag = multiWaitTag;
            m_pReceiver = pReceiver;

            multiWaitHolderInitFunction(&m_MultiWaitHolder);
            m_MultiWaitHolder.userData = multiWaitTag;
            nn::os::LinkMultiWaitHolder(pMultiWait, &m_MultiWaitHolder);
        }

        void Deactivate() NN_NOEXCEPT
        {
            if(!m_pReceiver)
            {
                return;
            }
            nn::os::UnlinkMultiWaitHolder(&m_MultiWaitHolder);
            m_pReceiver = nullptr;
        }

        std::shared_ptr<ThreadMessageResultReceiver> GetReceiver() NN_NOEXCEPT
        {
            return m_pReceiver;
        }

        bool IsActive() const NN_NOEXCEPT
        {
            return m_pReceiver != nullptr;
        }

    private:
        uintptr_t m_MultiWaitTag;
        std::shared_ptr<ThreadMessageResultReceiver> m_pReceiver;
        nn::os::MultiWaitHolderType m_MultiWaitHolder;
    };
}


namespace {
    void ProcessRequestToPrepareSleepMessage(ThreadMessageChannel* pOvldChannel) NN_NOEXCEPT
    {
        NN_DEVOVL_APPM_LOG("Received RequestToPrepareSleep\n");

        // スレッドをサスペンド
        {
            auto r = pOvldChannel->SendMessage(AppletMessage_RequestSuspend, nullptr);
            r->WaitCompletion();
        }

        NN_DEVOVL_APPM_LOG("Suspend all threads complete\n");

        // スリープ準備完了＆復帰待ち
        nn::ae::AllowToEnterSleepAndWait();

        NN_DEVOVL_APPM_LOG("Wake all threads\n");
        // スレッドを起こす
        {
            pOvldChannel->SendMessage(AppletMessage_RequestResume, nullptr);
        }
        NN_DEVOVL_APPM_LOG("Wake all threads complete\n");
    }

    void ProcessExitMessage(ThreadMessageChannel* pOvldChannel, ThreadMessageChannel* pOvlnChannel) NN_NOEXCEPT
    {
        NN_DEVOVL_APPM_LOG("Received Exit\n");
        // スレッドを終了
        {
            auto r = pOvldChannel->SendMessage(AppletMessage_RequestExit, nullptr);
            r->WaitCompletion();
        }
        {
            auto r = pOvlnChannel->SendMessage(AppletMessage_RequestExit, nullptr);
            r->WaitCompletion();
        }
    }

    void ProcessLaunchMenuMessage(ThreadMessageChannel* pOvldChannel, ResultWaiter* pMenuResultWaiter, nn::os::MultiWaitType* pMultiWait, AppletMessage message) NN_NOEXCEPT
    {
        if(pMenuResultWaiter->IsActive())
        {
            NN_DEVOVL_APPM_LOG("  Menu is already launched\n");
            return;
        }
        auto pResponse = std::make_shared<AppletMessageMenuResponse>();
        framework::g_HidHandlingModeManager.BeginInteraction();
        auto pReceiver = pOvldChannel->SendMessage(message, pResponse);
        pMenuResultWaiter->Activate(
            pMultiWait,
            MultiWaitTag_MenuComplete,
            pReceiver,
            [&](nn::os::MultiWaitHolderType* p){ pReceiver->InitializeCompletionMultiWaitHolder(p); }
        );
    }

    void ProcessCloseMenuMessage(ThreadMessageChannel* pOvldChannel) NN_NOEXCEPT
    {
        pOvldChannel->SendMessage(AppletMessage_CloseMenu, nullptr);
    }

    void ProcessMenuResponse(ResultWaiter* pMenuResultWaiter, AppletMessageThreadParameter* pParam, nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
    {
        NN_DEVOVL_APPM_LOG("Received Menu Response\n");
        // Menu が終了した
        NN_SDK_ASSERT(pMenuResultWaiter->IsActive());

        NN_ABORT_UNLESS(pMenuResultWaiter->GetReceiver()->TryWaitComplete());


        // Sleep や Shotdown は電源ボタンのエミュレーションで実現するので
        // 先に Hid 入力の受け付けを止める（止めないと自分に通知が来る）
        framework::g_HidHandlingModeManager.EndInteraction();

        auto pReceiver = pMenuResultWaiter->GetReceiver();
        auto result = pReceiver->GetResult();

        //メニューを終了させておかないと別メニューを起動できない
        pMenuResultWaiter->Deactivate();

        if(result == ThreadMessageResult_Success)
        {
            auto pResponse = std::static_pointer_cast<AppletMessageMenuResponse>(pReceiver->GetUserData());
            NN_SDK_ASSERT_NOT_NULL(pResponse);
            auto code = pResponse->GetResultCode();

            if(code == AppletMessageResponseMenuResultCode_Sleep)
            {
                nn::ae::NotifyShortPressingPowerButtonForDebug();
            }
            else if(code == AppletMessageResponseMenuResultCode_Shutdown)
            {
                nn::ae::NotifyLongPressingPowerButtonForDebug();
            }
        }
    }

    void ProcessShowApplicationLogoMessage(ThreadMessageChannel* pOvldChannel) NN_NOEXCEPT
    {
        auto pInfo = std::make_shared<AppletMessageApplicationLogoInfo>();
        pInfo->SetApplicationId(nn::ae::GetApplicationIdForLogo());
        pOvldChannel->SendMessage(AppletMessage_ShowApplicationLogo, pInfo);
    }

    void ProcessHideApplicationLogoMessage(ThreadMessageChannel* pOvldChannel) NN_NOEXCEPT
    {
        auto r = pOvldChannel->SendMessage(AppletMessage_HideApplicationLogo, nullptr);
        r->WaitCompletion();
    }

    void ProcessForceHideApplicationLogoMessage(ThreadMessageChannel* pOvldChannel) NN_NOEXCEPT
    {
        auto r = pOvldChannel->SendMessage(AppletMessage_ForceHideApplicationLogo, nullptr);
        r->WaitCompletion();
    }

    void ProcessShutdownMessage() NN_NOEXCEPT
    {
        framework::g_HidHandlingModeManager.EndInteraction();
        nn::ae::NotifyLongPressingPowerButtonForDebug();
    }

    //----------------------------------------------------------------

    void DispatchAppletMessage(nn::ae::Message message, AppletMessageThreadParameter* pParam, ResultWaiter* pMenuResultWaiter, nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
    {
        switch(message)
        {
        case nn::ae::Message_RequestToPrepareSleep:
            ProcessRequestToPrepareSleepMessage(pParam->pToOverlayDisplayChannel);
            break;
        case nn::ae::Message_DetectShortPressingHomeButton:
            NN_DEVOVL_APPM_LOG("Received DetectShortPressingHomeButton\n");
            if(pMenuResultWaiter->IsActive())
            {
                ProcessCloseMenuMessage(pParam->pToOverlayDisplayChannel);
            }
            break;
        case nn::ae::Message_DetectLongPressingHomeButton:
            NN_DEVOVL_APPM_LOG("Received DetectLongPressingHomeButton\n");
            ProcessLaunchMenuMessage(pParam->pToOverlayDisplayChannel, pMenuResultWaiter, pMultiWait, AppletMessage_LaunchQuickMenu);
            break;
        case nn::ae::Message_DetectMiddlePressingPowerButton:
            NN_DEVOVL_APPM_LOG("Received DetectMiddlePressingPowerButton\n");
            ProcessLaunchMenuMessage(pParam->pToOverlayDisplayChannel, pMenuResultWaiter, pMultiWait, AppletMessage_LaunchPowerMenu);
            break;
        case nn::ae::Message_OperationModeChanged:
            NN_DEVOVL_APPM_LOG("Received DetectOperationModeChanged\n");
            break;
        case nn::ae::Message_ShowApplicationLogo:
            NN_DEVOVL_APPM_LOG("Received ShowApplicationLogo\n");
            ProcessShowApplicationLogoMessage(pParam->pToOverlayDisplayChannel);
            break;
        case nn::ae::Message_HideApplicationLogo:
            NN_DEVOVL_APPM_LOG("Received HideApplicationLogo\n");
            ProcessHideApplicationLogoMessage(pParam->pToOverlayDisplayChannel);
            break;
        case nn::ae::Message_ForceHideApplicationLogo:
            NN_DEVOVL_APPM_LOG("Received ForceHideApplicationLogo\n");
            ProcessForceHideApplicationLogoMessage(pParam->pToOverlayDisplayChannel);
            break;
        default:
            NN_DEVOVL_APPM_LOG("Received UnknownMessage(0x%08X)\n", static_cast<uint32_t>(message));
        }
    }

    void DispatchAppletMessageForExhibitionMode(nn::ae::Message message, AppletMessageThreadParameter* pParam, ResultWaiter* pMenuResultWaiter, nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
    {
        NN_UNUSED(pMenuResultWaiter);
        NN_UNUSED(pMultiWait);
        switch(message)
        {
        case nn::ae::Message_RequestToPrepareSleep:
            ProcessRequestToPrepareSleepMessage(pParam->pToOverlayDisplayChannel);
            break;
        case nn::ae::Message_DetectShortPressingHomeButton:
            NN_DEVOVL_APPM_LOG("Received DetectShortPressingHomeButton\n");
            break;
        case nn::ae::Message_DetectLongPressingHomeButton:
            NN_DEVOVL_APPM_LOG("Received DetectLongPressingHomeButton\n");
            break;
        case nn::ae::Message_DetectMiddlePressingPowerButton:
            NN_DEVOVL_APPM_LOG("Received DetectMiddlePressingPowerButton\n");
            // 電源ボタン中押しでシャットダウンにする
            ProcessShutdownMessage();
            break;
        case nn::ae::Message_OperationModeChanged:
            NN_DEVOVL_APPM_LOG("Received DetectOperationModeChanged\n");
            break;
        case nn::ae::Message_ShowApplicationLogo:
            NN_DEVOVL_APPM_LOG("Received ShowApplicationLogo\n");
            break;
        case nn::ae::Message_HideApplicationLogo:
            NN_DEVOVL_APPM_LOG("Received HideApplicationLogo\n");
            break;
        case nn::ae::Message_ForceHideApplicationLogo:
            NN_DEVOVL_APPM_LOG("Received ForceHideApplicationLogo\n");
            break;
        default:
            NN_DEVOVL_APPM_LOG("Received UnknownMessage(0x%08X)\n", static_cast<uint32_t>(message));
        }
    }
}

void AppletMessageThreadFunction(void* pArg) NN_NOEXCEPT
{
    NN_DEVOVL_APPM_LOG("AppletMessageThread Start\n");
    auto pParam = reinterpret_cast<AppletMessageThreadParameter*>(pArg);

    RunMode* pRunMode = pParam->pRunMode;
    NN_SDK_ASSERT_NOT_NULL(pRunMode);

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);
    NN_UTIL_SCOPE_EXIT{ nn::os::FinalizeMultiWait(&multiWait); };

    nn::os::MultiWaitHolderType appletMessageHolder;
    nn::os::InitializeMultiWaitHolder(&appletMessageHolder, pParam->pMessageReceivedEvent);
    appletMessageHolder.userData = MultiWaitTag_AppletMessage;
    nn::os::LinkMultiWaitHolder(&multiWait, &appletMessageHolder);
    NN_UTIL_SCOPE_EXIT{
        nn::os::UnlinkMultiWaitHolder(&appletMessageHolder);
        nn::os::FinalizeMultiWaitHolder(&appletMessageHolder);
    };

    ResultWaiter menuResultWaiter;
    NN_UTIL_SCOPE_EXIT{ menuResultWaiter.Deactivate(); };

    for(;;)
    {
        auto pSignaled = nn::os::WaitAny(&multiWait);

        if(pSignaled->userData == MultiWaitTag_AppletMessage)
        {
            auto message = nn::ae::WaitForNotificationMessage(pParam->pMessageReceivedEvent);
            // 終了メッセージを処理
            if(message == nn::ae::Message_Exit)
            {
                ProcessExitMessage(pParam->pToOverlayDisplayChannel, pParam->pToOverlayNotificationChannel);
                goto EXIT;
            }
            // その他のメッセージを処理
            if(pRunMode->IsExhibitionMode())
            {
                DispatchAppletMessageForExhibitionMode(message, pParam, &menuResultWaiter, &multiWait);
            }
            else
            {
                DispatchAppletMessage(message, pParam, &menuResultWaiter, &multiWait);
            }
        }
        else if(pSignaled->userData == MultiWaitTag_MenuComplete)
        {
            ProcessMenuResponse(&menuResultWaiter, pParam, &multiWait);
        }
        else
        {
            NN_ABORT("[devovl][appm] unknown signal (tag=%llX)\n", static_cast<uint64_t>(pSignaled->userData));
        }
    }

EXIT:
    NN_DEVOVL_APPM_LOG("AppletMessageThread Exit\n");
}
