﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm>
#include <memory>

#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

#include "DevMenu_SaveDataListScene.h"
#include "../Applications/DevMenu_ApplicationsCommon.h"
#include "../DevMenu_Result.h"

//#define DEBUG_DATA

namespace devmenu { namespace savedata {

namespace {
    const int MaxApplicationCount = 256;
    const int MaxSaveDataIdCount = 32;

} // end of anonymous namespace

/*********************************
 * clas SaveDataListView
 *********************************/

SaveDataListView::SaveDataListView( const glv::Rect& parentClipRegion, CallbackFunction callback ) NN_NOEXCEPT
    : CustomVerticalListView( parentClipRegion ), m_OnDrawCallback( callback )
{
    SetTouchAndGo( true );
    glv::Style* pStyle = new glv::Style();
    pStyle->color = glv::Style::standard().color;
    pStyle->color.selection.set( 0.1f, 0.85f, 0.2f );
    style( pStyle );
    font().size( 25.0f );

    // 親クラスで OKボタン( A )に対して Update::Action イベントが発生するので Hack.
#if defined ( NN_DEVMENUSYSTEM )
    changePadClickDetectableButtons( glv::BasicPadEventType::Button::Ok::Mask | glv::BasicPadEventType::Button::Start::Mask | glv::BasicPadEventType::Button::Select::Mask | glv::BasicPadEventType::Button::Y::Mask );
    changePadClickDetectableButtons( glv::DebugPadEventType::Button::Ok::Mask | glv::DebugPadEventType::Button::Start::Mask | glv::DebugPadEventType::Button::Select::Mask | glv::DebugPadEventType::Button::Y::Mask );
#elif defined( NN_CUSTOMERSUPPORTTOOL )
    changePadClickDetectableButtons( glv::BasicPadEventType::Button::Ok::Mask );
    changePadClickDetectableButtons( glv::DebugPadEventType::Button::Ok::Mask );
#else
    changePadClickDetectableButtons( glv::BasicPadEventType::Button::Ok::Mask | glv::BasicPadEventType::Button::Start::Mask | glv::BasicPadEventType::Button::Select::Mask );
    changePadClickDetectableButtons( glv::DebugPadEventType::Button::Ok::Mask | glv::DebugPadEventType::Button::Start::Mask | glv::DebugPadEventType::Button::Select::Mask );
#endif
}

void SaveDataListView::OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT
{
    this->font().getBounds( outWidth, outHeight, item.saveDataIdWideStr );
    outWidth = this->width();
}

/**
 * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
 */
void SaveDataListView::OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT
{
    NN_UNUSED( item );
    m_OnDrawCallback( item, contentRegion );
}


/*********************************
 * class SaveDataListSceneBase
 *********************************/

const float SaveDataListSceneBase::HeaderDescLineFontSize   = 25.0f;
const float SaveDataListSceneBase::HeaderDescLineLineHeight = SaveDataListSceneBase::HeaderDescLineFontSize + 4.0f;

SaveDataListSceneBase::SaveDataListSceneBase( const AbstractOperators& op, Page* pParentPage, const glv::Rect& rect, bool isDefaultbackButtonCallbackUsed, const std::function< void() >& backButtonCallback ) NN_NOEXCEPT
    : SaveDataSceneBase( pParentPage, rect, isDefaultbackButtonCallbackUsed, backButtonCallback )
    , m_Op( op )
    , m_pItems( nullptr )
    , m_pLabelNoItem( nullptr )
    , m_Footer( this, glv::Rect( width() - ( Default::ListMarginLeft + Default::ListMarginRight ), Default::FooterRegion ) )
    , m_pContainer( nullptr )
    , m_pScrollContainer( nullptr )
{
    // ヘッダ
    auto pHeader = new Header( new glv::Label( "", glv::Label::Spec( glv::Place::TL, 0.0f, 0.0f, SaveDataListSceneBase::HeaderDescLineFontSize ) ),
                              glv::Rect( width() - ( Default::ListMarginLeft + Default::ListMarginRight + 30.0f ), SaveDataListSceneBase::HeaderDescLineLineHeight ) );
    {
        *this << pHeader;
        pHeader->anchor( glv::Place::TL ).pos( Default::ListMarginLeft + 9.0f, 8.0f );
        m_pHeader = pHeader;
    }

    // フッタ
    *this << m_Footer;

    // Adding a scrollable container
    m_pContainer = new glv::View( glv::Rect( devmenu::savedata::Default::ListMarginLeft,
        m_pHeader->height() + 30.0f,
        width() - ( Default::ListMarginLeft + Default::ListMarginRight ),
        height() - ( Default::HeaderRegion + Default::FooterRegion + 25.0f ) ) );
    m_pContainer->disable( glv::Property::DrawBack | glv::Property::FocusHighlight | glv::Property::DrawBorder );

    *this << m_pContainer;
    m_pScrollContainer = new ScrollableBoxView( "<", glv::Rect( 0.0f, 0.0f, m_pContainer->width(), m_pContainer->height() ) );
    *m_pContainer << *m_pScrollContainer;

    // アイテムなしメッセージ
    glv::Label* pLabelNoItem = new glv::Label( "No save data is created.", glv::Label::Spec( glv::Place::CC, 0.0f, 0.0f, 32.0f ) );
    *m_pContainer << pLabelNoItem;
    m_pLabelNoItem = pLabelNoItem;

    disable( glv::Visible );
}

void SaveDataListSceneBase::EntryProperties( const SaveDataListView::CollectionType* pItems ) NN_NOEXCEPT
{
    if ( true == pItems->empty() )
    {
        auto* const pLabelNoItem = m_pLabelNoItem;
        pLabelNoItem->enable( glv::Property::t::Visible );
        pLabelNoItem->bringToFront();
        m_Footer.disable( glv::Property::Visible );

        // Focus 可能な View が無くなるのでメインメニューにフォーカスを戻す。
        GetRootSurfaceContext()->MoveFocusToMenuTabs();
    }
    else
    {
        m_pLabelNoItem->disable( glv::Property::t::Visible );
        m_Footer.enable( glv::Property::Visible );
    }
}

void SaveDataListSceneBase::FinalizeProperties() NN_NOEXCEPT
{
    if ( nullptr != m_pItems )
    {
        delete m_pItems;
        m_pItems = nullptr;

#if defined( DEBUG_DATA )
        g_DebugDataCount = 0;
#endif
    }
}

void SaveDataListSceneBase::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );
}

void SaveDataListSceneBase::QuickSort( SaveDataListView::CollectionType* pItems ) NN_NOEXCEPT
{
    std::sort( pItems->begin(), pItems->end() );
}

/**************************************
 * class SaveDataListSceneBase::Header
 **************************************/
SaveDataListSceneBase::Header::Header( glv::Label* pPageTitleLabel, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::Group( rect ), m_pPageTitle( pPageTitleLabel )
{
    // Update the header styles
    glv::Style* pHeaderStyle = new glv::Style();
    pHeaderStyle->color.text.set( 0.6f, 0.6f, 0.6f );
    m_pPageTitle->style( pHeaderStyle );

    m_pTitleTable = new glv::Table( "<", 0.0f, 0.0f, glv::Rect( rect.width(), rect.height() ) );

    *m_pTitleTable << m_pPageTitle;
    // Creates a Divider
    glv::Divider* pHeaderDivider = new glv::Divider( 10.0f, 2.0f );
    *m_pTitleTable << pHeaderDivider;
    m_pTitleTable->arrange();
    *this << m_pTitleTable;
}


/*************************************
 * class SaveDataListSceneBase::Footer
 *************************************/
SaveDataListSceneBase::Footer::Footer( SaveDataListSceneBase* pParent, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::Group( rect )
{
    anchor( glv::Place::BL ).pos( Default::ListMarginLeft + 8.0f, -1 * Default::FooterRegion + 4.0f );
}

/*********************************
 * class SaveDataTileListSceneBase
 *********************************/
SaveDataTileListSceneBase::SaveDataTileListSceneBase( const AbstractOperators& op, Page* pParentPage, const glv::Rect& rect, bool isDefaultbackButtonCallbackUsed, const std::function< void() >& backButtonCallback ) NN_NOEXCEPT
    : SaveDataListSceneBase( op, pParentPage, rect, isDefaultbackButtonCallbackUsed, backButtonCallback )
    , m_ExportButton( " A: Export ", nullptr,  CommonValue::InitialFontSize, 0.0f )
    , m_OptionButton( " -/+: Option ", nullptr, CommonValue::InitialFontSize, 0.0f )
    , m_pSelectedSaveDataProperty( nullptr )
    , m_IsSaveDataTileFocusedOnRefresh( false )
{
    // 操作説明表示
    auto pFooterButtons = new glv::Group( glv::Rect( width(), Default::FooterRegion - 20.0f ) );

    pFooterButtons->anchor( glv::Place::TL ).pos( 0.0f, 0.0f );
    {
        glv::space_t posOptionButton = 16.0f + 16.0f;

        // エクスポートボタン
        m_ExportButton.anchor( glv::Place::TL ).pos( Default::ListMarginLeft, 0.0f );
        m_ExportButton.SetCallback( [&] { m_Op.ExportSaveData( GetSceneType() ); } );
        *pFooterButtons << m_ExportButton;
        posOptionButton += m_ExportButton.right();

        // オプションボタン
        m_OptionButton.anchor( glv::Place::TL ).pos( posOptionButton, 0.0f );
        m_OptionButton.SetCallback( [&] {
            if ( nullptr != m_pSelectedSaveDataProperty )
            {
                const SaveDataPropertyType* pCurrentProperty = m_pSelectedSaveDataProperty;
                if ( nullptr != pCurrentProperty )
                {
                    ShowSaveDataDetails( pCurrentProperty );
                }
            }
        } );
        *pFooterButtons << m_OptionButton;
        m_Footer << pFooterButtons;

        m_ExportButton.disable( glv::Property::FocusHighlight );
        m_OptionButton.disable( glv::Property::FocusHighlight );
    }
};

void SaveDataTileListSceneBase::VerifySaveData() NN_NOEXCEPT
{
    auto pView = new MessageView( true );

    // Set up the Title and Message
    pView->AddMessage( "Verify Save Data" );
    pView->AddMessage( "Verify the selected save data?" );

    pView->AddButton( "Cancel" );

    pView->AddButton(
        "Verify",
        [&]( void* pParam, nn::TimeSpan& timespan )
        {
            if ( nullptr == m_pSelectedSaveDataProperty )
            {
                DEVMENU_LOG( "Could not verify Save Data - Selected Value was null.\n" );
                return;
            }

            StartProgressModalThread(
                "Verifying save data",
                [&]( bool* pOutIsSuccess ) -> nn::Result
                    {
                        NN_UNUSED( pOutIsSuccess );
                        const auto& saveDataInfo = m_pSelectedSaveDataProperty->GetSaveDataInfo();

                        // Note: Verify で validity のメンバ変数を変更しておきたいので、ひとまず const_cast する
                        //       validity によって表示が変わるが、Scene 遷移で QueryProperties() -> Fetch() すると値がクリアされるので注意
                        return const_cast< SaveDataPropertyType* >( m_pSelectedSaveDataProperty )->Verify( saveDataInfo.saveDataSpaceId, saveDataInfo.saveDataId );
                    },
                [&]( const nn::Result& result, bool isSuccess )
                    {
                        NN_UNUSED( isSuccess );
                        OnCompleteVerifyProgress( result );
                    }
            );
        },
        nullptr,
        MessageView::ButtonTextColor::Green
    );

    this->StartModal( pView );
}

void SaveDataTileListSceneBase::OnCompleteVerifyProgress( const nn::Result& result ) NN_NOEXCEPT
{
    if ( result.IsSuccess() )
    {
        if ( m_pSelectedSaveDataProperty->IsValid() )
        {
            this->SetModalViewMessage("Save data is normal.");
        }
        else
        {
            this->SetModalViewMessage("Save data is corrupted.");
        }

        // Note: セーブデータを Fetch しないでテキストだけを変更するようにしたいが、ひとまず妥協。
        QueryProperties();
    }
    else
    {
        char errorMessage[64];
        nn::util::SNPrintf( errorMessage, sizeof( errorMessage ), "Failed to verify save data info (0x%08x)\n", result.GetInnerValueForDebug() );
        this->SetModalViewMessage( std::string( errorMessage ) );
    }
}

void SaveDataTileListSceneBase::DeleteSaveData() NN_NOEXCEPT
{
    auto pView = new MessageView( true );

    // Set up the Title and Message
    pView->AddMessage( "Delete Save Data" );
    pView->AddMessage( "Delete the selected save data?" );

    pView->AddButton("Cancel");

    // Add the button with the Lambda that will delete the Save Data
    pView->AddButton(
        "Delete",
        []( void* pParam, nn::TimeSpan& timespan )
        {
            auto* pSelf = reinterpret_cast< SaveDataTileListSceneBase* >( pParam );
            NN_ABORT_UNLESS_NOT_NULL( pSelf );

            const auto* pCurrentProperty = pSelf->m_pSelectedSaveDataProperty;
            if ( nullptr == pCurrentProperty )
            {
                DEVMENU_LOG( "Could not delete Save Data - Selected Value was null.\n" );
            }
            else
            {
                const auto result = pCurrentProperty->Delete();
                if ( result.IsFailure() )
                {
                    DEVMENU_LOG( "Failed to delete save data info (0x%08x)\n", result.GetInnerValueForDebug() );
                    auto pResultView = new MessageView( false );
                    char errorMessage[ 64 ];
                    nn::util::SNPrintf( errorMessage, sizeof( errorMessage ), "Failed to delete save data info (0x%08x)\n", result.GetInnerValueForDebug() );
                    pResultView->AddMessage( std::string( errorMessage ) );
                    pResultView->AddButton( "Close" );
                    pSelf->StartModal( pResultView, true, true );
                    return;
                }
                pSelf->QueryProperties();

                // Since Delete is only available from the Save Data Details scene, close the details here when the item is successfully deleted.
                pSelf->m_Op.CloseSaveDataDetails( pSelf->GetSceneType() );
            }
        },
        this,
        MessageView::ButtonTextColor::Red
    );

    this->StartModal( pView );
}

void SaveDataTileListSceneBase::CorruptSaveData() NN_NOEXCEPT
{
    auto pView = new MessageView( true );

    // Set up the Title and Message
    pView->AddMessage( "Corrupt Save Data" );
    pView->AddMessage( "Corrupt the selected save data?" );

    pView->AddButton( "Cancel" );

    pView->AddButton(
        "Corrupt",
        []( void* pParam, nn::TimeSpan& timespan )
        {
            auto* pSelf = reinterpret_cast< SaveDataTileListSceneBase* >( pParam );
            NN_ABORT_UNLESS_NOT_NULL( pSelf );

            const auto* pCurrentProperty = pSelf->m_pSelectedSaveDataProperty;
            if ( nullptr == pCurrentProperty )
            {
                DEVMENU_LOG( "Could not corrupt Save Data - Selected Value was null.\n" );
                return;
            }

            const auto result = pCurrentProperty->Corrupt();
            if ( result.IsFailure() )
            {
                DEVMENU_LOG( "Failed to corrupt save data info (0x%08x)\n", result.GetInnerValueForDebug() );
                auto pResultView = new MessageView( false );
                char errorMessage[ 64 ];
                nn::util::SNPrintf( errorMessage, sizeof( errorMessage ), "Failed to corrupt save data info (0x%08x)\n", result.GetInnerValueForDebug() );
                pResultView->AddMessage( std::string( errorMessage ) );
                pResultView->AddButton( "Close" );
                pSelf->StartModal( pResultView, true, true );
                return;
            }

            // Note: セーブデータを Fetch しないでテキストだけを変更するようにしたいが、ひとまず妥協
            pSelf->QueryProperties();

            // ToDo: 他との整合性がとれたらコメントアウトする
            // pSelf->m_Op.UpdateSaveDataDetails( pSelf->m_pSelectedSaveDataProperty );
        },
        this,
        MessageView::ButtonTextColor::Red
    );

    this->StartModal( pView );
}

void SaveDataTileListSceneBase::EnableFooterButtons( bool isEnabled ) NN_NOEXCEPT
{
    if ( isEnabled )
    {
        m_ExportButton.enable( glv::Property::Visible );
        m_OptionButton.enable( glv::Property::Visible );
    }
    else
    {
        m_ExportButton.disable( glv::Property::Visible );
        m_OptionButton.disable( glv::Property::Visible );
    }
}

float SaveDataTileListSceneBase::AddSaveDataListTileToScrollContainer(
    const nn::ncm::ApplicationId& applicationId, nn::fs::SaveDataSpaceId spaceId, const glv::Rect& tileRect, bool isCurrentSelectedSaveDataValid, nn::fs::SaveDataId currentSelectedSaveDataId ) NN_NOEXCEPT
{
    float scrollTopPosition = -1.0f; // Invalid value

    if ( nn::fs::SaveDataSpaceId::SdSystem == spaceId || nn::fs::SaveDataSpaceId::SdUser == spaceId )
    {
        if ( !nn::fs::IsSdCardInserted() )
        {
            return scrollTopPosition;
        }
    }

    // SaveDataApplicationScene でも全セーブデータ情報を取得しているが、大したコストでないので再度読み込んで更新する
    std::unique_ptr< nn::fs::SaveDataIterator > iter;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::OpenSaveDataIterator( &iter, spaceId ) );

    while ( NN_STATIC_CONDITION( true ) )
    {
        const size_t index = m_pItems->size();
        m_pItems->resize( index + 1 );
        const auto fetchResult = m_pItems->at( index ).Fetch( *iter );

        if ( DevMenuSaveData_Exists == fetchResult )
        {
            // If we were not given a specific application ID, or the ID given matches the current item, add it to the display
            if ( nn::fs::InvalidApplicationId == applicationId || m_pItems->at( index ).GetApplicationId().value == applicationId.value )
            {
                SaveDataListTile* pSaveDataTile = new SaveDataListTile( tileRect, m_pItems->at( index ), spaceId,
                    [&]( const SaveDataPropertyType* pProperty ){ m_pSelectedSaveDataProperty = pProperty; } );
                *m_pScrollContainer << pSaveDataTile;

                // Attach click events
                pSaveDataTile->attach( OnSaveDataTileClickNotification, glv::Update::Clicked, this );

                // Check if this tile was the save data that was previously selected
                if ( m_IsSaveDataTileFocusedOnRefresh &&
                    isCurrentSelectedSaveDataValid &&
                    currentSelectedSaveDataId == m_pItems->at( index ).GetSaveDataInfo().saveDataId )
                {
                    this->SetFocus( pSaveDataTile );
                    SetLastFocusedView( pSaveDataTile );
                    scrollTopPosition = pSaveDataTile->top();
                }
            }
        }
        else if ( DevMenuSaveData_None == fetchResult || DevMenuSaveData_Failure == fetchResult )
        {
            m_pItems->resize( index );
            break;
        }
        else
        {
            NN_ABORT( "Not reach here\n" );
        }
    }

    return scrollTopPosition;
}

void SaveDataTileListSceneBase::QueryPropertiesImpl( const nn::ncm::ApplicationId& applicationId, const std::vector< nn::fs::SaveDataSpaceId >& spaceIdList ) NN_NOEXCEPT
{
    // Refresh list
    m_pScrollContainer->Refresh();

    // Store current save data id
    bool isCurrentSelectedSaveDataValid = false;
    nn::fs::SaveDataId currentSelectedSaveDataId = 0ULL;

    if ( nullptr != m_pSelectedSaveDataProperty )
    {
        isCurrentSelectedSaveDataValid = true;
        currentSelectedSaveDataId = m_pSelectedSaveDataProperty->GetSaveDataInfo().saveDataId;

        // Reset the selected tile
        m_pSelectedSaveDataProperty = nullptr;
    }

    // Create save data list tiles
    const glv::Rect tileRect( m_pScrollContainer->GetInnerRect().width(), 50.0f );
    float scrollTopPosition = 0.0f; // Scroll list to the top

    for ( auto spaceId : spaceIdList )
    {
        const auto position = AddSaveDataListTileToScrollContainer( applicationId, spaceId, tileRect, isCurrentSelectedSaveDataValid, currentSelectedSaveDataId );
        if ( 0.0f < position )
        {
            scrollTopPosition = position;
        }
    }

    m_pScrollContainer->ScrollTo( scrollTopPosition );

    // Reset the automatic focus to false
    m_IsSaveDataTileFocusedOnRefresh = false;
}

void SaveDataTileListSceneBase::ShowSaveDataDetails( const SaveDataPropertyType* const pSaveDataProperty ) NN_NOEXCEPT
{
    m_IsSaveDataTileFocusedOnRefresh = true;
    m_Op.UpdateSaveDataDetails( pSaveDataProperty );
    m_Op.ShowSaveDataDetails( GetSceneType() );
}

void SaveDataTileListSceneBase::OnSaveDataTileClickNotification( const glv::Notification& notification ) NN_NOEXCEPT
{
    const auto pSelf = notification.receiver< SaveDataTileListSceneBase >();

    auto& g = reinterpret_cast< glv::GLV& >( pSelf->root() );
    NN_ASSERT( glv::GLV::valid( &g ) );

    // Check if the A button was pressed
    if ( g.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::A >() ||
        g.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::A >() )
    {
        pSelf->m_Op.ExportSaveData( pSelf->GetSceneType() );
    }
    else if ( g.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::Start >() ||
        g.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::Start >() ||
        g.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::Select >() ||
        g.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::Select >() )
    {
        const SaveDataPropertyType* pCurrentProperty = pSelf->m_pSelectedSaveDataProperty;
        if ( nullptr != pCurrentProperty )
        {
            pSelf->ShowSaveDataDetails( pCurrentProperty );
        }
    }
}

/*********************************
 * class SaveDataListTile
 *********************************/

const glv::Label::Spec SaveDataListTile::DefaultLabelSpec( glv::Place::CL, 0.0f, 0.0f, 22.0f );

SaveDataListTile::SaveDataListTile( const glv::Rect& rect, SaveDataPropertyType& item, nn::fs::SaveDataSpaceId spaceId, const std::function< void( const SaveDataPropertyType* ) >& callback ) NN_NOEXCEPT
    : glv::View( rect )
    , m_pType( new glv::Label( item.saveDataTypeWideStr, SaveDataListTile::DefaultLabelSpec ) )
    , m_pSaveDataId( new glv::Label( item.systemSaveDataIdStr, SaveDataListTile::DefaultLabelSpec ) )
    , m_pTimeStamp( new glv::Label( "", SaveDataListTile::DefaultLabelSpec ) )
    , m_pSeperator( new glv::Label( "|", SaveDataListTile::DefaultLabelSpec ) )
    , m_Table( " < < < > ", 10.0f, 10.0f )
    , m_SaveDataProperty( item )
    , m_OnFocusCallback( callback )
{
    *this << ( m_Table << m_pType << m_pSeperator << m_pSaveDataId << m_pTimeStamp );
    m_Table.arrange().fit( false );

    // By default, make Save Data ID and spacer not visible
    m_pSeperator->disable( glv::Property::Visible );
    m_pSaveDataId->disable( glv::Property::Visible );

    if ( nn::fs::SaveDataSpaceId::System == spaceId )
    {
        if ( nn::fs::InvalidUserId != item.GetSaveDataInfo().saveDataUserId )
        {
            m_pSaveDataId->setValue( std::string( m_pSaveDataId->getValue() ) + std::string( " (Account)" ) );
        }
        else if ( CommonValue::OwnSaveDataMeta.id == item.GetSaveDataInfo().systemSaveDataId )
        {
            m_pSaveDataId->setValue( std::string( m_pSaveDataId->getValue() ) + std::string( " (DevMenu)" ) );
        }

        // We are displaying System Save Data, make Save Data ID visible
        m_pSeperator->enable( glv::Property::Visible );
        m_pSaveDataId->enable( glv::Property::Visible );
    }

    m_Table.maximize();
    m_Table.disable( glv::Property::HitTest );
}

// Note: 未使用。詳細画面からの戻りでテキストカラーだけを更新するための仕込み。
void SaveDataListTile::UpdateTextColor() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC( glv::Style, s_CurrentStyle );

    // Check if the Save Data is valid
    if ( !m_SaveDataProperty.IsValid() )
    {
        // This Save Data is corrupted, change it's color to red
        s_CurrentStyle = glv::Style::standard();
        s_CurrentStyle.color.text.set( 1.0f, 0.0f, 0.0f );
        m_pType->style( &s_CurrentStyle );
        m_pSaveDataId->style( &s_CurrentStyle );
        m_pSeperator->style( &s_CurrentStyle );
    }
}

bool SaveDataListTile::onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT
{
    NN_UNUSED( glvRoot );

    switch ( event )
    {
    case glv::Event::FocusGained:
        // Update the selected Save Data ID
        m_OnFocusCallback( &m_SaveDataProperty );
        break;
    default:
        break;
    }

    return true;
}

/************************************
 * class SaveDataApplicationTileView
 ************************************/
SaveDataApplicationTileView::SaveDataApplicationTileView
( const glv::Rect& rect ) NN_NOEXCEPT
    : glv::View( rect )
    , m_Table( "< < <, | < <", 5.0f, 10.0f )
    , m_pApplicationTitle( new glv::Label( GLV_TEXT_API_STRING_UTF16( "N/A" ), glv::Label::Spec( glv::Place::CL, 0.0f, 0.0f, CommonValue::InitialFontSize ) ) )
    , m_pIdValue( new glv::Label( "" ) )
{
    auto pTableSpacer = new devmenu::Spacer( 8.0f, 16.0f );
    auto pSaveDataIcon = new devmenu::IconLabel( devmenu::IconCodePoint::SaveData, glv::Label::Spec( glv::Place::TL, 0.0f, 0.0f, 20.0f ) );
    auto pIdLabel = new glv::Label( "ID:" );

    m_Table << pTableSpacer << pSaveDataIcon << m_pApplicationTitle << pIdLabel << m_pIdValue;

    *this << m_Table;
    m_Table.arrange().fit( false );

    m_Table.maximize();
    m_Table.disable( glv::Property::HitTest );
    this->disable( glv::Property::DrawBorder | glv::Property::HitTest | glv::Property::Controllable | glv::Property::DrawBack );
}

void SaveDataApplicationTileView::UpdateSaveDataApplicationTileViewValues( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    std::unique_ptr< char[] > buffer;
    size_t dataSize;

    const auto result = application::GetApplicationControlData( &dataSize, &buffer, nn::ns::ApplicationControlSource::Storage, applicationId );

    if ( result.IsSuccess() )
    {
        nn::ns::ApplicationControlDataAccessor accessor( buffer.get(), dataSize );
        const nn::ns::ApplicationControlProperty& property = accessor.GetProperty();

        // Set the Application Title as page header
        glv::WideString applicationTitle;
        glv::CopyWithConvertWideString( applicationTitle, property.GetDefaultTitle().name );
        m_pApplicationTitle->setValue( applicationTitle );
    }
    else
    {
        // Could not load the Application Title set ID as header??
        m_pApplicationTitle->setValue( glv::WideString( GLV_TEXT_API_STRING_UTF16( "Unable to load Application Name" ) ) );
    }

    // Set the Application ID
    char applicationIdFormatted[ 32 ];
    nn::util::SNPrintf( applicationIdFormatted, sizeof( applicationIdFormatted ), "0x%016llX", applicationId.value );
    m_pIdValue->setValue( std::string( applicationIdFormatted ) );

    // 表示位置を期待通りにするために一度 remove して再度 add する
    m_Table.ReAddOwnselfToParent( this );
}

/*****************************************
 * class SaveDataApplicationTileContainer
 *****************************************/
SaveDataApplicationTileContainer::SaveDataApplicationTileContainer(
        SaveDataApplicationTileView* pSaveDataApplicationTile,
        const glv::Rect& rect,
        const nn::ncm::ApplicationId& applicationId,
        const std::function< void( const nn::ncm::ApplicationId* ) >& callback ) NN_NOEXCEPT
    : glv::View( rect )
    , m_ApplicationId( applicationId )
    , m_pSaveDataApplicationTile( pSaveDataApplicationTile )
    , m_OnFocusCallback( callback )
{
    this->enable( glv::Property::DrawBorder | glv::Property::FocusHighlight | glv::Property::HitTest | glv::Property::Controllable );
}

SaveDataApplicationTileContainer::~SaveDataApplicationTileContainer() NN_NOEXCEPT
{
    // Detach the Content tile so that it doesn't get destroyed on Scrollable container refresh
    m_pSaveDataApplicationTile->remove();
    m_pSaveDataApplicationTile = nullptr;
}

void SaveDataApplicationTileContainer::onDraw( glv::GLV& glvContext ) NN_NOEXCEPT
{
    // Check if the the given tile index is already attached to this container
    if ( m_pSaveDataApplicationTile->parent != this )
    {
        // If it is not, update the tile contents by passing this container's item information
        m_pSaveDataApplicationTile->UpdateSaveDataApplicationTileViewValues( m_ApplicationId );

        // Attach the ApplicationTile to this container
        *this << m_pSaveDataApplicationTile;
    }

    // Call the parent view's Draw
    glv::View::onDraw( glvContext );
}

bool SaveDataApplicationTileContainer::onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT
{
    NN_UNUSED( glvRoot );

    switch ( event )
    {
    case glv::Event::FocusGained:
        // Update the selected Save Data ID
        m_OnFocusCallback( &m_ApplicationId );
        break;
    default:
        break;
    }

    return true;
}

const nn::ncm::ApplicationId SaveDataApplicationTileContainer::GetApplicationId() NN_NOEXCEPT
{
    return m_ApplicationId;
}

/*************************************
 * class ApplicationSaveDataListScene
 *************************************/

ApplicationSaveDataListScene::ApplicationSaveDataListScene( const AbstractOperators& op, Page* pParentPage, const glv::Rect& rect ) NN_NOEXCEPT
    : SaveDataListSceneBase( op, pParentPage, rect, true )
    , m_pSelectedApplicationId( nullptr )
    , m_IsApplicationTileFocusedOnRefresh( false )
{
    // ヘッダ
    m_pHeader->SetPageTitle( "Applications with User Save Data" );

    // Create new button
    devmenu::Button* pDetailsButton = new devmenu::Button( " -/+: Details ", [&] { ShowSaveDataTypeListScene(); }, CommonValue::InitialFontSize, 0.0f );

    // Add new button
    *this << pDetailsButton;

    // Disable focus highlight so that button is not selectable via Joycons
    pDetailsButton->disable( glv::Property::FocusHighlight );

    pDetailsButton->pos( m_Footer.left(), m_Footer.top() );

    for ( int i = 0; i < MaxCountOfVisibleSaveDataApplicationsTiles; i++ )
    {
        m_SaveDataApplicationsTilesList.push_back( new SaveDataApplicationTileView( glv::Rect( m_pScrollContainer->GetInnerRect().width(), 80.0f ) ) );
    }
}

void ApplicationSaveDataListScene::ShowSaveDataTypeListScene() NN_NOEXCEPT
{
    if ( nullptr != m_pSelectedApplicationId )
    {
        // Load the Application Scene for the selected Application Id
        ShowSaveDataTypeListScene( *m_pSelectedApplicationId );
    }
}

void ApplicationSaveDataListScene::QueryProperties() NN_NOEXCEPT
{
    // No need to refresh if this page is not visible
    if ( !enabled( glv::Property::Visible ) )
    {
        return;
    }

    FinalizeProperties();
    m_pItems = new SaveDataListView::CollectionType();
    m_pItems->reserve( MaxApplicationCount );

    // Group Save Data by Application ID
    const std::vector< nn::fs::SaveDataSpaceId > spaceIdList = { nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::SdUser };
    QueryPropertiesImpl( spaceIdList );
    QuickSort( m_pItems );

    // プロパティリスト登録
    EntryProperties( m_pItems );

    // Regular footer buttons are not used in this scene, disable them
    m_Footer.disable( glv::Property::Visible );
}

void ApplicationSaveDataListScene::QueryPropertiesImpl( const std::vector< nn::fs::SaveDataSpaceId >& spaceIdList ) NN_NOEXCEPT
{
    // Refresh list
    m_pScrollContainer->Refresh();

    // Create a map to keep track of how many unique applications exists
    std::map< nn::Bit64, nn::ncm::ApplicationId > applicationList;

    for ( auto spaceId : spaceIdList )
    {
        if ( nn::fs::SaveDataSpaceId::SdSystem == spaceId || nn::fs::SaveDataSpaceId::SdUser == spaceId )
        {
            if ( !nn::fs::IsSdCardInserted() )
            {
                continue;
            }
        }

        {
            std::unique_ptr< nn::fs::SaveDataIterator > iter;

            // 一時的に nn::fs::OpenSaveDataIterator が Abort しないようにする
            nn::fs::SetEnabledAutoAbort( false );

            const auto result = nn::fs::OpenSaveDataIterator( &iter, spaceId );

            nn::fs::SetEnabledAutoAbort( true );

            if ( result.IsFailure() )
            {
                DEVMENU_LOG( "Failed to open save data iterator as 0x%08x\n", result.GetInnerValueForDebug() );
                return;
            }

            while ( NN_STATIC_CONDITION( true ) )
            {
                const size_t index = m_pItems->size();
                m_pItems->resize( index + 1 );
                const auto fetchResult = m_pItems->at( index ).Fetch( *iter );

                if ( DevMenuSaveData_Exists == fetchResult )
                {
                    // Store this application to the map
                    applicationList[ m_pItems->at( index ).GetSaveDataInfo().applicationId.value ] = m_pItems->at( index ).GetSaveDataInfo().applicationId;
                }
                else if ( DevMenuSaveData_None == fetchResult || DevMenuSaveData_Failure == fetchResult )
                {
                    m_pItems->resize( index );
                    break;
                }
                else
                {
                    NN_ABORT( "Not reach here\n" );
                }
            }
        }
    }

    // Loop through the map and create application tiles
    const glv::Rect tileRect( m_pScrollContainer->GetInnerRect().width(), 80.0f );
    float scrollTopPosition = 0.0f; // Scroll list to the top
    auto countOfVisibleSaveDataApplicationsTiles = 0;

    for ( const auto& pair : applicationList )
    {
        // Create an Application Tile
        SaveDataApplicationTileContainer* pApplicationTile = new SaveDataApplicationTileContainer(
            m_SaveDataApplicationsTilesList.at( countOfVisibleSaveDataApplicationsTiles % MaxCountOfVisibleSaveDataApplicationsTiles ),
            tileRect, pair.second, [&]( const nn::ncm::ApplicationId* applicationId ){ m_pSelectedApplicationId = applicationId; } );
        *m_pScrollContainer << pApplicationTile;

        // Attach click events
        pApplicationTile->attach( OnApplicationTileClickNotification, glv::Update::Clicked, this );

        if ( m_IsApplicationTileFocusedOnRefresh && nullptr != m_pSelectedApplicationId && m_pSelectedApplicationId->value == pair.second.value )
        {
            // This was the last application that had focus. Reset the page focus to this element if the list is visible
            if ( enabled( glv::Property::Visible ) )
            {
                this->SetFocus( pApplicationTile );
                SetLastFocusedView( pApplicationTile );
                // Scroll the list to the focused element
                scrollTopPosition = pApplicationTile->top();
            }
        }
        countOfVisibleSaveDataApplicationsTiles++;
    }

    m_pScrollContainer->ScrollTo( scrollTopPosition );

    // Reset the automatic focus to false
    m_IsApplicationTileFocusedOnRefresh = false;
}

void ApplicationSaveDataListScene::FinalizeProperties() NN_NOEXCEPT
{
    SaveDataListSceneBase::FinalizeProperties();
}

void ApplicationSaveDataListScene::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    SaveDataListSceneBase::OnLoopBeforeSceneRenderer( context, events );
}

glv::View* ApplicationSaveDataListScene::GetFocusableChild() NN_NOEXCEPT
{
    // Toriaezu: DropDown がベター
    return this;
}

void ApplicationSaveDataListScene::Refresh() NN_NOEXCEPT
{
    QueryProperties();
}

void ApplicationSaveDataListScene::OnApplicationTileClickNotification( const glv::Notification& notification ) NN_NOEXCEPT
{
    const auto pTileView = notification.sender< SaveDataApplicationTileContainer >();
    const auto pSelf = notification.receiver< ApplicationSaveDataListScene >();
    auto& glvRoot = reinterpret_cast< glv::GLV& >( pSelf->root() );
    NN_ASSERT( glv::GLV::valid( &glvRoot ) );

    // Check if the +/- buttons were pressed
    if ( glvRoot.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::Start >() ||
        glvRoot.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::Start >() ||
        glvRoot.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::Select >() ||
        glvRoot.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::Select >() )
    {
        // Display the Save Data Application Scene
        pSelf->ShowSaveDataTypeListScene( pTileView->GetApplicationId() );
    }
}

void ApplicationSaveDataListScene::ShowSaveDataTypeListScene( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    m_IsApplicationTileFocusedOnRefresh = true;
    m_Op.UpdateSaveDataTypeList( applicationId );
    m_Op.ShowSaveDataTypeListScene( GetSceneType() );
}

/*********************************
 * class SaveDataTypeListScene
 *********************************/

SaveDataTypeListScene::SaveDataTypeListScene( const AbstractOperators& op, Page* pParentPage, const glv::Rect& rect, const std::function< void() >& backButtonCallback ) NN_NOEXCEPT
    : SaveDataTileListSceneBase( op, pParentPage, rect, false, backButtonCallback )
    , m_pBackButton( new devmenu::Button( "< Back", [&] { op.CloseSaveDataTypeListScene( SceneType_ApplicationSaveDataList ); }, glv::Rect( Default::ButtonWidthBack, 35.0f ) ) )
    , m_ApplicationId( nn::fs::InvalidApplicationId )
{
    m_pBackButton->anchor( glv::Place::TL ).pos( 12.0f, 6.0f );
    *this << m_pBackButton;

    // Update the location of the header and list until we can move the back button to the footer
    m_pHeader->t = m_pBackButton->bottom() + 15.0f;
    m_pContainer->t = m_pHeader->bottom() + 22.0f;
    // Reposition the footer since we moved the Container
    m_Footer.t = m_pContainer->bottom() + 22.0f;
}

void SaveDataTypeListScene::SetDefaultFocusedView() NN_NOEXCEPT
{
    // Sets default focus on the Back Button
    auto& g = reinterpret_cast< glv::GLV& >( this->root() );
    NN_ASSERT( glv::GLV::valid( &g ) );
    g.setFocus( m_pBackButton );
}

void SaveDataTypeListScene::UpdateSelectedApplicationInformation( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    // Update application id
    m_ApplicationId = applicationId;

    // Load Application Information (if available)
    std::unique_ptr< char[] > buffer;
    size_t dataSize;

    const auto result = application::GetApplicationControlData( &dataSize, &buffer, nn::ns::ApplicationControlSource::Storage, applicationId );

    if ( result.IsSuccess() )
    {
        nn::ns::ApplicationControlDataAccessor accessor( buffer.get(), dataSize );
        const nn::ns::ApplicationControlProperty& property = accessor.GetProperty();

        // Set the Application Title as page header
        m_pHeader->SetPageTitle( property.GetDefaultTitle().name );
    }
    else
    {
        // Could not load the Application Title set ID as header??
        m_pHeader->SetPageTitle( "Unable to load Application Name" );
    }

    // Sets the default focus
    SetDefaultFocusedView();
}

void SaveDataTypeListScene::QueryProperties() NN_NOEXCEPT
{
    // No need to refresh if this page is not visible
    if ( !enabled( glv::Property::Visible ) )
    {
        return;
    }

    FinalizeProperties();
    m_pItems = new SaveDataListView::CollectionType();
    m_pItems->reserve( MaxSaveDataIdCount );

    // Only load save data for applications that match the given application ID
    const std::vector< nn::fs::SaveDataSpaceId > spaceIdList = { nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::SdUser };
    QueryPropertiesImpl( m_ApplicationId, spaceIdList );
    QuickSort( m_pItems );

    // プロパティリスト登録
    EntryProperties( m_pItems );
}

void SaveDataTypeListScene::FinalizeProperties() NN_NOEXCEPT
{
    SaveDataListSceneBase::FinalizeProperties();
}

void SaveDataTypeListScene::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    SaveDataListSceneBase::OnLoopBeforeSceneRenderer( context, events );
    this->UpdateExecutionProgress();
}

/**
 * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
 */
glv::View* SaveDataTypeListScene::GetFocusableChild() NN_NOEXCEPT
{
    return m_pBackButton;
}

void SaveDataTypeListScene::Refresh() NN_NOEXCEPT
{
    QueryProperties();
}

void SaveDataTypeListScene::ExportSaveData() NN_NOEXCEPT
{
    if ( nullptr != m_pSelectedSaveDataProperty )
    {
        const auto* pProperty = m_pSelectedSaveDataProperty;
        auto pView = new MessageView( true );

        if ( pProperty->IsValid() )
        {
            // Set up the Title and Message
            pView->AddMessage( "Export Save Data" );
            pView->AddMessage( "Export the selected save data to SD card?" );

            pView->AddButton( "Cancel" );

            // Add the button with the Lambda that will delete the Save Data
            pView->AddButton(
                "Export",
                [&]( void* pParam, nn::TimeSpan& timespan )
            {
                if ( nullptr != m_pSelectedSaveDataProperty )
                {
                    StartProgressModalThread(
                        "Exporting save data",
                        [&]( bool* pOutIsSuccess ) -> nn::Result
                            {
                                NN_UNUSED( pOutIsSuccess );
                                return m_pSelectedSaveDataProperty->ExportSaveData();
                            },
                        [&]( const nn::Result& result, bool isSuccess )
                            {
                                NN_UNUSED( isSuccess );
                                OnCompleteExportProgress( result );
                            }
                    );
                }
                else
                {
                    DEVMENU_LOG("Could not delete Save Data - Selected Value was null.\n");
                }
            },
            nullptr,
            MessageView::ButtonTextColor::Green
            );
        }
        else
        {
            pView->AddMessage( "This save data is corrupted and cannot be exported." );
            pView->AddButton( "Close" );
        }
        this->StartModal( pView );
    }
}

void SaveDataTypeListScene::OnCompleteExportProgress(const nn::Result& result) NN_NOEXCEPT
{
    if (result.IsSuccess())
    {
        this->SetModalViewMessage( std::string( "Save data has been exported." ) );
    }
    else
    {
#if !defined ( NN_CUSTOMERSUPPORTTOOL )
        std::string errorMessage;
        CreateErrorMessage( &errorMessage, std::string( "Failed to export save data." ), result );
        this->SetModalViewMessage( errorMessage );
#else
        if ( ResultSizeOverError::Includes( result ) )
        {
            std::string errorMessage;
            CreateErrorMessage( &errorMessage, std::string( "Save data beyond 2GB size can not be exported." ), result );
            this->SetModalViewMessage( errorMessage );
        }
        else
        {
            std::string errorMessage;
            CreateErrorMessage( &errorMessage, std::string( "Failed to export save data." ), result );
            this->SetModalViewMessage( errorMessage );
        }
#endif
    }
}

void SaveDataTypeListScene::ExportRawSaveData() NN_NOEXCEPT
{
    if ( nullptr != m_pSelectedSaveDataProperty )
    {
        const auto* pProperty = m_pSelectedSaveDataProperty;
        auto pView = new MessageView( true );

        if ( pProperty->IsValid() )
        {
            // Set up the Title and Message
            pView->AddMessage( "Export(+Raw) Save Data" );
            pView->AddMessage( "Export the selected save data (added raw image) to SD card?" );

            pView->AddButton( "Cancel" );

            // Add the button with the Lambda that will delete the Save Data
            pView->AddButton(
                "Export(+Raw)",
                [&]( void* pParam, nn::TimeSpan& timespan )
            {
                if ( nullptr != m_pSelectedSaveDataProperty )
                {
                    StartProgressModalThread(
                        "Exporting save data",
                        [&]( bool* pOutIsSuccess ) -> nn::Result
                            {
                                NN_UNUSED( pOutIsSuccess );
                                return m_pSelectedSaveDataProperty->ExportRawSaveData();
                            },
                        [&]( const nn::Result& result, bool isSuccess )
                            {
                                NN_UNUSED( isSuccess );
                                OnCompleteExportProgress( result );
                            }
                    );
                }
                else
                {
                    DEVMENU_LOG("Could not delete Save Data - Selected Value was null.\n");
                }
            },
            nullptr,
            MessageView::ButtonTextColor::Green
            );
        }
        else
        {
            pView->AddMessage( "This save data is corrupted and cannot be exported." );
            pView->AddButton( "Close" );
        }
        this->StartModal( pView );
    }
}

/*********************************
 * class SystemSaveListScene
 *********************************/

SystemSaveDataListScene::SystemSaveDataListScene( const AbstractOperators& op, Page* pParentPage, const glv::Rect& rect ) NN_NOEXCEPT
    : SaveDataTileListSceneBase( op, pParentPage, rect, true )
{
    // ヘッダ
    m_pHeader->SetPageTitle( "System Save Data" );
}

void SystemSaveDataListScene::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    SaveDataListSceneBase::OnLoopBeforeSceneRenderer( context, events );
}

void SystemSaveDataListScene::QueryProperties() NN_NOEXCEPT
{
    // No need to refresh if this page is not visible
    if ( !enabled( glv::Property::Visible ) )
    {
        return;
    }

    FinalizeProperties();

    m_pItems = new SaveDataListView::CollectionType();
    m_pItems->reserve( MaxApplicationCount );
    QueryPropertiesImpl( nn::fs::InvalidApplicationId, std::vector< nn::fs::SaveDataSpaceId >( 1, nn::fs::SaveDataSpaceId::System ) );

    // プロパティリスト登録
    EntryProperties( m_pItems );
}

void SystemSaveDataListScene::FinalizeProperties() NN_NOEXCEPT
{
    SaveDataListSceneBase::FinalizeProperties();
}

void SystemSaveDataListScene::ExportSaveData() NN_NOEXCEPT
{
    DEVMENU_LOG( "Export SaveData(to be implemented)\n" );
}

void SystemSaveDataListScene::ExportRawSaveData() NN_NOEXCEPT
{
    DEVMENU_LOG( "Export(+Raw) SaveData(to be implemented)\n" );
}

savedata::SceneType SystemSaveDataListScene::GetSceneType() NN_NOEXCEPT
{
#if defined ( NN_DEVMENUSYSTEM )
    return SceneType_SystemSaveDataList;
#else
    // Note:  Since the System Save Data scene should never be available except in DevMenuSystem, and thus SceneType_SystemSaveData is not defined.
    //        This should never be hit, and is here primarily for compilation reasons, as the method is defined as pure virtual.
    DEVMENU_LOG( "Warning: Returned incorrect Scene Type for SystemSaveListScene!\n" );
    return SceneType_ApplicationSaveDataList;
#endif
}

void SystemSaveDataListScene::Refresh() NN_NOEXCEPT
{
    QueryProperties();
}

}} // ~namespace devmenu::savedata, ~namespace devmenu
