﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/



#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssystem/fs_PartitionFileSystem.h>
#include <nn/spl/spl_Api.h>

#include "DevMenu_SaveData.h"
#include "DevMenu_SaveDataConfig.h"
#include "DevMenu_SaveDataCustomerSupportTool.h"
#include "../DevMenu_Result.h"

namespace devmenu { namespace savedata {

namespace {

    struct PartitionFileSystemInfo
    {
        uint64_t entriesCount;
        uint64_t fileOffset;
    };

    const nn::Result CreateDirectoryRecursive( const std::string& destPath ) NN_NOEXCEPT
    {
        // パスからディレクトリパスを作成
        auto directoryPath = destPath;
        while ( *directoryPath.rbegin() != '/' )
        {
            directoryPath.erase( directoryPath.size() - 1 );
        }

        std::unique_ptr< char[] > workBuffer( new char[ directoryPath.size() + 1 ] );
        char* path = workBuffer.get();
        memset( path, 0x0, directoryPath.size() + 1 );
        for ( int i = 0; i < directoryPath.size(); ++i )
        {
            path[ i ] = directoryPath[ i ];
            if ( directoryPath[ i ] != '/' )
            {
                continue;
            }
            if ( i - 1 < 0 )
            {
                // iがマイナスになる場合は失敗させる
                // この処理に入るのは先頭が / の不正なパスが設定されている時です
                return ResultApplicationError();
            }
            if ( directoryPath[ i - 1 ] == ':' )
            {
                continue;
            }
            DEVMENU_LOG( "create dir %s\n", path );
            // 既にディレクトリが存在する場合は何もしない
            const auto result = nn::fs::CreateDirectory( path );
            if ( !result.IsSuccess() && !nn::fs::ResultPathAlreadyExists::Includes( result ) )
            {
                return result;
            }
        }
        NN_RESULT_SUCCESS;
    }

    const nn::Result CopySaveDataDirectoryForNsaveImpl( const std::string& destPath, const std::string& srcPath, bool isJournalingEnabled, int64_t journalSize ) NN_NOEXCEPT
    {
        DEVMENU_LOG( "copy %s to %s\n", srcPath.c_str(), destPath.c_str() );

        nn::fs::DirectoryHandle directoryHandle;
        nn::fs::DirectoryEntry* directoryEntryArea = nullptr;
        const std::string saveDataStorageName = GetStorageName( destPath );

        int64_t readEntryCount = int64_t();

        if ( isJournalingEnabled )
        {
            if ( journalSize == 0 )
            {
                DEVMENU_LOG( "[ERROR] To copy savedata, must set destSaveDataJournalSize." );
                return ResultApplicationError();
            }
            if ( journalSize < 0x4000 ) // 16 KB
            {
                DEVMENU_LOG( "[WARNING] JournalSize is too small. (size: %llx)", journalSize );
            }
        }

        // 子エントリ情報の取得 (ファイルのみ)
        NN_RESULT_DO( GetFsEntries( &directoryHandle, &directoryEntryArea, &readEntryCount, srcPath, nn::fs::OpenDirectoryMode_File ) );

        for ( auto i = 0; i < readEntryCount; i++ )
        {
            nn::fs::FileHandle readHandle, writeHandle;
            int64_t fileSize;

            // コピー対象ディレクトリ以外のファイルは書き戻さない
            if ( std::strlen( directoryEntryArea[ i ].name ) < DataMoveInfo::DirectoryStrLength )
            {
                continue;
            }
            else if ( IsSaveDataInfoFile( directoryEntryArea[ i ].name + DataMoveInfo::DirectoryStrLength ) )
            {
                continue;
            }

            std::string readPath = srcPath + directoryEntryArea[ i ].name;
            std::string writePath = destPath + &directoryEntryArea[ i ].name[ DataMoveInfo::DirectoryStrLength ];

            NN_RESULT_DO(CreateDirectoryRecursive(writePath));

            NN_RESULT_DO( nn::fs::OpenFile( &readHandle, readPath.c_str(), nn::fs::OpenMode_Read ) );
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile( readHandle );
            };

            NN_RESULT_DO(nn::fs::GetFileSize( &fileSize, readHandle ) );
            if ( ExistsFile( writePath ) )
            {
                DEVMENU_LOG( "  skipped %s\n", directoryEntryArea[ i ].name );
                continue;
            }
            NN_RESULT_DO(nn::fs::CreateFile( writePath.c_str(), fileSize ) );
            if ( isJournalingEnabled )
            {
                NN_RESULT_DO( nn::fs::CommitSaveData( saveDataStorageName.c_str() ) );
            }

            const int64_t maxBufferSize = 1024 * 1024;
            const int64_t writeBufferSize = isJournalingEnabled ? ( journalSize > maxBufferSize ? maxBufferSize : journalSize ) : maxBufferSize;
            std::unique_ptr< char[] > workBuffer( new char[ writeBufferSize ] );
            const int64_t journalSizeToUse = isJournalingEnabled ? journalSize : fileSize;

            for ( int64_t journalingOffset = 0; journalingOffset < fileSize; journalingOffset += journalSizeToUse )
            {
                const int64_t workJounalSize = std::min( fileSize - journalingOffset, journalSizeToUse );

                // ファイル Open から Close の 1 ループスコープ
                {
                    NN_RESULT_DO( nn::fs::OpenFile( &writeHandle, writePath.c_str(), nn::fs::OpenMode_Write ) );
                    NN_UTIL_SCOPE_EXIT
                    {
                        nn::fs::CloseFile( writeHandle );
                    };

                    for ( int64_t bufferOffset = 0; bufferOffset < workJounalSize; bufferOffset += writeBufferSize )
                    {
                        const int64_t offset = journalingOffset + bufferOffset;
                        const int64_t writeSize = std::min( workJounalSize - bufferOffset, writeBufferSize );
                        NN_RESULT_DO( nn::fs::ReadFile( readHandle, offset, workBuffer.get(), static_cast< int >( writeSize ) ) );
                        NN_RESULT_DO( nn::fs::WriteFile( writeHandle, offset, workBuffer.get(), static_cast< int >( writeSize ), nn::fs::WriteOption() ) );
                    }
                    NN_RESULT_DO( nn::fs::FlushFile( writeHandle ) );
                }
                if ( isJournalingEnabled )
                {
                    NN_RESULT_DO( nn::fs::CommitSaveData( saveDataStorageName.c_str() ) ) ;
                }
            }

            if ( isJournalingEnabled )
            {
                NN_RESULT_DO( nn::fs::CommitSaveData( saveDataStorageName.c_str() ) );
            }
        }
        delete[] directoryEntryArea;
        directoryEntryArea = nullptr;

        NN_RESULT_SUCCESS;
    }

    /**
     * @brief srcData で指定したデータのSha256HashからCmacを生成します。
     */
    const nn::Result ComputeCmacFromSha256Hash( nn::Bit8* pOutCmacData, void* pSrcData, size_t srcSize) NN_NOEXCEPT
    {
        // Sha256のハッシュを作成
        nn::crypto::Sha256Generator sha256;
        sha256.Initialize();
        sha256.Update( pSrcData, srcSize );
        static const int HashSize = 32;
        uint8_t hashData[ HashSize ];
        sha256.GetHash( hashData, HashSize );

        // Sha256のハッシュからCmacを作成
        static const nn::Bit8 GenerateAesKeySource[] =
        {
            0x0e, 0x39, 0x7f, 0x11, 0xc6, 0xd1, 0xda, 0x1f, 0x13, 0xa0, 0xcf, 0x11, 0x08, 0x23, 0xc2, 0x6d
        };
        static const nn::Bit8 LoadAesKeySource[] =
        {
            0x69, 0x7d, 0x6d, 0xd8, 0x7b, 0x32, 0x63, 0xde, 0x34, 0x1d, 0x7f, 0x91, 0xe0, 0xae, 0xd9, 0x26
        };
        nn::spl::InitializeForCrypto();
        nn::spl::AccessKey accessKey;
        static const int KeyGeneration = 0;
        static const int Option = 0x00000001;
        NN_RESULT_DO ( nn::spl::GenerateAesKek( &accessKey, GenerateAesKeySource, sizeof( GenerateAesKeySource ), KeyGeneration, Option ) );
        int slotIndex;
        NN_RESULT_DO ( nn::spl::AllocateAesKeySlot( &slotIndex ) );
        NN_RESULT_DO ( nn::spl::LoadAesKey( slotIndex, accessKey, LoadAesKeySource, sizeof( LoadAesKeySource ) ) );
        NN_RESULT_DO ( nn::spl::ComputeCmac( pOutCmacData, nn::spl::AesBlockSize, slotIndex, hashData, sizeof( hashData ) ) );
        nn::spl::Finalize();
        NN_RESULT_SUCCESS;
    }

    const std::string GetFileEntryName( const std::string& path ) NN_NOEXCEPT
    {
        size_t pos = path.find( ":" );
        if ( std::string::npos == pos )
        {
            return "";
        }
        return path.substr( pos + DataMoveInfo::MountDelimiterStrLength, path.size() - pos - DataMoveInfo::MountDelimiterStrLength ); // ":/"以降をFileEntryとして取得する
    }

    // 全てのエントリーを取得する
    const nn::Result GetAllFsEntries( const std::string& srcPath, const std::string& destPath, PartitionFileSystemInfo& fileSystemInfo,
        std::vector< nn::fssystem::PartitionFileSystemMeta::FileEntryForConstruct >& internalEntryArray ) NN_NOEXCEPT
    {
        DEVMENU_LOG( "check %s\n", srcPath.c_str());

        nn::fs::DirectoryHandle directoryHandle;
        nn::fs::DirectoryEntry* directoryEntryArea = nullptr;
        nn::fssystem::PartitionFileSystemMeta::FileEntryForConstruct entryArray;

        int64_t readEntryCount = int64_t();

        // 子エントリ情報の取得 (ファイルのみ)
        NN_RESULT_DO( GetFsEntries( &directoryHandle, &directoryEntryArea, &readEntryCount, srcPath, nn::fs::OpenDirectoryMode_File ) );
        fileSystemInfo.entriesCount += readEntryCount;

        // srcPath 直下のファイルコピー
        for ( auto i = 0; i < readEntryCount; i++ )
        {
            std::string entryPath;
            if ( IsSaveDataInfoFile( directoryEntryArea[ i ].name ) )
            {
                entryPath = GetFileEntryName( srcPath + directoryEntryArea[ i ].name );
            }
            else
            {
                // パス名には、一般開発者が展開して、SDにコピーしてすぐ使えるように
                // DateTime_TitleID/user/TitleID/Account/～で保存しておく
                entryPath = destPath + GetFileEntryName( srcPath + directoryEntryArea[ i ].name );
            }
            entryArray.offset = fileSystemInfo.fileOffset;
            entryArray.size = directoryEntryArea[ i ].fileSize;
            std::strncpy( entryArray.name, entryPath.c_str(), nn::fssystem::detail::PartitionFileSystemFormat::EntryNameLengthMax );
            fileSystemInfo.fileOffset += directoryEntryArea[ i ].fileSize; // ファイルサイズ分、オフセットをずらす
            internalEntryArray.push_back( entryArray );
        }
        delete[] directoryEntryArea;
        directoryEntryArea = nullptr;

        // 子エントリ情報の取得 (ディレクトリのみ)
        NN_RESULT_DO( GetFsEntries( &directoryHandle, &directoryEntryArea, &readEntryCount, srcPath, nn::fs::OpenDirectoryMode_Directory ) );
        fileSystemInfo.entriesCount += readEntryCount;

        // srcPath 直下のディレクトリを調べて、GetAllFsEntriesを再帰的に呼びだす
        for ( int i = 0; i < readEntryCount; i++ )
        {
            entryArray.offset = fileSystemInfo.fileOffset;
            entryArray.size = 0; // ディレクトリはサイズ0にして、データとしては何も書き込まない想定
            const std::string workSrcPath = srcPath + directoryEntryArea[ i ].name +'/';
            // パス名には、SDカード用のパスをいれる、上記参照
            const std::string entryPath = destPath + GetFileEntryName( workSrcPath );
            std::strncpy( entryArray.name, entryPath.c_str(), nn::fssystem::detail::PartitionFileSystemFormat::EntryNameLengthMax );
            internalEntryArray.push_back( entryArray );
            NN_RESULT_DO( GetAllFsEntries( workSrcPath, destPath, fileSystemInfo, internalEntryArray ) );
        }
        delete[] directoryEntryArea;
        directoryEntryArea = nullptr;
        NN_RESULT_SUCCESS;
    }


    /**
     * @brief path で指定したディレクトリ以下のディレクトリとファイルを削除します。
     */
    const nn::Result DeleteAllSaveData( const std::string& destPath, bool isJournalingEnabled ) NN_NOEXCEPT
    {
        DEVMENU_LOG( "delete %s\n", destPath.c_str() );

        nn::fs::DirectoryHandle directoryHandle;
        nn::fs::DirectoryEntry* directoryEntryArea = nullptr;
        int64_t readEntryCount = int64_t();

        // 子エントリ情報の取得 (ファイルのみ)
        NN_RESULT_DO( GetFsEntries( &directoryHandle, &directoryEntryArea, &readEntryCount, destPath, nn::fs::OpenDirectoryMode_File ) );

        // destPath 直下のファイル削除
        for ( auto i = 0; i < readEntryCount; ++i )
        {
            const std::string workDestFile = destPath + '/' + directoryEntryArea[ i ].name;
            NN_RESULT_DO( nn::fs::DeleteFile( workDestFile.c_str() ) );
        }
        delete[] directoryEntryArea;
        directoryEntryArea = nullptr;

        // 子エントリ情報の取得 (ディレクトリのみ)
        NN_RESULT_DO( GetFsEntries( &directoryHandle, &directoryEntryArea, &readEntryCount, destPath, nn::fs::OpenDirectoryMode_Directory ) );

        // srcPath 直下のディレクトリコピー -> CopySaveDataDirectory()
        for ( auto i = 0; i < readEntryCount; ++i )
        {
            const std::string workDestPath = destPath + '/' + directoryEntryArea[ i ].name;
            NN_RESULT_DO( nn::fs::DeleteDirectoryRecursively( workDestPath.c_str() ) );
        }

        if ( isJournalingEnabled )
        {
            NN_RESULT_DO( nn::fs::CommitSaveData( GetStorageName( destPath ).c_str() ) );
        }
        delete[] directoryEntryArea;
        directoryEntryArea = nullptr;
        NN_RESULT_SUCCESS;
    }

    // 末尾が'/'の場合には、ディレクトリとして解釈する
    bool IsDirectoryFileEntry( const std::string& fileName ) NN_NOEXCEPT
    {
        return fileName.back() == '/';
    }

    // SDカード上のパスからセーブデータのRootからの相対パスだけを取得する
    const std::string GetRelativeFilePath( const std::string& path, int searchPos ) NN_NOEXCEPT
    {
        size_t pos = 0;
        for ( auto i = 0; i < searchPos; ++i )
        {
            pos = path.find( "/", pos) + 1;
            if ( std::string::npos == pos )
            {
                return "";
            }
        }
        return path.substr( pos, path.size() - pos ); // ":/"以降をFileEntryとして取得する
    }

    // 全てのエントリーからアーカイブ領域に書き込む
    const nn::Result WriteAllFsEntries( void* pOutBuffer, size_t bufferSize, const std::string& srcPath, const std::vector< nn::fssystem::PartitionFileSystemMeta::FileEntryForConstruct >& internalEntryArray,
        char* pBackupXmlBuffer, int backupXmlBufferSize, char* pConfigXmlBuffer, int configXmlBufferSize) NN_NOEXCEPT
    {
        int64_t lastOffset = 0;
        // 全てのエントリーのファイルを読み込む
        // エントリーの最後２つはバッファ上の__BackupSaveDataInfo.xmlとconfig.xml
        for ( auto i = 0; i < internalEntryArray.size() - DataMoveInfo::XmlFileCount; ++i )
        {
            nn::fs::FileHandle readHandle;

            if ( IsDirectoryFileEntry( internalEntryArray[ i ].name ) )
            {   // ディクレトリは何もしない
                continue;
            }

            // パス名にはDateTime_TitleID/user/TitleID/Account/～になっているので、～部分だけを取り出す
            const std::string readPath = srcPath + GetRelativeFilePath( internalEntryArray[ i ].name, DataMoveInfo::DirectorySearchPosition );
            NN_RESULT_DO( nn::fs::OpenFile( &readHandle, readPath.c_str(), nn::fs::OpenMode_Read ) );
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile( readHandle );
            };

            void* writeBuffer = static_cast< char* >( pOutBuffer ) + internalEntryArray[ i ].offset;
            lastOffset = internalEntryArray[ i ].offset + internalEntryArray[ i ].size;
            NN_RESULT_DO( nn::fs::ReadFile( readHandle, 0, writeBuffer, static_cast< int >( internalEntryArray[ i ].size ) ) );
        }
        // __BackupSaveDataInfo.xmlを書き込む
        std::memcpy( static_cast< char* >( pOutBuffer ) + lastOffset, pBackupXmlBuffer, backupXmlBufferSize );
        // config.xmlを書き込む
        std::memcpy( static_cast< char* >( pOutBuffer ) + lastOffset + backupXmlBufferSize, pConfigXmlBuffer, configXmlBufferSize );
        NN_RESULT_SUCCESS;
    }

} // end of anonymous namespace

// DevMenu では使用しなくなったが CustomerSupportTool では使用する
const nn::Result GetFsEntries( nn::fs::DirectoryHandle* pOutDirectoryHandle, nn::fs::DirectoryEntry** ppDirectoryEntryArea, int64_t* pOutReadCount,
                            const std::string& directoryPath, nn::fs::OpenDirectoryMode mode ) NN_NOEXCEPT
{
    NN_RESULT_DO( nn::fs::OpenDirectory( pOutDirectoryHandle, directoryPath.c_str(), mode ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory( *pOutDirectoryHandle );
    };

    int64_t directoryEntryCount = int64_t();
    int64_t readEntryCount = int64_t();

    NN_RESULT_DO( nn::fs::GetDirectoryEntryCount( &directoryEntryCount, *pOutDirectoryHandle ) );

    // ToDo: メモリ不足エラーへの対処 ( DirectoryEntry 1 つあたり 800 byte あるので Entry 数がかなり多いとメモリを確保できない)
    *ppDirectoryEntryArea = reinterpret_cast< nn::fs::DirectoryEntry* >( new char[ sizeof( nn::fs::DirectoryEntry ) * ( directoryEntryCount + 1 ) ] );

    NN_RESULT_TRY( nn::fs::ReadDirectory( &readEntryCount, *ppDirectoryEntryArea, *pOutDirectoryHandle, directoryEntryCount + 1 ) )
    NN_RESULT_CATCH_ALL
    {
        if ( nullptr != ppDirectoryEntryArea )
        {
            delete[] *ppDirectoryEntryArea;
        }
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY;

    *pOutReadCount = readEntryCount;

    NN_RESULT_SUCCESS;
}

const nn::Result CopySaveDataDirectoryForNsave( const std::string& destPath, const std::string& srcPath,
    bool isJournalingEnabled, int64_t journalSize ) NN_NOEXCEPT
{
    nn::fs::FileHandle readHandle;
    int64_t fileSize;

    NN_RESULT_DO(nn::fs::OpenFile( &readHandle, srcPath.c_str(), nn::fs::OpenMode_Read ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile( readHandle );
    };

    NN_RESULT_DO(nn::fs::GetFileSize( &fileSize, readHandle ) );
    // nsaveファイルサイズがcmacのサイズより小さいことはないのでエラーとする
    if ( fileSize <= nn::spl::AesBlockSize )
    {
        return ResultApplicationError();
    }
    // nsaveファイルサイズ(cmac除く)が最大を超えていないか確認する
    else if ( fileSize - nn::spl::AesBlockSize > DataMoveInfo::NsaveFileMaxSize )
    {
        return ResultSizeOverError();
    }
    std::unique_ptr< char[] > buffer( new char[ fileSize ] );
    NN_RESULT_DO( nn::fs::ReadFile( readHandle, 0, buffer.get(), fileSize ) );

    // nsaveは終端にcmacが付与されているのでcmac部分を除いて検証する
    int64_t checkMacPoint = fileSize - nn::spl::AesBlockSize;
    nn::Bit8 cmacData[ nn::spl::AesBlockSize ] = {};
    ComputeCmacFromSha256Hash( cmacData, buffer.get(), fileSize - nn::spl::AesBlockSize );
    if ( !nn::crypto::IsSameBytes( cmacData, buffer.get() + checkMacPoint, nn::spl::AesBlockSize ) )
    {
        return ResultCmacError();
    }

    NN_RESULT_DO( DeleteAllSaveData( "save:/", true ) );
    std::unique_ptr<nn::fssystem::PartitionFileSystem> fs( new nn::fssystem::PartitionFileSystem() );
    nn::fs::MemoryStorage fsStorage( buffer.get(), fileSize );

    NN_RESULT_DO( fs->Initialize( &fsStorage ) );
    NN_RESULT_DO( nn::fs::fsa::Register( "nsave", std::move( fs ) ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::fsa::Unregister( "nsave" );
    };

    NN_RESULT_DO(CopySaveDataDirectoryForNsaveImpl( "save:/", "nsave:/", true, journalSize ) );
    NN_RESULT_SUCCESS;
}

const nn::Result CreatePartitionFsArchive( const std::string& destFile, const std::string& destPath, const std::string& srcPath,
    char* pBackupXmlBuffer, int backupXmlBufferSize, char* pConfigXmlBuffer, int configXmlBufferSize ) NN_NOEXCEPT
{
    DEVMENU_LOG( "create archive %s to %s\n", srcPath.c_str(), destFile.c_str() );

    nn::fssystem::PartitionFileSystemMeta metaEditor;

    // パス名はSDカードの絶対パスから相対パスに変換する
    const std::string entryPath = GetRelativeFilePath( destPath, DataMoveInfo::RelativePathPosition ) + '/';

    PartitionFileSystemInfo fileSystemInfo;
    fileSystemInfo.fileOffset = 0;
    fileSystemInfo.entriesCount = 0;
    std::vector<nn::fssystem::PartitionFileSystemMeta::FileEntryForConstruct> internalEntryArray;
    NN_RESULT_DO( GetAllFsEntries( srcPath, entryPath, fileSystemInfo, internalEntryArray ) );

    // __BackupSaveDataInfo.xmlをnsaveに挿入する
    nn::fssystem::PartitionFileSystemMeta::FileEntryForConstruct backupSaveDataInfoEntryArray;
    if ( internalEntryArray.size() == 0 )
    {
        backupSaveDataInfoEntryArray.offset = 0;
    }
    else
    {
        backupSaveDataInfoEntryArray.offset = internalEntryArray[ internalEntryArray.size() - 1 ].offset + internalEntryArray[ internalEntryArray.size() - 1 ].size;
    }
    backupSaveDataInfoEntryArray.size = backupXmlBufferSize;
    const std::string backupXmlEntryPath = entryPath + DataMoveInfo::ExportedSaveDataInfoFileName;
    std::strncpy( backupSaveDataInfoEntryArray.name, backupXmlEntryPath.c_str(), nn::fssystem::detail::PartitionFileSystemFormat::EntryNameLengthMax );
    internalEntryArray.push_back( backupSaveDataInfoEntryArray );
    fileSystemInfo.fileOffset = fileSystemInfo.fileOffset + backupXmlBufferSize;
    fileSystemInfo.entriesCount++;

    // config.xmlをnsaveに挿入する
    nn::fssystem::PartitionFileSystemMeta::FileEntryForConstruct configEntryArray;
    // __BackupSaveDataInfo.xmlが必ずinternalEntryArrayに入るのでinternalEntryArrayが空の場合を考慮しない
    configEntryArray.offset = internalEntryArray[ internalEntryArray.size() - 1 ].offset + backupXmlBufferSize;
    configEntryArray.size = configXmlBufferSize;
    const std::string configXmlEntryPath = entryPath.substr( 0 , DataMoveInfo::ConfigXmlRootPathStringSize ) + DataMoveInfo::ConfigFileName;
    std::strncpy( configEntryArray.name, configXmlEntryPath.c_str(), nn::fssystem::detail::PartitionFileSystemFormat::EntryNameLengthMax );
    internalEntryArray.push_back( configEntryArray );
    fileSystemInfo.fileOffset = fileSystemInfo.fileOffset + configXmlBufferSize;
    fileSystemInfo.entriesCount++;

    // メタデータの作成
    size_t bufferSize = 0;
    size_t fileDataSize = 0;
    size_t metaDataSize = 0;
    metaEditor.QueryMetaDataSize( &metaDataSize, internalEntryArray.data(), internalEntryArray.size() );
    // 読み込むファイルサイズ
    fileDataSize = metaDataSize + fileSystemInfo.fileOffset;
    // 作成するファイルサイズ(cmac除く)が最大を超えた場合は処理を失敗させる
    if ( fileDataSize > DataMoveInfo::NsaveFileMaxSize )
    {
        return ResultSizeOverError();
    }
    bufferSize = fileDataSize + nn::spl::AesBlockSize;

    std::unique_ptr< char[] > buffer( new char[ bufferSize ] );
    // メモリを初期化しておく
    std::memset( buffer.get(), 0x0, bufferSize );
    metaEditor.ConstructMetaData( buffer.get(), fileDataSize, internalEntryArray.data(), internalEntryArray.size() );

    NN_RESULT_DO( WriteAllFsEntries( buffer.get() + metaDataSize, fileSystemInfo.fileOffset, srcPath, internalEntryArray, pBackupXmlBuffer, backupXmlBufferSize, pConfigXmlBuffer, configXmlBufferSize ) );

    // CMACの埋め込み
    NN_RESULT_DO( ComputeCmacFromSha256Hash( reinterpret_cast< nn::Bit8* >( buffer.get() + fileDataSize ), buffer.get(), fileDataSize ) );

    // bufferをdestFileに書き込む
    {
        nn::fs::FileHandle writeHandle;
        NN_RESULT_DO( nn::fs::CreateFile( destFile.c_str(), static_cast< int >( bufferSize ) ) );
        NN_RESULT_DO( nn::fs::OpenFile( &writeHandle, destFile.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile( writeHandle );
        };
        NN_RESULT_DO( nn::fs::WriteFile( writeHandle, 0, buffer.get(), static_cast< int >( bufferSize ), nn::fs::WriteOption() ) );

        NN_RESULT_DO( nn::fs::FlushFile( writeHandle ) );
    }
    NN_RESULT_SUCCESS;
}

}} // ~namespace devmenu::savedata, ~namespace devmenu

