﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <cstring>

#include "qre.h"

#include "QREL.h"
#include "QREncode.h"
#include "../DevMenu_Config.h"


using namespace mw::qre;

#if 0
static CQREncode *g_QREncode;
#else
static CQREncode g_QREncode;
#endif

QREncoder::QREncoder()
{
#if 0
    if (!g_QREncode)
        g_QREncode = new CQREncode;
#endif
}

QREncoder::~QREncoder()
{
#if 0
    if (g_QREncode) {
        delete g_QREncode;
        g_QREncode = NULL;
    }
#else
    g_QREncode.ReleaseMemory();
#endif
}

/**
 * QRコード生成
 *
 * @param   qrInfo(in)    QRコード情報構造体
 * @return  生成結果（trueは正常：falseは失敗）
 */
bool QREncoder::Encode(EncodeData *qrInfo)
{
    return g_QREncode.QRE_EncodeData(qrInfo);
}

bool QREncoder::EncodeWithImage(EncodeData *qrInfo, const ImageInfo *imageInfo)
{
    return g_QREncode.QRE_EncodeWithImg(qrInfo, imageInfo);
}

/**
 * エラー情報取得
 *
 * @param   pMess(out)  エラー情報文字列格納用
 * @return  取得結果（trueは正常：falseは失敗）
 */
bool QREncoder::GetErrorMessage(char *pMess, int32_t size)
{
    return g_QREncode.QRE_GetErrorMessage(pMess, size);
}

/**
 * 分割数取得
 *
 * @return  分割数
 */
uint32_t QREncoder::GetDivCount()
{
    return g_QREncode.QRE_GetDivCount();
}

/**
 * フォーマットバージョン取得
 *
 * @return  フォーマットバージョン
 */
uint32_t QREncoder::GetVersion()
{
    return g_QREncode.QRE_GetVersion();
}

/**
 * BMPファイルサイズ取得
 *
 * @param   index(in)   分割番号 (0～15)
 * @return  BMPファイルサイズ
 */
uint32_t QREncoder::GetQRSize(uint32_t index, bool is_bmp)
{
    return g_QREncode.QRE_GetBMPSize(index, is_bmp);
}

/**
 * BMPデータ取得
 *
 * @param   pData(out) 画像データ
 * @param   size(in)   画像サイズ
 * @param   index(in)  分割番号(0～15)
 * @return  取得結果（trueは正常：falseは失敗）
 */
bool QREncoder::GetQRBMPData(uint8_t *buff, uint32_t size, uint32_t index, Rect *pos)
{
    return g_QREncode.QRE_GetBMPData(buff, size, index, pos);
}

bool QREncoder::GetQRData(ImageInfo *info, uint32_t index)
{
    return g_QREncode.QRE_GetBMPData(info, index);
}

uint32_t QREncoder::GetDesignQRImageSize(uint32_t version, uint32_t cellSize)
{
        return (version * 4 + 25) * cellSize / 3;
}

inline const int8_t *R(const int8_t *const ptr)
{
    const int8_t *pos = ptr, *ret = NULL;

    while (*pos) {
        if (*pos++ == '\\')
            ret = pos;
    }
    return ret ? ret : ptr;
}

enum ConstantValue_QREncodeLib
{
    HeapTableMax = 1024,
};

typedef struct {
    uint64_t size;
    uint8_t *ptr;
    bool use;
} HeapTable;

static uint8_t *heap = NULL;
static uint64_t heap_max = 0;
static uint64_t heap_use = 0;
static uint64_t heap_table_use = 0;
static HeapTable heap_table[HeapTableMax];

inline int32_t round32(int32_t i)
{
    return ((i + ((int32_t)31)) & ~((int32_t)31));
}

void InitHeap()
{
    heap_use = 0;
    heap_table_use = 0;
    memset(heap_table, 0, sizeof(HeapTable) * HeapTableMax);
}

#ifdef ISP_MEMORY_DEBUG
void *ISPMalloc(size_t size, const int8_t *file, int32_t line, const int8_t *func)
#else
void *QRE_Malloc(size_t size)
#endif
{
#ifdef ISP_MEMORY_DEBUG
NN_LOG("%s from %s[%d] %s: size %d", __func__, R(file), line, func, size);
#endif
    if (!heap) {
#ifdef ISP_MEMORY_DEBUG
NN_LOG(" ptr 0x%08X\n", NULL, __LINE__);
#endif
        return NULL;
    }
    size = round32((int32_t)size);
    uint64_t i;
    for (i = 1; i < heap_table_use; i++) {
        if (!heap_table[i].use && size <= heap_table[i].size) {
            heap_table[i].use = true;
#ifdef ISP_MEMORY_DEBUG
NN_LOG(" ptr 0x%08X from %d\n", heap_table[i].ptr, __LINE__);
#endif
            return heap_table[i].ptr;
        }
    }

    if (heap_max - heap_use < size) {
#ifdef ISP_MEMORY_DEBUG
NN_LOG(" ptr 0x%08X from %d\n", NULL, __LINE__);
#endif
        return NULL;
    }

    if (heap_table_use >= HeapTableMax) {
#ifdef ISP_MEMORY_DEBUG
NN_LOG(" ptr 0x%08X from %d\n", NULL, __LINE__);
#endif
        return NULL;
    }

    i = heap_table_use;
    heap_table[i].use = true;
    heap_table[i].size = size;
    heap_table[i].ptr = heap + heap_use;
    heap_use += size;
    heap_table_use++;
#ifdef ISP_MEMORY_DEBUG
NN_LOG(" ptr 0x%08X from %d\n", heap_table[i].ptr, __LINE__);
#endif
    return (void *)heap_table[i].ptr;
}

#ifdef ISP_MEMORY_DEBUG
void *ISPRealloc(void *ptr, size_t size,
    const int8_t *file, int32_t line, const int8_t *func)
#else
void *QRE_Realloc(void *ptr, size_t size)
#endif
{
#ifdef ISP_MEMORY_DEBUG
    NN_LOG("%s: from %s[%d] %s size = %d\n", __func__, R(file), line, func, size);
#endif
    if (!ptr)
        return QRE_Malloc(size);
#if 0
    int32_t *org_size = reinterpret_cast<int32_t *>(ptr);
    org_size -= 1;
    if (size < org_size[0]) {
        // org_size[0] = size;
        return ptr;
    }
    void *ret = QRE_Malloc(size);
    memcpy(ret, ptr, org_size[0]);
    QRE_Free(ptr);
#else
    void *ret = QRE_Malloc(size);
    memcpy(ret, ptr, size);
    QRE_Free(ptr);
#endif
    return ret;
}

#ifdef ISP_MEMORY_DEBUG
void ISPFree(void *ptr, const int8_t *file, int32_t line, const int8_t *func)
#else
void QRE_Free(void *ptr)
#endif
{
    uint64_t i = 0;
#ifdef ISP_MEMORY_DEBUG
NN_LOG("%s from %s[%d] %s: ptr 0x%08X\n", __func__, R(file), line, func, ptr);
#endif
    for (i = 1; i < heap_table_use; i++)
        if (heap_table[i].ptr == ptr) {
            heap_table[i].use = false;
            return;
        }
}

bool QREncoder::InitMemory(void *ptr, uint32_t size)
{
    if (!ptr || !size)
        return false;

    heap = reinterpret_cast<uint8_t *>(ptr);
    heap_max = size;
    InitHeap();

    return true;
}

uint32_t QREncoder::GetEncodeBufferSize(uint32_t data_size, uint32_t cell_size, uint32_t count)
{
    if (count) {
        return 1024 * 512 * 5 + 177 * 177 * 3 * cell_size * cell_size * count;
    }
    else
    {
        return 1024 * 512 * 5 + 177 * 177 * 3 * cell_size * cell_size * (data_size / 1200 + 1);
    }
}
